;**************************************************************************
; FILENAME: c2xxprog.asm - Generic name
;           c2xx_ver.asm  
;
; DESCRIPTION:
;   Flash Programmer control module with flash algorithms -P
;   called by the JTAG loader, PRG2xxw.exe
;   Uses the on-chip B0/1 RAM for algorithms and flash data buffer
;
; GLOBALS:
;
;   unsigned * PRG_bufaddr      Address of buffer for flash/program data
;   unsigned   PRG_bufsize      Size of program data buffer
;   unsigned   PRG_devsize      Size of programmable device
;   unsigned * PRG_paddr        First programming address
;   unsigned   PRG_page         Programming page
;   unsigned   PRG_length       Length of programming data
;   unsigned   PRG_status       Status of programming functions
;
; PUBLIC FUNCTIONS:
;   PRG_init                    Initialize system for programming
;   PRG_program                 Program a block
;   PRG_erase                   Erase a programmable device
;   PRG_verify                  Verify a block
;
; PRIVIATE FUNCTIONS:
;   None
;
; USAGE/LIMITATIONS
;   The global and public function symbol values must be available in the
;   COFF file for the loader to work properly. The functions are not
;   intended to be called by another program.
;
; NOTES:
;   The program needs three memory sections to operate:
;   Actual location of these sections in the memory is defined in the
;   linker command file - *.cmd
;
;       PRG_text    Executable section for this program
;       PRG_parms   Variable section for this program
;       PRG_data    Data buffer used to hold programming data
;
;   The size and location of each section  is user defined.  The
;   loader program will read the value of PRG_bufaddr and PRG_bufsize to
;   determine the location and size of the PRG_data section.
;
; JTAG loader command file format
;   PRG2xxw  -[options] c2xxprog.out       Flashcode.out
;  <JTAG loader)      <Flash algorithm>  <COFF file to be flashed>
;
;   Source : TI Tools group, Houston
;   Version : 2.0
;
;
; ***************************************************************************
; INCLUDE FILES
; ***************************************************************************
    .mmregs
    .include  "..\include\Svar20.h" ; Variable declarations (DAF-19MAR)

; ***************************************************************************
; PUBLIC DECLARATIONS
; ***************************************************************************
        .global PRG_init, PRG_program, PRG_erase, PRG_verify, PRG_stop
        .global PRG_bufaddr, PRG_bufsize, PRG_devsize, PRG_paddr, PRG_page
        .global PRG_length, PRG_status,PARMS
        .global PROTECT,SEG_ST,SEG_END
; ***************************************************************************
; PRIVATE DECLARATIONS
; ***************************************************************************
                       
BUFFER_SIZE .set    200         ;Size of program buffer size
                                ;Can be increased based available
                                ;memory

DEVICE_SIZE .set    0ffffh      ;Size of device to be programmed.
                                ;Default is maximum address range for
                                ;the F2xx

; ***************************************************************************
; Define the PRG_parm section
; ***************************************************************************
    .sect   "PRG_parm"          ;Actual location in memory defined is
PARMS:                          ;linker command file
PRG_bufaddr .word   PrgBuffer   ;Address of buffer for program
PRG_bufsize .word   BUFFER_SIZE ;Tells host of buffer size
PRG_devsize .word   DEVICE_SIZE ;Tells host of device size

*                               ;The following parameters will be redefined by
*                               ;PRG2xxw based on the flashcode.out
                                ;Algorithm Array variables
PRG_paddr   .word   0           ;First address to program
PRG_page    .word   0           ;Page to program
PRG_length  .word   0           ;Length of block to program
PRG_status  .word   1           ;Status of programming functions
Temp        .word   0           ;Temp location for verify
PROTECT     .word   0FF00h      ;Enable all 8 segments

; ***************************************************************************
; Define the PRG_data section
; ***************************************************************************
    .sect   "PRG_data"          ;Flash code data buffer 
PrgBuffer   .space  BUFFER_SIZE*16  ;Initializes buffer for program data
    .sect   "ary_var"           ; Initialize buffer to 0x0000
            .space  16*16
;
; ***************************************************************************
; Define the PRG_text section
; ***************************************************************************
    .sect   "PRG_text"          ; Control and algorithm module

; F**************************************************************************
; Name: PRG_init
; 
; Description
;   Initialize the F2xx device for programming
; 
; Inputs
;   None
; Outputs
;   int PRG_status              Pass =0; Fail = 1;
;
; Notes: Can be used to include device specific initialization before executing
;        Flash algorithms in PRG_erase,PRG_program, PRG_verify and PRG_stop
;
; F**************************************************************************
 
PRG_init:
		LDP	#0h			;(DAF-19MAR)
		SPLK	#0h, GPR0
;		OUT	GPR0, WSGR
		LDP	#00E0h			;(DAF-19MAR)
		SPLK	#006Fh, WD_CNTL
		SPLK	#05555h, WD_KEY
		SPLK	#0AAAAh, WD_KEY

	      ldp #PARMS		; Defines data page
	      splk    #0,PRG_status
            b       PRG_stop
;
; F**************************************************************************
; Name: PRG_program
; 
; Description
;   Transfers a block of data from RAM to Flash/programmable memory
;   PRG2xxw will execute this after PRG_init by default. If the command
;   line does not have any options -e or -v.
;   If the option specified is -e  then the execution sequence will be:
;   PRG_init > PRG_erase > PRG_program and then PRG_stop
;   If the option specified is -v, then the execution sequence will be:
;   PRG_init > PRG_program, PRG_verify and then PRG_stop
; Inputs
;   unsigned *PRG_paddr         Pointer to first address to program
;   unsigned  PRG_page          Page to program, Program == 0, Data == 1
;   unsigned  PRG_length        Number of words to program
;   unsigned *PRG_bufaddr       Pointer to program data buffer
;
; Outputs
;   int PRG_status              Pass =0; Fail = 1;
;
; Notes:
;   The PRG_page value is setup by the host but not used in this case.
;   The assumption is that the programmable device is in program space.
; F**************************************************************************
 
PRG_program:    b    Fail

;
; F**************************************************************************
; Name: PRG_erase
; 
; Description
;   Erase a programmed device. This module will be executed if the PRG2xxw
;   command line option specified is -e.
; Inputs
;   None:
;
; Outputs
;   int PRG_status      Pass =0; Fail = 1;
;
; Notes:
;   The erase is device specific. 
; F**************************************************************************
 
PRG_erase:  b    Fail
;
; F**************************************************************************
; Name: PRG_verify
; 
; Description
;   Verify a block of programmed data. This module will be excuted if the
;   command line option in PRG2xxw is -v.
; 
; Inputs                        Defined by host
;   unsigned *PRG_paddr         Pointer to first address to verify
;   unsigned  PRG_page          Page to verify, Program == 0, Data == 1
;   unsigned  PRG_length        Number of words to verify
;   unsigned *PRG_bufaddr       Pointer to verify data buffer
;
; Outputs
;   int PRG_status      Pass =0; Fail = 1;
;
; Notes:
;   The PRG_page value is setup by the host but not used in this case.
;   The assumption is that the programmable device is in program space.
; F**************************************************************************

PRG_verify:     
            ; Setup parameters
            ; ar7 points to the source data buffer in ram
            ; ar5 points to flash command addresses
            ldp     #PARMS
	      lar     ar7,PRG_bufaddr     ;Load source address
            mar     *,ar7               ;Arp = 7 
           
Verify16:            
            lacc    PRG_paddr
            tblr    Temp  
            add     #1            
            sacl    PRG_paddr   
                                    
            lacc    *+                  ;load flash data
            sub     Temp                ;Subtract to find difference
            bcnd    Fail,neq

            lacc    PRG_length          ;Load verify count
            sub     #1                  ;Decrement the count
            sacl    PRG_length          ;Save the count
            bcnd    Verify16,neq        ;Loop if count not == 0

            splk    #0,PRG_status       ;update algorithm status
            b       PRG_stop            ;Branch to stop

Fail:       ldp     #PARMS
	      splk    #1,PRG_status 
            b       PRG_stop


;
; F**************************************************************************
; Name: PRG_stop
; 
; Description
;       Exit routine for all programming functions.
; 
; Inputs
;       None
;
;
; Outputs
;       None
;
; Notes:
;       Exit point for all programming functions
;       The ESTOP opcode gets executed as a NOP when not in emulation
;       mode.  The "b $" will keep the program from running off if
;       not in emulation mode.
;
; F**************************************************************************
PRG_stop:

        nop
        .word   0BE90h                  ;SWI instruction
        nop
        b $
                .end



