/*
 * High-temperature wrapper for the F2806x CAN boot loader.
 *
 * These functions re-trim the oscillators for the current operating
 * temperature, then invoke the ROM boot loader.
 */

#include "DSP28x_Project.h"
#include <stdint.h>

void InitAdc(void);
void Osc1Comp (int16 sensorSample);
void Osc1Comp (int16 sensorSample);



//Read the temperature sensor value, then pass it to the oscillator compensation
//functions. Once that's done, call the CAN boot loader in ROM. The ADC and
//compensation code is copied from Example_2806xOscComp.
#pragma CODE_SECTION(CompensatedCANBootLoader, "LoaderWrapper")
void CompensatedCANBootLoader(void)
{
	Uint32 (*CAN_Boot_ptr)(void) = (Uint32 (*)(void))0x3ff4fc;
	void (*EntryFunc_ptr)(void);
	volatile Uint32 entryAddr;
	int16 sample;

	// Initialize the ADC and connect it to the temperature sensor
	InitAdc();
	EALLOW;
	AdcRegs.ADCCTL2.bit.ADCNONOVERLAP = 1;	//Enable non-overlap mode
	AdcRegs.ADCCTL1.bit.TEMPCONV  = 1; 		//Connect channel A5 internally to the temperature sensor
	AdcRegs.ADCSOC0CTL.bit.CHSEL  = 5; 		//Set SOC0 channel select to ADCINA5
	AdcRegs.ADCSOC0CTL.bit.ACQPS  = 25; 	//Set SOC0 acquisition period to 26 ADCCLK
	AdcRegs.INTSEL1N2.bit.INT1SEL = 0; 		//Connect ADCINT1 to EOC0
	AdcRegs.INTSEL1N2.bit.INT1E  =  1; 		//Enable ADCINT1


	//Force start of conversion on SOC0
	AdcRegs.ADCSOCFRC1.all = 0x01;

	//Wait for end of conversion.
	while(AdcRegs.ADCINTFLG.bit.ADCINT1 == 0){}  //Wait for ADCINT1
	AdcRegs.ADCINTFLGCLR.bit.ADCINT1 = 1;        //Clear ADCINT1

	asm(" RPT #255 || NOP");
	//Get temp sensor sample result from SOC1
	sample = AdcResult.ADCRESULT0;

	//Use temp sensor measurement to perform oscillator compensation even as temperature changes.
	Osc1Comp(sample);
	Osc2Comp(sample);

	//Later
	entryAddr = (*CAN_Boot_ptr)();
	EntryFunc_ptr = (void (*)(void))entryAddr;
	(*EntryFunc_ptr)();
}


//Microsecond delay function for when CPUCLK = INTOSC2. The frequency
//is assumed to be 11 MHz to provide extra margin at high-temp. The
//1000us ADC delay is taken from F2806x_Adc.c.
#define ADC_usDELAY  1000L
#define OSC_FREQ_MHZ 11L
#define DELAY_US_OSC(us) DSP28x_usDelay((OSC_FREQ_MHZ * (us)) / 5)

//This function is copied from F2806x_Adc.c and modified to use the delay function above
#pragma CODE_SECTION(InitAdc, "LoaderWrapperFuncs")
void InitAdc(void)
{
	// *IMPORTANT*
	// The Device_cal function, which copies the ADC calibration values from TI reserved
	// OTP into the ADCREFSEL and ADCOFFTRIM registers, occurs automatically in the
	// Boot ROM. If the boot ROM code is bypassed during the debug process, the
	// following function MUST be called for the ADC to function according
	// to specification. The clocks to the ADC MUST be enabled before calling this
	// function.
	// See the device data manual and/or the ADC Reference
	// Manual for more information.
	EALLOW;
	SysCtrlRegs.PCLKCR0.bit.ADCENCLK = 1;
	(*Device_cal)();
	EDIS;

	// To powerup the ADC the ADCENCLK bit should be set first to enable
	// clocks, followed by powering up the bandgap, reference circuitry, and ADC core.
	// Before the first conversion is performed a 5ms delay must be observed
	// after power up to give all analog circuits time to power up and settle

	// Please note that for the delay function below to operate correctly the
	// CPU_RATE define statement in the F2806x_Examples.h file must
	// contain the correct CPU clock period in nanoseconds.
	EALLOW;
	AdcRegs.ADCCTL1.bit.ADCBGPWD  = 1;      // Power ADC BG
	AdcRegs.ADCCTL1.bit.ADCREFPWD = 1;      // Power reference
	AdcRegs.ADCCTL1.bit.ADCPWDN   = 1;      // Power ADC
	AdcRegs.ADCCTL1.bit.ADCENABLE = 1;      // Enable ADC
	AdcRegs.ADCCTL1.bit.ADCREFSEL = 0;      // Select interal BG
	EDIS;

	DELAY_US_OSC(ADC_usDELAY);         // Delay before converting ADC channels

	EALLOW;
	AdcRegs.ADCCTL2.bit.CLKDIV2EN = 1;
	EDIS;

	DELAY_US_OSC(ADC_usDELAY);         // Delay before converting ADC channels
}



//Oscillator compensation routines copied from F2806x_OscComp.c

// Useful definitions
  #define FP_SCALE 32768       // Scale factor for Q15 fixed point numbers (2^15)
  #define FP_ROUND FP_SCALE/2  // Quantity added to Q15 numbers before converting
                               // to integer to round the number

//Amount to add to Q16.15 fixed point number to shift from a fine trim range of
//(-31 to 31) to (1 to 63).  This guarantees that the trim is positive and can
//therefore be efficiently rounded
  #define OSC_POSTRIM 32
  #define OSC_POSTRIM_OFF FP_SCALE*OSC_POSTRIM

//The following functions return reference values stored in OTP.

//Slope used to compensate oscillator 1 (fine trim steps / ADC code). Stored
//in fixed point Q15 format.
  #define getOsc1FineTrimSlope() (*(int16 (*)(void))0x3D7E90)()
//Oscillator 1 fine trim at high temp
  #define getOsc1FineTrimOffset() (*(int16 (*)(void))0x3D7E93)()
//Oscillator 1 coarse trim
  #define getOsc1CoarseTrim() (*(int16 (*)(void))0x3D7E96)()

//Slope used to compensate oscillator 2 (fine trim steps / ADC code). Stored
//in fixed point Q15 format.
  #define getOsc2FineTrimSlope() (*(int16 (*)(void))0x3D7E99)()
//Oscillator 2 fine trim at high temp
  #define getOsc2FineTrimOffset() (*(int16 (*)(void))0x3D7E9C)()
//Oscillator 2 coarse trim
  #define getOsc2CoarseTrim() (*(int16 (*)(void))0x3D7E9F)()

//ADC reading of temperature sensor at reference temperature for compensation
  #define getRefTempOffset() (*(int16 (*)(void))0x3D7EA2)()

//Define function for later use
Uint16 GetOscTrimValue(int Coarse, int Fine);


// This function uses the temperature sensor sample reading to perform internal oscillator 1 compensation with
// reference values stored in OTP.
#pragma CODE_SECTION(Osc1Comp, "LoaderWrapperFuncs")
void Osc1Comp (int16 sensorSample)
{
    int16 compOscFineTrim;

    EALLOW;
    compOscFineTrim = ((sensorSample - getRefTempOffset())*(int32)getOsc1FineTrimSlope()
                      + OSC_POSTRIM_OFF + FP_ROUND )/FP_SCALE + getOsc1FineTrimOffset() - OSC_POSTRIM;
    if(compOscFineTrim > 31){
		compOscFineTrim = 31;
	}
	else if(compOscFineTrim < -31){
		compOscFineTrim = -31;
	}
    SysCtrlRegs.INTOSC1TRIM.all = GetOscTrimValue(getOsc1CoarseTrim(), compOscFineTrim);
    EDIS;
}

// This function uses the temperature sensor sample reading to perform internal oscillator 2 compensation with
// reference values stored in OTP.
#pragma CODE_SECTION(Osc2Comp, "LoaderWrapperFuncs")
void Osc2Comp (int16 sensorSample)
{
    int16 compOscFineTrim;

    EALLOW;
    compOscFineTrim = ((sensorSample - getRefTempOffset())*(int32)getOsc2FineTrimSlope()
                      + OSC_POSTRIM_OFF + FP_ROUND )/FP_SCALE + getOsc2FineTrimOffset() - OSC_POSTRIM;

    if(compOscFineTrim > 31){
		compOscFineTrim = 31;
	}
	else if(compOscFineTrim < -31){
		compOscFineTrim = -31;
	}

    SysCtrlRegs.INTOSC2TRIM.all = GetOscTrimValue(getOsc2CoarseTrim(), compOscFineTrim);
    EDIS;
}

//This function packs the coarse and fine trim into
//the format of the oscillator trim register
#pragma CODE_SECTION(GetOscTrimValue, "LoaderWrapperFuncs")
Uint16 GetOscTrimValue(int Coarse, int Fine)
{
    Uint16 regValue = 0;

    if(Fine < 0)
    {
        regValue = ((-Fine) | 0x20) << 9;
    }
    else
    {
        regValue = Fine << 9;
    }
    if(Coarse < 0)
    {
        regValue |= ((-Coarse) | 0x80);
    }
    else
    {
        regValue |= Coarse;
    }
    return regValue;
}
