;;#############################################################################
;;! \file CRC16.asm
;;!
;;! \brief  16-bit CRC
;;! \author Vishal Coelho
;;!
;;! \date   Sep 04, 2014
;;!
;;
;;  Group:            C2000
;;  Target Family:    C28x
;;
;; (C)Copyright $YEAR$, Texas Instruments, Inc.
;;#############################################################################
;;$TI Release: PACKAGE NAME $
;;$Release Date: PACKAGE RELEASE DATE $
;;#############################################################################
;;
;;*****************************************************************************
;; includes
;;*****************************************************************************
;;
;;*****************************************************************************
;; globals
;;*****************************************************************************
    ; Module definition for external reference
        .def      _CRC16_calc
        .def      _CRC16_calc_alt
;;
;;=============================================================================
;; uint16_t CRC16_calc(uint16_t accumulator, uint16_t nBytes, uint16_t parity,
;;                     uint16_t *tablePtr, uint16_t *msgPtr)
;;
;; \brief 16-bit CRC using a table lookup method
;; Calculate the CRC of a string of bytes, by taking each byte as an index into
;; a CRC lookup table and then XOR'ing the table entry successively into the
;; accumulator.
;;
;; \param accumulator AL     seed value for the accumulator
;; \param nBytes      AH     number of bytes
;; \param parity      SP[-1] 0 start CRC at low byte of first message word
;;                           1 start CRC at high byte of the first message word
;; \param tablePtr    XAR4   pointer to the CRC lookup table
;; \param msgPtr      XAR5   pointer to the message
;; \return            AL     16-bit crc result
;;
;; Stack defines
;;
;;   |_______|
;;   |_______|<- Stack Pointer   (SP) <---SP
;;   |_______|<- STK_PARITY      (SP-1)
;;
;; cycles: odd parity  : 14 + (n-1)*8
;;         even parity : 13 + (n-1)*8
;;=============================================================================
;;
	.asg 	  1, LOCAL_FRAME_SIZE
	.asg	  1, STK_PARITY

_CRC16_calc:
    PUSH      XAR1
    MOVL      XAR1, #0                           ; Make sure AR1H = 0
    SUBB      AH, #1                             ; AH  = nBytes - 1
    MOVZ      AR0, AH                            ; AR0 = nBytes - 1
    TBIT      *-SP[STK_PARITY], #0               ; if(parity = 0) TC = 0 else TC = 1

    SBF       _CRC16_calc_evenparity, NTC        ; if(parity = 0) start from LSB
    ADDB      XAR5, #1                           ;        else    start from MSB (XAR5++)
_CRC16_calc_evenparity:
_CRC16_calc_loop:
	XOR       AL, *XAR5++                        ; accumulator ^= msg[i]
    MOVB      AR1, AL.LSB                        ; AR1 = accumulator(i) ^ msg[i]
    LSR       AL, #8                             ; AL >> 8
    XOR       AL, *+XAR4[AR1]                    ; accumulator ^= tbl[accumulator(i-1) ^ msg[i-1]]
    BANZ      _CRC16_calc_loop, AR0--

    POP       XAR1
    LRETR

	.unasg	  LOCAL_FRAME_SIZE
	.unasg    STK_PARITY
;;
;;=============================================================================
;; uint16_t CRC16_calc_alt(uint16_t accumulator, uint16_t nBytes, uint16_t parity,
;;                     uint16_t *tablePtr, uint16_t *msgPtr)
;;
;; \brief 16-bit CRC using a table lookup method and a repeat block iterator
;; Calculate the CRC of a string of bytes, by taking each byte as an index into
;; a CRC lookup table and then XOR'ing the table entry successively into the
;; accumulator.
;;
;; \param accumulator AL     seed value for the accumulator
;; \param nBytes      AH     number of bytes
;; \param parity      SP[-1] 0 start CRC at low byte of first message word
;;                           1 start CRC at high byte of the first message word
;; \param tablePtr    XAR4   pointer to the CRC lookup table
;; \param msgPtr      XAR5   pointer to the message
;; \return            AL     16-bit crc result
;;
;; Stack defines
;;
;;   |_______|
;;   |_______|<- Stack Pointer   (SP) <---SP
;;   |_______|<- STK_PARITY      (SP-1)
;;   |_______|<- STK_NBYTES      (SP-2)
;;
;; cycles: even parity:
;;    nBytes    cycles
;;      1         33
;;      2   	  42
;;      2x		  22 + (n/2 *  12) + 10
;;      2x+1      22 + (n/2 *  12) + 14
;; odd parity: add 1 to the above counts
;;=============================================================================
;;
	.asg 	  2, LOCAL_FRAME_SIZE
	.asg	  1, STK_PARITY
	.asg      2, STK_NBYTES

_CRC16_calc_alt:
    PUSH      XAR1
    MOVL      XAR1, #0                           ; Make sure AR1H = 0
    TBIT      *-SP[STK_PARITY], #0               ; if(parity = 0) TC = 0 else TC = 1
    SBF       _CRC16_calc_alt_evenparity, NTC    ; if(parity = 0) start from LSB
    ADDB      XAR5, #1                           ;        else    start from MSB (XAR5++)
_CRC16_calc_alt_evenparity:
    MOV       *-SP[STK_NBYTES], AH               ; Save nBytes to local frame
    AND       AH, AH, 0xFFFE                     ; AH  = multiples of 2
    SBF       _CRC16_calc_alt_LTEonebyte, EQ     ; if(AH == 0) goto "<= 1 byte"
                                                 ;        else start block calculation of crcs   
    LSR       AH, #1                             ; AH  = nBytes/2
    SUBB      AH, #1                             ; AH  = nBytes/2 - 1
    MOVZ      AR0, AH                            ; AR0 = nBytes/2 - 1

    .align    2
    NOP                                          ; Odd align the Repeat block
    RPTB      _CRC16_calc_alt_loop, AR0              ; Repeat nBytes times
	XOR       AL, *XAR5++                        ; accumulator ^= msg[i]
    MOVB      AR1, AL.LSB                        ; AR1 = accumulator(i) ^ msg[i]
    LSR       AL, #8                             ; AL >> 8
    XOR       AL, *+XAR4[AR1]                    ; accumulator ^= tbl[accumulator(i-1) ^ msg[i-1]]
    
	XOR       AL, *XAR5++                        ; accumulator ^= msg[i]
    MOVB      AR1, AL.LSB                        ; AR1 = accumulator(i) ^ msg[i]
    LSR       AL, #8                             ; AL >> 8
    XOR       AL, *+XAR4[AR1]                    ; accumulator ^= tbl[accumulator(i-1) ^ msg[i-1]]
_CRC16_calc_alt_loop:
_CRC16_calc_alt_LTEonebyte:
    TBIT      *-SP[STK_NBYTES], #0               ; if(nBytes is odd) TC = 0 else TC = 1
    SBF       _CRC16_calc_alt_end, NTC           ; if(nBytes is even) goto end
                                                 ;              else  do final byte crc
	XOR       AL, *XAR5++                        ; accumulator ^= msg[i]
    MOVB      AR1, AL.LSB                        ; AR1 = accumulator(i) ^ msg[i]
    LSR       AL, #8                             ; AL >> 8
    XOR       AL, *+XAR4[AR1]                    ; accumulator ^= tbl[accumulator(i-1) ^ msg[i-1]]    
_CRC16_calc_alt_end:
    POP       XAR1
    LRETR
	.unasg 	  LOCAL_FRAME_SIZE
	.unasg	  STK_PARITY
	.unasg    STK_NBYTES
;;#############################################################################
;;  End of File
;;#############################################################################
