/*
 * sysclkcfg.h
 */

#ifndef _SYSCLKCFG_H_
#define _SYSCLKCFG_H_


#ifdef __cplusplus
extern "C" {
#endif

#include <stdint.h>

/*
 * Primary Internal Oscillator (INTOSC2)
 * ====================================
 * At power-up, the device is clocked from an on-chip 10 MHz oscillator
 * (INTOSC2). INTOSC2 is the primary internal clock source, and is the
 * default system clock at reset. It is used to run the boot ROM and can be
 * used as the system clock source for the application. Note that INTOSC2's
 * frequency tolerance is too loose to meet the timing requirements for CAN
 * and USB, so an external clock must be used to support those features.
 *
 * Backup Internal Oscillator (INTOSC1)
 * ====================================
 * The device also includes a redundant on-chip 10 MHz oscillator (INTOSC1).
 * INTOSC1 is a backup clock source that normally only clocks the watchdog
 * timers and missing clock detection circuit (MCD). If MCD is enabled and a
 * missing system clock is detected, the system PLL is bypassed and all system
 * clocks are connected to INTOSC1 automatically. INTOSC1 may also be manually
 * selected as the system and auxiliary clock source for debug purposes.
 */

/*
 * CLKSRCCTL1 Register, Page 300
 * -----------------------------------------------------------------------
 * | 31    06 |   05    |    04   |     03     |    02    | 01        00 |
 * | RESERVED | WDHALT1 | XTALOFF | INTOSC2OFF | RESERVED | OSCCLKSRCSEL |
 * -----------------------------------------------------------------------
 * WDHALTI: This bit determines if the WD is functional in HALT mode or not.
 *          0 = WD is not functional in the HALT mode. Clock to WD is gated when
 *              system enters HALT mode. Additionally, INTOSC1 and INTOSC2 are
 *              powered-down when system enters HALT mode
 *          1 = WD is functional in the HALT mode. Clock to WD is not gated and
 *              INTOSC1/2 are not powered-down when system enters HALT mode.
 *
 * XTALOFF: This bit turns external oscillator off:
 *          0 = Crystal (External) Oscillator On (default on reset)
 *          1 = Crystal (External) Oscillator Off
 *
 * INTOSC2OFF: This bit turns oscillator 2 off:
 *          0 = Internal Oscillator 2 On (default on reset)
 *          1 = Internal Oscillator 2 Off
 *
 * OSCCLKSRCSEL:  This bit selects the source for OSCCLK.
 *          00 = INTOSC2 (default on reset)
 *          01 = External Oscillator (XTAL)
 *          10 = INTOSC1
 *          11 = reserved (default to INTOSC1)
 */

// Clock Source: XTAL
#define CLKSRCCTL1_REG_VALUE        (  ( ( ( uint32_t )( 1 ) ) << 3 ) |     \
                                       ( ( ( uint32_t )( 1 ) ) << 0 )       \
                                    )

#define OSCCLK_FREQ_HZ          ( 20000000  )
#define SYSCLK_FREQ_HZ          ( 200000000 )
/*
 * SYSPLLMULT Register, Page 306 TRM
 * ========================================
 * |31   10 | 09   08 |    07    | 06  00 |
 * |RSERVED |   FMUL  | RESERVED |  IMULT |
 * ----------------------------------------
 * FMULT: 	SYSPLL Fractional Multiplier:
 * 		  	00 Fractional Multiplier = 0
 * 		  	01 Fractional Multiplier = 0.25
 * 		  	10 Fractional Multiplier = 0.5
 * 		  	11 Fractional Multiplier = 0.75
 *
 * IMULT: 	SYSPLL Integer Multiplier:
 * 			For 0000000 Fout = Fref (PLLBYPASS) Integer Multiplier = 1
 * 			0000001 Integer Multiplier = 1
 * 			0000010 Integer Multiplier = 2
 * 			0000011 Integer Multiplier = 3
 * 			.......
 * 			1111111 Integer Multiplier = 127
 * OSCCLK = 20MHz
 * SYSCLK = 200MHz = ( SYSPLLMULT.IMULT + SYSPLLMULT.FMULT ) / ( 2 ^ SYSCLKDIVSEL.PLLSYSCLKDIV )
 *        = ( 10 + 0.0 ) / 2^0
 */
#define SYSPLLMULT_REG_VALUE         (  ( ( ( uint32_t )( 0 ) )  << 8 ) |   \
                                        ( ( ( uint32_t )( 10 ) ) << 0 )     \
                                     )

/*
 * SYSCLKDIVSEL Register, Page 311 TRM
 * ===========================
 * | 31    06 | 05        00 |
 * | RESERVED | PLLSYSCLKDIV |
 * ---------------------------
 * PLLSYSCLKDIV:  	PLLSYSCLK Divide Select: This bit selects the divider setting for the
 * 					PLLSYSCLK.
 * 					000000 = /1
 * 					000001 = /2
 * 					000010 = /4 (default on reset)
 * 					000011 = /6
 * 					000100 = /8
 * 					......
 * 					111111 = /126
 *
 * OSCCLK = 20MHz
 * SYSCLK = 200MHz = ( SYSPLLMULT.IMULT + SYSPLLMULT.FMULT ) / ( 2 ^ SYSCLKDIVSEL.PLLSYSCLKDIV )
 *        = ( 10 + 0.0 ) / 2^0
 */
#define SYSCLKDIVSEL_REG_VALUE          ( ( uint32_t )( 0 ) )

void sysclkcfg( void );

#ifdef __cplusplus
}
#endif

#endif /* _SYSCLKCFG_H_ */
