//###########################################################################
// FILE:   main.c
// TITLE:  Unsigned Modulus on the CLA
//
//! \addtogroup cpu01_example_list
//! <h1>CLA Unsigned Modulus (cla_unsigned_modulus_cpu01)</h1>
//!
//! In this example, Task 1 of the CLA runs unsigned modulus on a
//! sequence of numbers
//!
//! \b Memory \b Allocation \n
//!  - CLA1 to CPU Message RAM
//!    - rem - Remainder
//!    - quo - quotient
//!  - CPU to CLA1 Message RAM
//!    - num - numerator
//!    - den - denominator
//!
//! \b Watch \b Variables \n
//! - num - numerator
//! - den - denominator
//! - rem - remainder
//! - quo - quotient
//!
//
//###########################################################################
// $TI Release:  $
// $Release Date: $
//###########################################################################

#include "F28x_Project.h"     // Device Headerfile and Examples Include File
#include "cla_c_shared.h"

//*****************************************************************************
// defines
//*****************************************************************************

//CLA defines
#define WAITSTEP 	asm(" RPT #255 || NOP")

//*****************************************************************************
// globals
//*****************************************************************************
//Task 1 (C) Variables

#pragma DATA_SECTION(num,"CpuToCla1MsgRAM")
uint16_t num;
#pragma DATA_SECTION(den,"CpuToCla1MsgRAM")
uint16_t den;
#pragma DATA_SECTION(rem,"Cla1ToCpuMsgRAM")
uint16_t rem;
#pragma DATA_SECTION(quo,"Cla1ToCpuMsgRAM")
uint16_t quo;

//Task 2 (C) Variables

//Task 3 (C) Variables

//Task 4 (C) Variables

//Task 5 (C) Variables

//Task 6 (C) Variables

//Task 7 (C) Variables

//Task 8 (C) Variables

//Common (C) Variables

uint32_t pass=0;
uint32_t fail=0;

//*****************************************************************************
// function prototypes
//*****************************************************************************
void CLA_configClaMemory(void);
void CLA_initCpu1Cla1(void);
void INT_init(void);

__interrupt void cla1Isr1();
__interrupt void cla1Isr2();
__interrupt void cla1Isr3();
__interrupt void cla1Isr4();
__interrupt void cla1Isr5();
__interrupt void cla1Isr6();
__interrupt void cla1Isr7();
__interrupt void cla1Isr8();
__interrupt void epwmIsr8();

//*****************************************************************************
// Start of main
//*****************************************************************************
void main(void)
{

	// Locals
	uint16_t i, j, q, r;
	uint32_t t;

// Step 1. Initialize System Control:
// PLL, WatchDog, enable Peripheral Clocks
// This example function is found in the F28M3Xx_SysCtrl.c file.
    InitSysCtrl();

// Step 2. Clear all interrupts and initialize PIE vector table:
// Disable CPU i nterrupts
    DINT;

// Initialize the PIE control registers to their default state.
// The default state is all PIE interrupts disabled and flags
// are cleared.
// This function is found in the F28M3Xx_PieCtrl.c file.
    InitPieCtrl();

// Disable CPU interrupts and clear all CPU interrupt flags:
    IER = 0x0000;
    IFR = 0x0000;

// Initialize the PIE vector table with pointers to the shell Interrupt
// Service Routines (ISR).
// This will populate the entire table, even if the interrupt
// is not used in this example.  This is useful for debug purposes.
// The shell ISR routines are found in F28M3Xx_DefaultIsr.c.
// This function is found in F28M3Xx_PieVect.c.
    InitPieVectTable();

// Step 3. Configure the CLA memory spaces first followed by
// the CLA task vectors
    CLA_configClaMemory();
    CLA_initCpu1Cla1();

// Step 4. Enable global Interrupts and higher priority real-time debug events:
    INT_init();
    EINT;  // Enable Global interrupt INTM
    ERTM;  // Enable Global realtime interrupt DBGM


    // Step 5. Run the test
    for (i = 0 ; i <= 0xFFFF; i++){
    	for(j = 0; j <= 0xFFFF; j++){
    		num = i; den = j;
//    		q   = num / den;
//    		r   = num % den;
    		t   = __rpt_subcu(num, den, 15);
    		r   = (t >> 16) & 0xFFFF;
    		q   = t & 0xFFFF;
    	    Cla1ForceTask1andWait();
    	    if((quo == q) && (rem == r)){
    	    	pass++;
    	    }else{
    	    	fail++;
    	    }
    	}
    }

    while(1);
}

//*****************************************************************************
// function definitions
//*****************************************************************************

void INT_init(void)
{
    EALLOW;
    // Configure the vectors for the end-of-task interrupt for all
    // 8 tasks
    PieVectTable.CLA1_1_INT   = &cla1Isr1;
    PieVectTable.CLA1_2_INT   = &cla1Isr2;
    PieVectTable.CLA1_3_INT   = &cla1Isr3;
    PieVectTable.CLA1_4_INT   = &cla1Isr4;
    PieVectTable.CLA1_5_INT   = &cla1Isr5;
    PieVectTable.CLA1_6_INT   = &cla1Isr6;
    PieVectTable.CLA1_7_INT   = &cla1Isr7;
    PieVectTable.CLA1_8_INT   = &cla1Isr8;

    // Enable CLA interrupts at the group and subgroup levels
    PieCtrlRegs.PIEIER11.bit.INTx1 = 1;

    IER |= (M_INT11);
    EDIS;
}
void CLA_configClaMemory(void)
{
	extern uint32_t Cla1funcsRunStart, Cla1funcsLoadStart, Cla1funcsLoadSize;
	EALLOW;
#ifdef CPU1
	// Enable CPU1.CLA1
	DevCfgRegs.DC1.bit.CPU1_CLA1 = 1;
#endif //CPU1

#ifdef _FLASH
	// Copy over code from FLASH to RAM
	memcpy((uint32_t *)&Cla1funcsRunStart, (uint32_t *)&Cla1funcsLoadStart,
			(uint32_t)&Cla1funcsLoadSize);
#endif //_FLASH

	// Initialize and wait for CLA1ToCPUMsgRAM
	MemCfgRegs.MSGxINIT.bit.INIT_CLA1TOCPU = 1;
	while(MemCfgRegs.MSGxINITDONE.bit.INITDONE_CLA1TOCPU != 1){};

	// Initialize and wait for CPUToCLA1MsgRAM
	MemCfgRegs.MSGxINIT.bit.INIT_CPUTOCLA1 = 1;
	while(MemCfgRegs.MSGxINITDONE.bit.INITDONE_CPUTOCLA1 != 1){};

	// Select LS0/1RAM to be the programming space for the CLA
	// First configure the CLA to be the master for LS5 and then
	// set the space to be a program block
	MemCfgRegs.LSxMSEL.bit.MSEL_LS0 = 1;
	MemCfgRegs.LSxCLAPGM.bit.CLAPGM_LS0 = 1;

	MemCfgRegs.LSxMSEL.bit.MSEL_LS1 = 1;
	MemCfgRegs.LSxCLAPGM.bit.CLAPGM_LS1 = 1;

	//Next configure LS4RAM and LS5RAM as data spaces for the CLA
	// First configure the CLA to be the master for LS0(1) and then
    // set the spaces to be code blocks
	MemCfgRegs.LSxMSEL.bit.MSEL_LS4 = 1;
	MemCfgRegs.LSxCLAPGM.bit.CLAPGM_LS4 = 0;

	MemCfgRegs.LSxMSEL.bit.MSEL_LS5 = 1;
	MemCfgRegs.LSxCLAPGM.bit.CLAPGM_LS5 = 0;

	EDIS;
}

void CLA_initCpu1Cla1(void)
{
	//  Compute all CLA task vectors
	// On Type-1 CLAs the MVECT registers accept full 16-bit task addresses as
	// opposed to offsets used on older Type-0 CLAs
	EALLOW;
	Cla1Regs.MVECT1 = (uint16_t)(&Cla1Task1);
	Cla1Regs.MVECT2 = (uint16_t)(&Cla1Task2);
	Cla1Regs.MVECT3 = (uint16_t)(&Cla1Task3);
	Cla1Regs.MVECT4 = (uint16_t)(&Cla1Task4);
	Cla1Regs.MVECT5 = (uint16_t)(&Cla1Task5);
	Cla1Regs.MVECT6 = (uint16_t)(&Cla1Task6);
	Cla1Regs.MVECT7 = (uint16_t)(&Cla1Task7);
	Cla1Regs.MVECT8 = (uint16_t)(&Cla1Task8);

	// Enable IACK instruction to start a task on CLA in software
	// for all  8 CLA tasks
	asm("   RPT #3 || NOP");
    Cla1Regs.MCTL.bit.IACKE = 1;
    Cla1Regs.MIER.all 	= 0x00FF;

	DmaClaSrcSelRegs.CLA1TASKSRCSEL1.bit.TASK1 = 0; // Software forced only

}
//*****************************************************************************
// ISR
//*****************************************************************************
__interrupt void cla1Isr1 ()
{
    // Acknowledge the end-of-task interrupt for task 1
    PieCtrlRegs.PIEACK.all = M_INT11;
    EDIS;
}

__interrupt void cla1Isr2 ()
{
	asm(" ESTOP0");
}

__interrupt void cla1Isr3 ()
{
	asm(" ESTOP0");
}

__interrupt void cla1Isr4 ()
{
	asm(" ESTOP0");
}
__interrupt void cla1Isr5 ()
{
	asm(" ESTOP0");
}
__interrupt void cla1Isr6 ()
{
	asm(" ESTOP0");
}
__interrupt void cla1Isr7 ()
{
	asm(" ESTOP0");
}
__interrupt void cla1Isr8 ()
{
	asm(" ESTOP0");
}

__interrupt void epwmIsr8()
{
	asm(" ESTOP0");
}
// End of File
