/* header files ***************************************************************/
//#include "f28003x_device.h"
#include "device.h"
#include "driverlib.h"
#include "Timer.h"
/* local macro definitions ****************************************************/
#define MAX_SYSTEM_TIMER   8
/* local types definitions ****************************************************/
typedef struct
{
	uint64_t u64target;
	uint32_t u32period;
	void     (*callbackFuncPtr)(void*);
	void*    callbackArgPtr;
	uint8_t  u8repeat;
} sysTimer_t;

/* local variable declarations ************************************************/
static uint64_t u64sysTimerTick = 0; // Never overflow with 64 bit and plus one every 20us
static uint16_t u16sysTimerFlag = 0; // 16 bit indicates max support 16 timer
static sysTimer_t astrsysTimer[MAX_SYSTEM_TIMER] = {0};
//static uint16_t timer20us = 0;

/* global variable declarations ***********************************************/

/* internal function prototypes ***********************************************/


/*******************************************************************************
 * Timer_u16StartTask()
 * Description: Register the timer.
 *
 * Inputs:      u32tickOffset   -> Tick offset at the first u32target
 *              u32period       -> Implement period of timer
 *              callbackFunc -> The callback function when timer timeout
 *              callbackArg  -> Input argument for callback function
 *              u8repeat       -> 0 indicate one-shot timer, 1 indicate repeater timer
 *
 * Return:      Timer ID, if INVALID_TIMER_ID indicate NO available timer
*******************************************************************************/
uint16_t Timer_u16StartTask(uint32_t u32tickOffset, uint32_t u32period, void (*callbackFunc)(void*), void* callbackArg, uint16_t u8repeat)
{
	uint16_t u16timerId;

	for (u16timerId = 0; u16timerId < MAX_SYSTEM_TIMER; u16timerId++)
	{
		// Find out the unused timer then break
		if (!(u16sysTimerFlag & (1 << u16timerId))) break;
	}

	if (u16timerId >= MAX_SYSTEM_TIMER) return INVALID_TIMER_ID;

	// Set the bit to indicate this timer has used
	u16sysTimerFlag |= (1 << u16timerId);

	uint64_t u64currentSystemTimer = Timer_u64getSystemTick();

	astrsysTimer[u16timerId].u64target = u64currentSystemTimer + u32period + u32tickOffset;
	astrsysTimer[u16timerId].u32period = u32period;
	astrsysTimer[u16timerId].u8repeat = u8repeat;
	astrsysTimer[u16timerId].callbackFuncPtr = callbackFunc;
	astrsysTimer[u16timerId].callbackArgPtr = callbackArg;

	return u16timerId;
}

/*******************************************************************************
 * releaseTimers()
 * Description: Clear the timer flag bit to release the timer.
 *
 * Inputs:      u16timerId -> The timer ID
 *
 * Return:      None
*******************************************************************************/
/*void releaseTimers(uint16_t u16timerId)
{
	if (u16timerId < MAX_SYSTEM_TIMER)
	{
		// Clear the bit to indicate this timer is unused
		u16sysTimerFlag &= ~(1 << u16timerId);
	}
}*/

/*******************************************************************************
 * Timer_Check()
 * Description: Check the timeout of all registered timer, if timeout, execute
 *              this timer callback function.
 *              Called this function on the main loop.
 *
 * Inputs:      None
 *
 * Return:      None
*******************************************************************************/
void Timer_Check(void)
{
	uint16_t u16timerId;
	uint64_t u64currentSystemTimer = Timer_u64getSystemTick();

	// Check all timer
	for (u16timerId = 0; u16timerId < MAX_SYSTEM_TIMER; u16timerId++)
	{
		// Change the time error to sign number is for the subtraction overflow of timer tick
		if ((u16sysTimerFlag & (1 << u16timerId)) && ((int64_t)(u64currentSystemTimer - astrsysTimer[u16timerId].u64target) > 0))
		{// The registered timer had timeout
			if (astrsysTimer[u16timerId].u8repeat)
			{
				// This is repeated timer, don't clear timer registered flag and update the u32target value
				astrsysTimer[u16timerId].u64target += (uint64_t)astrsysTimer[u16timerId].u32period; // Ensure the callback period
				//astrsysTimer[u16timerId].u32target = currentSystemTimer + astrsysTimer[u16timerId].period;     // Ensure the callback interval
			}
			else
			{
				// Clear the bit to indicate this timer is unused
				u16sysTimerFlag &= ~(1 << u16timerId);
			}
			// Execute this timer callback function with arguments
			(astrsysTimer[u16timerId].callbackFuncPtr)(astrsysTimer[u16timerId].callbackArgPtr);
		}
	}
}

/*******************************************************************************
 * Timer_SystemTick()
 * Description: System timer tick plus one every called, normally this function
 *              called on timer/PMW interrupt with fixed frequency.
 *              NOTE: SYS_TICKS_PER_SEC is related with the called fixed frequency,
 *              shall modify SYS_TICKS_PER_SEC according to called fixed frequency.
 *
 * Inputs:      None
 *
 * Return:      None
*******************************************************************************/
#pragma CODE_SECTION(Timer_SystemTick, ".TI.ramfunc");
void Timer_SystemTick(void)
{
	u64sysTimerTick++;	// Count up the system timer
	//timer20us++; //count up 20us timer
}

/*******************************************************************************
 * getSystemTimer()
 * Description: Get the system timer tick.
 *
 * Inputs:      None
 *
 * Return:      Current system timer tick
*******************************************************************************/
uint64_t Timer_u64getSystemTick(void)
{
	ENTER_CRITICAL();
	uint64_t u64temp = u64sysTimerTick;
	EXIT_CRITICAL();
	return u64temp;
}


//end
