//#############################################################################
//
// FILE:   device.c
//
// TITLE:  Device setup for examples.
//
//#############################################################################
//
//
// 
// C2000Ware v5.04.00.00
//
// Copyright (C) 2024 Texas Instruments Incorporated - http://www.ti.com
//
// Redistribution and use in source and binary forms, with or without 
// modification, are permitted provided that the following conditions 
// are met:
// 
//   Redistributions of source code must retain the above copyright 
//   notice, this list of conditions and the following disclaimer.
// 
//   Redistributions in binary form must reproduce the above copyright
//   notice, this list of conditions and the following disclaimer in the 
//   documentation and/or other materials provided with the   
//   distribution.
// 
//   Neither the name of Texas Instruments Incorporated nor the names of
//   its contributors may be used to endorse or promote products derived
//   from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// $
//#############################################################################

//
// Included Files
//
#include "device.h"
#include "driverlib.h"
#include "f28x_project.h"
#ifdef __cplusplus
using std::memcpy;
#endif
#ifdef CMDTOOL
#include "device_cmd.h"
#endif

static void vid_SigleStage_PWM_init(void);

//*****************************************************************************
//
// Function to initialize the device. Primarily initializes system control to a
// known state by disabling the watchdog, setting up the SYSCLKOUT frequency,
// and enabling the clocks to the peripherals.
//
// Note : In case XTAL is used as the PLL source, it is recommended to invoke
// the Device_verifyXTAL() before configuring PLL
//
//*****************************************************************************
void Device_init(void)
{
    //
    // Disable the watchdog
    //
    SysCtl_disableWatchdog();
#ifdef CMDTOOL
    CMD_init();
#endif

#ifdef _FLASH
#ifndef CMDTOOL
    //
    // Copy time critical code and flash setup code to RAM. This includes the
    // following functions: InitFlash();
    //
    // The RamfuncsLoadStart, RamfuncsLoadSize, and RamfuncsRunStart symbols
    // are created by the linker. Refer to the device .cmd file.
    //
    memcpy(&RamfuncsRunStart, &RamfuncsLoadStart, (size_t)&RamfuncsLoadSize);
#endif
    //
    // Call Flash Initialization to setup flash waitstates. This function must
    // reside in RAM.
    //
    Flash_initModule(FLASH0CTRL_BASE, FLASH0ECC_BASE, DEVICE_FLASH_WAITSTATES);
#endif

#ifdef CPU1

    //
    // Set up PLL control and clock dividers
    //
    SysCtl_setClock(DEVICE_SETCLOCK_CFG);

    //
    // Make sure the LSPCLK divider is set to the default (divide by 4)
    //
    SysCtl_setLowSpeedClock(SYSCTL_LSPCLK_PRESCALE_4);

    //
    // These asserts will check that the #defines for the clock rates in
    // device.h match the actual rates that have been configured. If they do
    // not match, check that the calculations of DEVICE_SYSCLK_FREQ and
    // DEVICE_LSPCLK_FREQ are accurate. Some examples will not perform as
    // expected if these are not correct.
    //
    ASSERT(SysCtl_getClock(DEVICE_OSCSRC_FREQ) == DEVICE_SYSCLK_FREQ);
    ASSERT(SysCtl_getLowSpeedClock(DEVICE_OSCSRC_FREQ) == DEVICE_LSPCLK_FREQ);

#ifndef _FLASH
    //
    // Call Device_cal function when run using debugger
    // This function is called as part of the Boot code. The function is called
    // in the Device_init function since during debug time resets, the boot code
    // will not be executed and the gel script will reinitialize all the
    // registers and the calibrated values will be lost.
	// Sysctl_deviceCal is a wrapper function for Device_Cal
    //
    SysCtl_deviceCal();
#endif

#endif

    //
    // Turn on all peripherals
    //
    Device_enableAllPeripherals();
}

#ifdef CPU1
//*****************************************************************************
//
// Function to boot CPU2
// Available bootmodes :
//      - BOOTMODE_BOOT_TO_FLASH_BANK0_SECTOR0
//      - BOOTMODE_BOOT_TO_FLASH_BANK0_SECTOR127_END
//      - BOOTMODE_BOOT_TO_FLASH_BANK1_SECTOR0
//      - BOOTMODE_BOOT_TO_FLASH_BANK2_SECTOR0
//      - BOOTMODE_BOOT_TO_FLASH_BANK3_SECTOR0
//      - BOOTMODE_BOOT_TO_FLASH_BANK4_SECTOR0
//      - BOOTMODE_BOOT_TO_FLASH_BANK4_SECTOR127_END
//      - BOOTMODE_BOOT_TO_SECURE_FLASH_BANK0_SECTOR0
//      - BOOTMODE_BOOT_TO_SECURE_FLASH_BANK1_SECTOR0
//      - BOOTMODE_BOOT_TO_SECURE_FLASH_BANK2_SECTOR0
//      - BOOTMODE_BOOT_TO_SECURE_FLASH_BANK3_SECTOR0
//      - BOOTMODE_BOOT_TO_SECURE_FLASH_BANK4_SECTOR0
//      - BOOTMODE_IPC_MSGRAM_COPY_BOOT_TO_M1RAM
//      - BOOTMODE_BOOT_TO_M0RAM
//      - BOOTMODE_BOOT_TO_FWU_FLASH
//      - BOOTMODE_BOOT_TO_FWU_FLASH_ALT1
//      - BOOTMODE_BOOT_TO_FWU_FLASH_ALT2
//      - BOOTMODE_BOOT_TO_FWU_FLASH_ALT3
//
// Note that while using BOOTMODE_IPC_MSGRAM_COPY_BOOT_TO_M1RAM,
// BOOTMODE_IPC_MSGRAM_COPY_LENGTH_xxxW must be ORed with the bootmode parameter
//
// This function must be called after Device_init function
//
//*****************************************************************************
void Device_bootCPU2(uint32_t bootmode)
{
    //
    // Configure the CPU1TOCPU2IPCBOOTMODE register
    //
    IPC_setBootMode(IPC_CPU1_L_CPU2_R,
                    (BOOT_KEY | bootmode));

    //
    // Set IPC Flag 0
    //
    IPC_setFlagLtoR(IPC_CPU1_L_CPU2_R, IPC_FLAG0);

    //
    // Bring CPU2 out of reset. Wait for CPU2 to go out of reset.
    //
    SysCtl_controlCPU2Reset(SYSCTL_CORE_DEACTIVE);
    while(SysCtl_isCPU2Reset() == 0x1U);
}
#endif

//*****************************************************************************
//
// Function to turn on all peripherals, enabling reads and writes to the
// peripherals' registers.
//
// Note that to reduce power, unused peripherals should be disabled.
//
//*****************************************************************************
void Device_enableAllPeripherals(void)
{
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_DMA);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TIMER0);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TIMER1);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TIMER2);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CPUBGCRC);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_ERAD);
#ifdef CPU1
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CLA1);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CLA1BGCRC);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_GTBCLKSYNC);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_EMIF1);
#endif

    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_EPWM1);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_EPWM2);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_EPWM3);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_EPWM4);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_EPWM5);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_EPWM6);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_EPWM7);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_EPWM8);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_EPWM9);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_EPWM10);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_EPWM11);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_EPWM12);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_EPWM13);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_EPWM14);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_EPWM15);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_EPWM16);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_EPWM17);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_EPWM18);

    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_ECAP1);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_ECAP2);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_ECAP3);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_ECAP4);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_ECAP5);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_ECAP6);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_ECAP7);

    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_EQEP1);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_EQEP2);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_EQEP3);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_EQEP4);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_EQEP5);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_EQEP6);

    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_SD1);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_SD2);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_SD3);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_SD4);

    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_SCIA);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_SCIB);

    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_UARTA);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_UARTB);

    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_SPIA);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_SPIB);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_SPIC);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_SPID);

    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_I2CA);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_I2CB);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_PMBUSA);

    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CANA);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_MCANA);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_MCANB);

    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_USBA);

    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_ADCA);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_ADCB);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_ADCC);

    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CMPSS1);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CMPSS2);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CMPSS3);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CMPSS4);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CMPSS5);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CMPSS6);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CMPSS7);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CMPSS8);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CMPSS9);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CMPSS10);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CMPSS11);

    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_DACA);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_DACC);

    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CLB1);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CLB2);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CLB3);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CLB4);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CLB5);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CLB6);

    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_FSITXA);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_FSITXB);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_FSIRXA);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_FSIRXB);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_FSIRXC);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_FSIRXD);

    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_LINA);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_LINB);


    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_DCC0);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_DCC1);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_DCC2);

    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_ECAT);

    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_HRCAL0);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_HRCAL1);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_HRCAL2);

    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_AESA);

    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_EPG1);

    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_ADCCHECKER1);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_ADCCHECKER2);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_ADCCHECKER3);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_ADCCHECKER4);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_ADCCHECKER5);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_ADCCHECKER6);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_ADCCHECKER7);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_ADCCHECKER8);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_ADCSEAGGRCPU1);
}

//*****************************************************************************
//
// Function to disable pin locks and enable pullups on GPIOs.
//
//*****************************************************************************
void Device_initGPIO(void)
{
    //
    // Disable pin locks.
    //
    GPIO_unlockPortConfig(GPIO_PORT_A, 0xFFFFFFFF);
    GPIO_unlockPortConfig(GPIO_PORT_B, 0xFFFFFFFF);
    GPIO_unlockPortConfig(GPIO_PORT_C, 0xFFFFFFFF);
    GPIO_unlockPortConfig(GPIO_PORT_D, 0xFFFFFFFF);
    GPIO_unlockPortConfig(GPIO_PORT_E, 0xFFFFFFFF);
    GPIO_unlockPortConfig(GPIO_PORT_F, 0xFFFFFFFF);
    GPIO_unlockPortConfig(GPIO_PORT_H, 0xFFFFFFFF);

    EALLOW;
    //
    // Enable PWM5A,B on GPIO8,GPIO9
//    //
//    GpioCtrlRegs.GPAPUD.bit.GPIO0 = 1u;        // Pull up disabled
//    GpioCtrlRegs.GPAGMUX1.bit.GPIO0 = 0u;
//    GpioCtrlRegs.GPAMUX1.bit.GPIO0 = 1u;       // Configure GPIO0 as EPWM1A
//    GpioCtrlRegs.GPAPUD.bit.GPIO0 = 1u;        // Pull up disabled
//    GpioCtrlRegs.GPAGMUX1.bit.GPIO0 = 0u;
//    GpioCtrlRegs.GPAMUX1.bit.GPIO0 = 1u;       // Configure GPIO1 as EPWM1B

//    GpioCtrlRegs.GPHPUD.bit.GPIO227 = 1u;        // Pull up disabled
//    GpioCtrlRegs.GPHMUX1.bit.GPIO227 = 1u;       //normal GPIO
//    GpioCtrlRegs.GPHDIR.bit.GPIO227 = 0u;        //GPIO25 = output
//    AnalogSubsysRegs.AGPIOCTRLH.bit.GPIO227 = 1u;
//    GpioCtrlRegs.GPHPUD.bit.GPIO229 = 1u;        // Pull up disabled
//    GpioCtrlRegs.GPHMUX1.bit.GPIO229 = 1u;       //normal GPIO
//    GpioCtrlRegs.GPHDIR.bit.GPIO229 = 0u;        //GPIO25 = output
//    AnalogSubsysRegs.AGPIOCTRLH.bit.GPIO229 = 1u;
//    GpioCtrlRegs.GPAPUD.bit.GPIO14 = 1u;        // Pull up disabled
//    GpioCtrlRegs.GPAMUX1.bit.GPIO14 = 0u;       //normal GPIO
//    GpioCtrlRegs.GPADIR.bit.GPIO14 = 1u;        //GPIO25 = output


    //debug pin on GPIO12
//    GpioCtrlRegs.GPAPUD.bit.GPIO12 = 1u;        // Pull up disabled
//    GpioCtrlRegs.GPAMUX1.bit.GPIO12 = 0u;       //normal GPIO
//    GpioCtrlRegs.GPADIR.bit.GPIO12 = 1u;        //GPIO25 = output
//    GpioDataRegs.GPASET.bit.GPIO12 = 1u;        //write prohibited
//
//    //debug pin on GPIO13
//    GpioCtrlRegs.GPAPUD.bit.GPIO13 = 1u;        // Pull up disabled
//    GpioCtrlRegs.GPAMUX1.bit.GPIO13 = 0u;       //normal GPIO
//    GpioCtrlRegs.GPADIR.bit.GPIO13 = 1u;        //GPIO25 = output
//    GpioDataRegs.GPASET.bit.GPIO13 = 1u;        //write prohibited

    GPIO_setPinConfig(GPIO_227_GPIO227);
    // AIO -> Analog mode selected
    GPIO_setAnalogMode(227, GPIO_ANALOG_ENABLED);
    // Analog PinMux for A2
    GPIO_setPinConfig(GPIO_229_GPIO229);
    // AIO -> Analog mode selected
    GPIO_setAnalogMode(229, GPIO_ANALOG_ENABLED);
    //
    // OUTPUTXBAR3 -> myOUTPUTXBAR0 Pinmux
    //
    GPIO_setPinConfig(GPIO_14_OUTPUTXBAR3);

    EDIS;
}


extern __interrupt void Cla1Task1();

void Device_initCLA(void)
{
#ifdef _FLASH
#ifndef CMDTOOL // Linker command tool is not used

    extern uint32_t Cla1ProgRunStart, Cla1ProgLoadStart, Cla1ProgLoadSize;
    extern uint32_t Cla1ConstRunStart, Cla1ConstLoadStart, Cla1ConstLoadSize;

    //
    // Copy the program and constants from FLASH to RAM before configuring
    // the CLA
    //
    memcpy((uint32_t *)&Cla1ProgRunStart, (uint32_t *)&Cla1ProgLoadStart,
           (uint32_t)&Cla1ProgLoadSize);
    memcpy((uint32_t *)&Cla1ConstRunStart, (uint32_t *)&Cla1ConstLoadStart,
        (uint32_t)&Cla1ConstLoadSize );


#endif //CMDTOOL
#endif //_FLASH


    EALLOW;
#pragma diag_suppress=770
    //
    // CLA Task 1
    //

    Cla1Regs.MVECT1 = (uint16_t)&Cla1Task1;
    DmaClaSrcSelRegs.CLA1TASKSRCSEL1.bit.TASK1 = CLA_TRIGGER_SOFTWARE;
#pragma diag_warning=770

    Cla1Regs.MCTL.bit.IACKE = 1;
    Cla1Regs.MIER.all = 0xFF;
    EDIS;

}

//*****************************************************************************
//
// Function to configure the ADC.
//
//*****************************************************************************
void Device_initADC(void)
{
    ADC_setVREF(ADCA_BASE, ADC_REFERENCE_EXTERNAL, ADC_REFERENCE_3_3V);
//    ADC_setVREF(ADCB_BASE, ADC_REFERENCE_EXTERNAL, ADC_REFERENCE_3_3V);
//    ADC_setVREF(ADCC_BASE, ADC_REFERENCE_EXTERNAL, ADC_REFERENCE_3_3V);

    EALLOW;
    /*ADC clock prescaler /1 */
    AdcaRegs.ADCCTL2.bit.PRESCALE = ADC_CLK_DIV_1_0;
//    AdcbRegs.ADCCTL2.bit.PRESCALE = ADC_CLK_DIV_1_0;
//    AdccRegs.ADCCTL2.bit.PRESCALE = ADC_CLK_DIV_1_0;

//    AdcaRegs.ADCCTL2.bit.RESOLUTION = ADC_RESOLUTION_12BIT;
//    AdcaRegs.ADCCTL2.bit.SIGNALMODE = ADC_MODE_SINGLE_ENDED;

    /* ADC interrupt pulse generation method*/
    AdcaRegs.ADCCTL1.bit.INTPULSEPOS = 1u;
//    AdcbRegs.ADCCTL1.bit.INTPULSEPOS = 1u;
//    AdccRegs.ADCCTL1.bit.INTPULSEPOS = 1u;

    /* ADC analog circuit in ADC core are powered on, with a delay of 1ms*/
    AdcaRegs.ADCCTL1.bit.ADCPWDNZ = 1u;
//    AdcbRegs.ADCCTL1.bit.ADCPWDNZ = 1u;
//    AdccRegs.ADCCTL1.bit.ADCPWDNZ = 1u;

    AdcaRegs.ADCCTL1.bit.TDMAEN = 0u;

    AdcaRegs.ADCBURSTCTL.bit.BURSTEN = 0;

//    AdcaRegs.ADCSOCPRICTL.bit.SOCPRIORITY = ADC_PRI_ALL_ROUND_ROBIN;

    /* ADC SOC and interrupt configuration*/
    AdcaRegs.ADCINTSEL1N2.bit.INT1E = 1u;       //enable ITN1 flag
    AdcaRegs.ADCINTSEL1N2.bit.INT1CONT = 0u;
    AdcaRegs.ADCINTSEL1N2.bit.INT1SEL = 0u;      //end of SOC0 will set INT1 flag
    AdcaRegs.ADCINTFLGCLR.bit.ADCINT1 = 1u;     //make sure INT1 flag is cleared
    /*------------------ADCa-------------------*/
    //A6 for AD_VAC_SENSE
    AdcaRegs.ADCSOC0CTL.bit.CHSEL = 4u;
    AdcaRegs.ADCSOC0CTL.bit.TRIGSEL = ADC_TRIGGER_EPWM1_SOCA;
    AdcaRegs.ADCSOC0CTL.bit.ACQPS = 9u;     //10 cycles

    EDIS;
}

//*****************************************************************************
//
// Function to configure the ePWM.
//
//*****************************************************************************
void Device_initePWM(void)
{
    EALLOW;
    // Disable TBCLK before EPWM Initialization
    CpuSysRegs.PCLKCR0.bit.TBCLKSYNC = 0u;
    HWREGH(CLKCFG_BASE + SYSCTL_O_PERCLKDIVSEL) &= 0xFE;
    EDIS;

    vid_SigleStage_PWM_init();

    EALLOW;
    // Enable TBCLK after EPWM Initialization
    CpuSysRegs.PCLKCR0.bit.TBCLKSYNC = 1u;
    EPwm1Regs.TBCTL.bit.SWFSYNC = 1u;

    EDIS;
}
//#define PFC_PWM_60K                 1429u
//static void vid_SigleStage_PWM_init(void)
//{
//    /*-------ePWM5A/B-----------------*/
//    //configure counter mode
//    EPwm1Regs.TBCTL.bit.CTRMODE = TB_COUNT_UP;//TB_COUNT_UPDOWN;
//    EPwm1Regs.TBCTL.bit.PHSEN = TB_DISABLE;
//    EPwm1Regs.TBCTL.bit.HSPCLKDIV = EPWM_HSCLOCK_DIVIDER_1;
//    EPwm1Regs.TBCTL.bit.CLKDIV = EPWM_CLOCK_DIVIDER_1;
//    EPwm1Regs.TBCTL.bit.SWFSYNC = 0u;
//    EPwm1Regs.TBCTL.bit.PRDLD = TB_IMMEDIATE;          // set immediate load the active register
//    //Period = 120M /Freq_PWM. Period100k = 120M/100K=1200 TBPRD = Period/2 +1 = 601;
//    EPwm1Regs.TBPRD = PFC_PWM_60K;//PSFB_PEROID_NORMAL-1;
//    EPwm1Regs.TBPHS.bit.TBPHS = 0u;
//    EPwm1Regs.TBCTR = 0u;
//    EPwm1Regs.CMPA.bit.CMPA = PFC_PWM_60K/2;//(PSFB_PEROID_NORMAL-1)>>1;              // Fix duty at 50%
//    EPwm1Regs.CMPB.bit.CMPB = PFC_PWM_60K/4;
//    //configure the shadow register, load CMP when CTR == 0
//    EPwm1Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO_PRD;   //has no effect in immediate mode
//    EPwm1Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;
//    EPwm1Regs.CMPCTL.bit.LOADBMODE = CC_CTR_ZERO_PRD;   //has no effect in immediate mode
//    EPwm1Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;
//    //configure action qualifier register
//    EPwm1Regs.AQCTLA.bit.CAU = AQ_SET;              //output high level when TBTCR=CMPA rising edge
////    EPwm1Regs.AQCTLA.bit.CAD = AQ_CLEAR;            //output low level when TBTCR=CMPA falling edge
//    EPwm1Regs.AQCTLA.bit.PRD = AQ_CLEAR;
//    EPwm1Regs.AQCTLB.bit.CAU = AQ_CLEAR;            //output low level when TBTCR=CMPA rising dege
//    EPwm1Regs.AQCTLB.bit.ZRO = AQ_SET;              //output high level when TBCTR=CMPA falling edge
//
//    //configure dead band register
//    EPwm1Regs.DBCTL.bit.IN_MODE = DBA_ALL;
//    EPwm1Regs.DBCTL.bit.POLSEL = DB_ACTV_HIC;
//    EPwm1Regs.DBCTL.bit.OUT_MODE = DB_FULL_ENABLE;
//    EPwm1Regs.DBRED.bit.DBRED = 1u;         //rising edge time delay
//    EPwm1Regs.DBFED.bit.DBFED = 1u;         //falling edge time delay
//
//    EPwm1Regs.ETSEL.bit.INTEN = 0u;             //enable PWM interrupt
//    EPwm1Regs.ETSEL.bit.INTSEL = ET_CTR_ZERO;//ET_CTR_PRDZERO;//ET_CTR_PRD;    //trigger interrupt at CTR=PRD
//    EPwm1Regs.ETSEL.bit.SOCAEN = 1u;            //enable SOC on A group
//    EPwm1Regs.ETSEL.bit.SOCASEL = ET_CTR_PRD;   //
//
//    EPwm1Regs.ETPS.bit.SOCAPRD = ET_1ST;
//    EPwm1Regs.ETPS.bit.INTPRD = ET_2ND;         //generate INT on 2nd event
//
//    EPwm1Regs.ETSEL.bit.INTEN = 1u;
//
//}
void Xbar_Init(void)
{
    EALLOW;

    OutputXbarRegs.EXT64_OUTPUTLATCHENABLE.bit.OUTPUT3 = 0u;
    OutputXbarRegs.EXT64_OUTPUTINV.bit.OUTPUT3 = 0u;
    OutputXbarRegs.EXT64_OUTPUT3MUX0TO15CFG.bit.MUX0 = 0u;//Mux configuration
    OutputXbarRegs.EXT64_OUTPUT3MUXENABLE.bit.MUX0 = 1u;

    EDIS;


}

#define myCMPSS0_BASE CMPSS1_BASE
void Device_initeCMPSS(void)
{
    EALLOW;

    AnalogSubsysRegs.CMPHNMXSEL.bit.CMP1HNMXSEL = 1u;// Select the value for CMP1HPMXSEL.
    AnalogSubsysRegs.CMPLNMXSEL.bit.CMP1LNMXSEL = 0u;// Select the value for CMP1LPMXSEL.
    //
    // Sets the configuration for the high comparator.
    //
    Cmpss1Regs.COMPCTL.bit.COMPHSOURCE = 0u;
    Cmpss1Regs.COMPCTL.bit.COMPHINV = 0u;
    Cmpss1Regs.COMPCTL.bit.ASYNCHEN = 0u;
    //
    // Sets the configuration for the low comparator.
    //
    Cmpss1Regs.COMPCTL.bit.COMPLSOURCE = 0u;
    Cmpss1Regs.COMPCTL.bit.COMPLINV = 0u;
    Cmpss1Regs.COMPCTL.bit.ASYNCLEN = 0u;
    //
    // Sets the configuration for the internal comparator DACs.
    //
    Cmpss1Regs.COMPDACHCTL.bit.SWLOADSEL = 0u;
    Cmpss1Regs.COMPDACHCTL.bit.DACSOURCE = 0u;
    Cmpss1Regs.COMPDACHCTL.bit.SELREF = 0u;

    Cmpss1Regs.COMPDACLCTL.bit.DACSOURCE = 0u;

    Cmpss1Regs.DACHVALS.bit.DACVAL = 2048u;// Sets the value of the internal DAC of the high comparator.
    Cmpss1Regs.DACLVALS.bit.DACVAL = 0u;// Sets the value of the internal DAC of the low comparator.

    Cmpss1Regs.DACHVALS2.bit.DACVAL = 0u;// Sets the value of the internal DAC of the high comparator for Diode Emulation Support.
    Cmpss1Regs.DACLVALS2.bit.DACVAL = 0u;// Sets the value of the internal DAC of the low comparator for Diode Emulation Support.

    Cmpss1Regs.COMPDACHCTL2.bit.DEACTIVESEL = CMPSS_DEACTIVE_EPWM1;// Set the DEACTIVE signal source for CMPSS
    //
    //  Configures the digital filter of the high comparator.
    //
    Cmpss1Regs.CTRIPHFILCLKCTL = 1023;
    Cmpss1Regs.CTRIPHFILCLKCTL2.bit.CLKPRESCALEU = (uint8_t)(1023 >> 16);
    Cmpss1Regs.CTRIPHFILCTL.bit.SAMPWIN = 31;
    Cmpss1Regs.CTRIPHFILCTL.bit.THRESH = 30;
    //
    // Configures the digital filter of the low comparator.
    //
    Cmpss1Regs.CTRIPLFILCLKCTL = 0u;
    Cmpss1Regs.CTRIPLFILCLKCTL2.bit.CLKPRESCALEU = 0u;
    Cmpss1Regs.CTRIPLFILCTL.bit.SAMPWIN = 0u;
    Cmpss1Regs.CTRIPLFILCTL.bit.THRESH = 0u;

    Cmpss1Regs.CTRIPHFILCTL.bit.FILINIT = 1u;// Initializes the digital filter of the high comparator.

    Cmpss1Regs.COMPCTL.bit.CTRIPOUTHSEL = 2u;//Sets the output signal configuration for the high comparator.
    Cmpss1Regs.COMPCTL.bit.CTRIPHSEL = 2u;
    //
    // Sets the output signal configuration for the low comparator.
    //
    Cmpss1Regs.COMPCTL.bit.CTRIPOUTLSEL = 0u;
    Cmpss1Regs.COMPCTL.bit.CTRIPLSEL = 0u;

    Cmpss1Regs.COMPHYSCTL.bit.COMPHYS = 0u;// Sets the comparator hysteresis settings.
    //
    // Configures the comparator subsystem's high ramp generator.
    //
    CMPSS_configRampHigh(myCMPSS0_BASE, CMPSS_RAMP_DIR_DOWN, 0U,0U,0U,1U,true);
    //
    // Configures the comparator subsystem's low ramp generator.
    //
    CMPSS_configRampLow(myCMPSS0_BASE, CMPSS_RAMP_DIR_DOWN, 0U,0U,0U,1U,true);

    Cmpss1Regs.RAMPHCTLS.bit.RAMPCLKDIV = CMPSS_RAMP_CLOCK_DIV1;// Configures the high comparator's ramp generator clock divider
    Cmpss1Regs.RAMPLCTLS.bit.RAMPCLKDIV = CMPSS_RAMP_CLOCK_DIV1;// Configures the low comparator's ramp generator clock divider

    Cmpss1Regs.COMPSTSCLR.bit.HSYNCCLREN = 0u;// Disables reset of HIGH comparator digital filter output latch on PWMSYNC
    Cmpss1Regs.COMPSTSCLR.bit.LSYNCCLREN = 0u;// Disables reset of LOW comparator digital filter output latch on PWMSYNC
    //

    //
    Cmpss1Regs.COMPDACHCTL.bit.BLANKSOURCE = 0u;// Sets the ePWM module blanking signal that holds trip in reset.

    Cmpss1Regs.COMPDACHCTL.bit.BLANKEN = 0;// Disables an ePWM blanking signal from holding trip in reset.
    //
    // Configures whether or not the digital filter latches are reset by PWMSYNC
    //
    Cmpss1Regs.COMPSTSCLR.bit.HSYNCCLREN = 0u;
    Cmpss1Regs.COMPSTSCLR.bit.LSYNCCLREN = 0u;

    EDIS;

    EALLOW;
    Cmpss1Regs.COMPCTL.bit.COMPDACE = 1u;// Enables the CMPSS module.
    EDIS;

    //
    // Delay for CMPSS DAC to power up.
    //
    DEVICE_DELAY_US(500);

    Xbar_Init();
}



//*****************************************************************************
//
// Function to verify the XTAL frequency
// freq is the XTAL frequency in MHz
// The function return true if the the actual XTAL frequency matches with the
// input value
//
// Note that this function assumes that the PLL is not already configured and
// hence uses SysClk freq = 10MHz for DCC calculation
//
//*****************************************************************************
bool Device_verifyXTAL(float freq)
{
    //
    // Use DCC to verify the XTAL frequency using INTOSC2 as reference clock
    //

    //
    // Turn on XTAL and wait for it to power up using X1CNT
    //
    SysCtl_turnOnOsc(SYSCTL_OSCSRC_XTAL);
    SysCtl_clearExternalOscCounterValue();
    while(SysCtl_getExternalOscCounterValue() != SYSCTL_X1CNT_X1CNT_M);

    //
    // Enable DCC0 clock
    //
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_DCC0);

    //
    // Insert atleast 5 cycles delay after enabling the peripheral clock
    //
    asm(" RPT #5 || NOP");

    //
    // Configures XTAL as CLKSRC0 and INTOSC2 as CLKSRC1
    // Fclk0 = XTAL frequency (input parameter)
    // Fclk1 = INTOSC2 frequency = 10MHz
    //
    // Configuring DCC error tolerance of +/-1%
    // INTOSC2 can have a variance in frequency of +/-10%
    //
    // Assuming PLL is not already configured, SysClk freq = 10MHz
    //
    // Note : Update the tolerance and INTOSC2 frequency variance as necessary.
    //
    return (DCC_verifyClockFrequency(DCC0_BASE,
                                     DCC_COUNT1SRC_INTOSC2, 10.0F,
                                     DCC_COUNT0SRC_XTAL, freq,
                                     1.0F, 10.0F, 10.0F));

}

//*****************************************************************************
//
// Error handling function to be called when an ASSERT is violated
//
//*****************************************************************************
void __error__(const char *filename, uint32_t line)
{
    //
    // An ASSERT condition was evaluated as false. You can use the filename and
    // line parameters to determine what went wrong.
    //
    ESTOP0;
}
