#include "dmctype.h"
#include "cm.h"
#include "driverlib_cm.h"
#include "I2C_ReadTest.h"

#define I2C_CLK_TRANSITION			10 // us

// Local function
void CM_I2C_StartMultiByteWrite( void );
void CM_I2C_StartMultiByteRead( void );
void CM_I2C_SendStop( void );
EI2C_ERROR_STATUS CM_I2C_ErrorCheck( void );
EI2C_STATUS CM_I2C_ReadMultiBytesWithStartAddr( Uint16 Slave_address, Uint16 Start_address, Uint16 *pRcvBuffer, Uint16 Rcv_DataBytes );

//---------------------------------------------------------
// Note:
// CM-I2C Write/Read procedures all follow Chapter 46.3.6
// in F2838x technical sheet.
//---------------------------------------------------------
void CM_I2C_StartMultiByteWrite( void )
{
	// write xxx0x011 to I2CMCS
	I2C_setMasterConfig( I2C0_BASE, I2C_MASTER_CMD_BURST_SEND_START );
}

void CM_I2C_StartMultiByteRead( void )
{
	// write xxx01011 to I2CMCS
	I2C_setMasterConfig( I2C0_BASE, I2C_MASTER_CMD_BURST_RECEIVE_START );
}

void CM_I2C_SendStop( void )
{
	// write xxx0x100 to I2CMCS
	I2C_setMasterConfig( I2C0_BASE, I2C_MASTER_CMD_BURST_SEND_STOP );
}

EI2C_ERROR_STATUS CM_I2C_ErrorCheck( void )
{
	uint32_t MCS_Error;
	Uint16 MCS_BusyCnt = 0;

	do {
		MCS_Error = HWREG(I2C0_BASE + I2C_O_MCS);

		// address NACK
		if( MCS_Error & I2C_MCS_ADRACK ) {
			// error by NACK will automatically send STOP condition
			// ref: https://e2e.ti.com/support/microcontrollers/c2000/f/171/p/964793/3568622
//			CM_I2C_SendStop();
			return I2C_ERR_NO_ADDRESS_ACK;
		}
		else if( MCS_Error & I2C_MCS_DATACK ) {
			CM_I2C_SendStop();
			return I2C_ERR_NO_DATA_ACK;
		}
		// lost arbitration
		else if( MCS_Error & I2C_MCS_ARBLST ) {
			return I2C_ERR_ARBL;
		}
		// module error
		else if( MCS_Error & I2C_MCS_ERROR ) {
			CM_I2C_SendStop();
			return I2C_ERR_MODULE_ERR;
		}
		// busy
		else if( MCS_Error & I2C_MCS_BUSY ) {
			// limited polling: retry failed
			if( ++MCS_BusyCnt > MAX_MASTER_BUSY_RETRY_CNT ) {
				CM_I2C_SendStop();
				return I2C_ERR_BUSY;
			}
			else {
				// next polling
				continue;
			}
		}
		// Succeed
		else {
			return I2C_ERR_NONE;
		}
	} while( true );
}

EI2C_STATUS CM_I2C_ReadMultiBytesWithStartAddr( Uint16 Slave_address, Uint16 Start_address, Uint16 *pRcvBuffer, Uint16 Rcv_DataBytes )
{
	Uint16 ByteIndex = 0;
	Uint16 BuffIndex = 0;
	Uint16 Start_address_bytes[ 2 ] = { 0 };
	EI2C_ERROR_STATUS ErrorStatus;

	// init start address bytes
	Start_address_bytes[ 0 ] = ( ( Start_address >> 8 ) & 0x00FF );
	Start_address_bytes[ 1 ] = Start_address & 0x00FF;

	//-------------------------------------
	// first sending start address
	//-------------------------------------
	I2C_setSlaveAddress( I2C0_BASE, Slave_address, I2C_MASTER_WRITE );

	I2C_putMasterData( I2C0_BASE, Start_address_bytes[ 0 ] );

	// wait for bus not busy
	if( I2C_isBusBusy( I2C0_BASE ) == true ) {
		return I2C_STAT_WRITE_BB;
	}

	CM_I2C_StartMultiByteWrite();

	// avoid polling master busy too fast
	DEVICE_DELAY_US(20);

	// wait for master busy and error check
	ErrorStatus = CM_I2C_ErrorCheck();
	if( ErrorStatus == I2C_ERR_BUSY ) {
		return I2C_STAT_WRITE_MB;
	}
	else if( ErrorStatus != I2C_ERR_NONE ) {
		return I2C_STAT_WRITE_FAIL;
	}

	// low address byte transmission
	// put data
	I2C_putMasterData( I2C0_BASE, Start_address_bytes[ 1 ] );

	// wait for one clock transition
	DEVICE_DELAY_US( I2C_CLK_TRANSITION );

	// write xxx0x101 to I2CMCS
	I2C_setMasterConfig( I2C0_BASE, I2C_MASTER_CMD_BURST_SEND_FINISH );

	// avoid polling master busy too fast
	DEVICE_DELAY_US(20);

	// wait for master busy and error check
	ErrorStatus = CM_I2C_ErrorCheck();
	if( ErrorStatus == I2C_ERR_BUSY ) {
		return I2C_STAT_WRITE_MB;
	}
	else if( ErrorStatus != I2C_ERR_NONE ) {
		return I2C_STAT_WRITE_FAIL;
	}

	//-------------------------------------
	// second receiving data
	//-------------------------------------
	I2C_setSlaveAddress( I2C0_BASE, Slave_address, I2C_MASTER_READ );

	// wait for bus not busy
	if( I2C_isBusBusy( I2C0_BASE ) == true ) {
		return I2C_STAT_READ_BB;
	}

	CM_I2C_StartMultiByteRead();

	// reception
	do {
		// avoid polling master busy too fast
		DEVICE_DELAY_US(20);

		// wait for master busy and error check
		ErrorStatus = CM_I2C_ErrorCheck();
		if( ErrorStatus == I2C_ERR_BUSY ) {
			return I2C_STAT_READ_MB;
		}
		else if( ErrorStatus != I2C_ERR_NONE ) {
			return I2C_STAT_READ_FAIL;
		}

		BuffIndex = ByteIndex >> 1;

		// low byte: even index
		if( ( ByteIndex & 0x0001 ) == 0 ) {
			pRcvBuffer[ BuffIndex ] = I2C_getMasterData( I2C0_BASE ) & 0x000000FFul;
		}
		// high byte: odd index
		else {
			pRcvBuffer[ BuffIndex ] |= ( ( I2C_getMasterData( I2C0_BASE ) & 0x000000FFul ) << 8 );
		}

		// the second last byte treatment
		if( ByteIndex == ( Rcv_DataBytes - 2 ) ) {
			// write xxx00101 to I2CMCS
			I2C_setMasterConfig( I2C0_BASE, I2C_MASTER_CMD_BURST_RECEIVE_FINISH );
		}
		// 0 ~ third last
		else if( ByteIndex < ( Rcv_DataBytes - 2 ) ) {
			// write xxx01001 to I2CMCS
			I2C_setMasterConfig( I2C0_BASE, I2C_MASTER_CMD_BURST_RECEIVE_CONT );
		}
		// caution: the last byte do nothing

		// index counts up
		ByteIndex++;
	} while( ByteIndex < Rcv_DataBytes );

	DEVICE_DELAY_US(20);

	// wait for master busy and error check
	ErrorStatus = CM_I2C_ErrorCheck();
	if( ErrorStatus == I2C_ERR_BUSY ) {
		return I2C_STAT_READ_MB;
	}
	else if( ErrorStatus != I2C_ERR_NONE ) {
		return I2C_STAT_READ_FAIL;
	}

	return I2C_STAT_READ_OK;
}
