#include "dmctype.h"
#include "cm.h"
#include "driverlib_cm.h"
#include "I2C_Test.h"

// Local function
void CM_I2C_StartMultiByteWrite( void );
void CM_I2C_StartMultiByteRead( void );
void CM_I2C_SendStop( void );
EI2C_ERROR_STATUS CM_I2C_ErrorCheck( void );
EI2C_STATUS CM_I2C_WriteMultiBytesWithStartAddr( Uint16 Slave_address, Uint16 Start_address, Uint16 *pTransBuffer, Uint16 Trans_DataBytes );

//---------------------------------------------------------
// Note:
// CM-I2C Write/Read procedures all follow Chapter 46.3.6
// in F2838x technical sheet.
//---------------------------------------------------------
void CM_I2C_StartMultiByteWrite( void )
{
	// write xxx0x011 to I2CMCS
	I2C_setMasterConfig( I2C0_BASE, I2C_MASTER_CMD_BURST_SEND_START );
}

void CM_I2C_StartMultiByteRead( void )
{
	// write xxx01011 to I2CMCS
	I2C_setMasterConfig( I2C0_BASE, I2C_MASTER_CMD_BURST_RECEIVE_START );
}

void CM_I2C_SendStop( void )
{
	// write xxx0x100 to I2CMCS
	I2C_setMasterConfig( I2C0_BASE, I2C_MASTER_CMD_BURST_SEND_STOP );
}

EI2C_ERROR_STATUS CM_I2C_ErrorCheck( void )
{
	uint32_t MCS_Error;
	Uint16 MCS_BusyCnt = 0;

	do {
		MCS_Error = HWREG(I2C0_BASE + I2C_O_MCS);

		// address NACK
		if( MCS_Error & I2C_MCS_ADRACK ) {
//			CM_I2C_SendStop();
			return I2C_ERR_NO_ADDRESS_ACK;
		}
		else if( MCS_Error & I2C_MCS_DATACK ) {
			CM_I2C_SendStop();
			return I2C_ERR_NO_DATA_ACK;
		}
		// lost arbitration
		else if( MCS_Error & I2C_MCS_ARBLST ) {
			return I2C_ERR_ARBL;
		}
		// module error
		else if( MCS_Error & I2C_MCS_ERROR ) {
			CM_I2C_SendStop();
			return I2C_ERR_MODULE_ERR;
		}
		// busy
		else if( MCS_Error & I2C_MCS_BUSY ) {
			// limited polling: retry failed
			if( ++MCS_BusyCnt > MAX_MASTER_BUSY_RETRY_CNT ) {
				CM_I2C_SendStop();
				return I2C_ERR_BUSY;
			}
			else {
				// next polling
				continue;
			}
		}
		// Succeed
		else {
			return I2C_ERR_NONE;
		}
	} while( true );
}

//-------------------------------------------------------------------------
// First sending start address, then data bytes from buffer saved in words
//-------------------------------------------------------------------------
EI2C_STATUS CM_I2C_WriteMultiBytesWithStartAddr( Uint16 Slave_address, Uint16 Start_address, Uint16 *pTransBuffer, Uint16 Trans_DataBytes )
{
	Uint16 ByteIndex = 0;
	Uint16 BuffIndex = 0;
	Uint16 Start_address_bytes[ 2 ] = { 0 };
	EI2C_ERROR_STATUS ErrorStatus;

	// init start address bytes
	Start_address_bytes[ 0 ] = ( ( Start_address >> 8 ) & 0x00FF );
	Start_address_bytes[ 1 ] = Start_address & 0x00FF;

	//-------------------------------------
	// first sending start address
	//-------------------------------------
	I2C_setSlaveAddress( I2C0_BASE, Slave_address, I2C_MASTER_WRITE );

	I2C_putMasterData( I2C0_BASE, Start_address_bytes[ 0 ] );

	// wait for bus not busy
	if( I2C_isBusBusy( I2C0_BASE ) == true ) {
		return I2C_STAT_WRITE_BB;
	}

	CM_I2C_StartMultiByteWrite();

	// avoid polling master busy too fast
	DEVICE_DELAY_US(20);

	// high address byte transmission
	// wait for master busy and error check
	ErrorStatus = CM_I2C_ErrorCheck();
	if( ErrorStatus == I2C_ERR_BUSY ) {
		return I2C_STAT_WRITE_MB;
	}
	else if( ErrorStatus == I2C_ERR_NO_ADDRESS_ACK ) {
		return I2C_STAT_WRITE_ADR_NACK;
	}
	else if( ErrorStatus != I2C_ERR_NONE ) {
		return I2C_STAT_WRITE_FAIL;
	}

	// low address byte transmission
	// put data
	I2C_putMasterData( I2C0_BASE, Start_address_bytes[ 1 ] );

	// high address byte transmission
	// wait for master busy and error check
	ErrorStatus = CM_I2C_ErrorCheck();
	if( ErrorStatus == I2C_ERR_BUSY ) {
		return I2C_STAT_WRITE_MB;
	}
	else if( ErrorStatus != I2C_ERR_NONE ) {
		return I2C_STAT_WRITE_FAIL;
	}

	// write xxx0x001 to I2CMCS
	I2C_setMasterConfig( I2C0_BASE, I2C_MASTER_CMD_BURST_SEND_CONT );

	//-------------------------------------
	// second transmitting data
	//-------------------------------------
	do {
		// avoid polling master busy too fast
		DEVICE_DELAY_US(20);

		// wait for master busy and error check
		ErrorStatus = CM_I2C_ErrorCheck();
		if( ErrorStatus == I2C_ERR_BUSY ) {
			return I2C_STAT_WRITE_MB;
		}
		else if( ErrorStatus != I2C_ERR_NONE ) {
			return I2C_STAT_WRITE_FAIL;
		}

		// put data
		BuffIndex = ByteIndex >> 1;
		// low byte: even index
		if( ( ByteIndex & 0x0001 ) == 0 ) {
			I2C_putMasterData( I2C0_BASE, (uint8_t)( pTransBuffer[ BuffIndex ] & 0x00FF ) );
		}
		// high byte: odd index
		else {
			I2C_putMasterData( I2C0_BASE, (uint8_t)( ( pTransBuffer[ BuffIndex ] >> 8 ) & 0x00FF ) );
		}

		// the last byte treatment
		if( ByteIndex == Trans_DataBytes - 1 ) {
			// write xxx00101 to I2CMCS
			I2C_setMasterConfig( I2C0_BASE, I2C_MASTER_CMD_BURST_SEND_FINISH );
		}
		else {
			// write xxx0x001 to I2CMCS
			I2C_setMasterConfig( I2C0_BASE, I2C_MASTER_CMD_BURST_SEND_CONT );
		}

		// index counts up
		ByteIndex++;
	} while( ByteIndex < Trans_DataBytes );

	// wait for master busy and error check
	ErrorStatus = CM_I2C_ErrorCheck();
	if( ErrorStatus == I2C_ERR_BUSY ) {
		return I2C_STAT_WRITE_MB;
	}
	else if( ErrorStatus != I2C_ERR_NONE ) {
		return I2C_STAT_WRITE_FAIL;
	}

	return I2C_STAT_WRITE_OK;
}