//----------------------------------------------------------------------------------
//	FILE:			Lighting_DCDC-Main.C
//
//	Description:	Led Backlighting closed loop control system, with ADC
//					s/w drivers, and a 2-pole/2-zero compensation/control law.
//
//	Version: 		2.2
//
//  Target:  		TMS320F2803x family
//
//	Type: 			Device Independent
//
//----------------------------------------------------------------------------------
//  Copyright Texas Instruments  2009-2010
//----------------------------------------------------------------------------------
//  Revision History:
//----------------------------------------------------------------------------------
//  Date	  | Description / Status
//----------------------------------------------------------------------------------
// 29-Jun-09 | Release 1.0  		Internal release (DAF)
// 08-Sep-09 | Release 2.0			Internal release (BRL)
// 09-Feb-10 | Release 2.1			Internal release (BRL)
// 29-Mar-10 | Release 2.2			External release (BRL)
//----------------------------------------------------------------------------------
//
// PLEASE READ - Useful notes about this Project

// Although this project is made up of several files, the most important ones are:
//	 "Lighting_DCDC-Main.C"	- this file
//		- Application Initialization, Peripheral config,
//		- Application management
//		- Slower background code loops and Task scheduling
//	 "Lighting_DCDC-DevInit_F28xxx.C
//		- Device Initialization, e.g. Clock, PLL, WD, GPIO mapping
//		- Peripheral clock enables
//		- DevInit file will differ per each F28xxx device series, e.g. F280x, F2833x,
//	 "Lighting_DCDC-ISR.asm
//		- Assembly level library Macros and any cycle critical functions are found here
//	 "ProjectSettings.h"
//		- Global defines (settings) project selections are found here
//		- This file is referenced by both C and ASM files.

// Code is made up of sections, e.g. "FUNCTION PROTOTYPES", "VARIABLE DECLARATIONS" ,..etc
//	each section has FRAMEWORK and USER areas.
//  FRAMEWORK areas provide useful ready made "infrastructure" code which for the most part
//	does not need modification, e.g. Task scheduling, ISR call, GUI interface support,...etc
//  USER areas have functional example code which can be modified by USER to fit their appl.
//
// Variables or parameters used for Multiple Channels are stored in Arrays,
//	the array index is the channel number, note: element zero (index=0) is not used
//
// Code can be compiled with various build options (Incremental Builds IBx), these
//  options are selected in file "ProjectSettings.h".  Note: "Rebuild All" compile
//  tool bar button must be used if this file is modified.
//----------------------------------------------------------------------------------

															
#include "Lighting_DCDC-Settings.h"			// supplies settings for this project
#include "PeripheralHeaderIncludes.h"	// include headers for this used device
#include "DSP2803x_EPWM_defines.h" 		// useful defines specific to EPWM

#include "SQMath.h"
#include "IQMathLib.h"

#include "ProcessorDependent.h"
#include "commros_user_regular.h"

//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
// FUNCTION PROTOTYPES
//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

// -------------------------------- FRAMEWORK --------------------------------------
void DeviceInit(void);
void ISR_Init(void);
void ISR_Run(void);
void InitFlash();
void MemCopy(Uint16**, Uint16**, Uint16**);

void InitCommros();

// State Machine function prototypes
//------------------------------------
// Alpha states
void A0(void);	//state A0
void B0(void);	//state B0
void C0(void);	//state C0

// A branch states
void A1(void);	//state A1
void A2(void);	//state A2
void A3(void);	//state A3

// B branch states
void B1(void);	//state B1
void B2(void);	//state B2
void B3(void);	//state B3

// C branch states
void C1(void);	//state C1
void C2(void);	//state C2
void C3(void);	//state C3

// Variable declarations
void (*Alpha_State_Ptr)(void);	// Base States pointer
void (*A_Task_Ptr)(void);		// State pointer A branch
void (*B_Task_Ptr)(void);		// State pointer B branch
void (*C_Task_Ptr)(void);		// State pointer C branch


// ---------------------------------- USER -----------------------------------------
void BuckSingle_CNF(int n, int prd, int mode, int phase);
void BuckDual_CNF(int n, int prd, int mode, int phase);
void ADC_CascSeqCNF(int ChSel[], int TrigSel[], int ACQPS, int Conv, int mode);


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
// VARIABLE DECLARATIONS - GENERAL
//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


// --------------------------------- COMMROS ---------------------------------------
extern struct Commros commros;

// -------------------------------- FRAMEWORK --------------------------------------

int16	VTimer0[4];			// Virtual Timers slaved of CPU Timer 0 (A events)
int16	VTimer1[4]; 		// Virtual Timers slaved of CPU Timer 1 (B events)
int16	VTimer2[4]; 		// Virtual Timers slaved of CPU Timer 2 (C events)
int16	HRmode;
int16 	BlinkStatePtr, LED_TaskPtr;

// Used to indirectly access all EPWM modules, very useful!
volatile struct EPWM_REGS *ePWM[] = 
 				  { &EPwm1Regs,			//intentional (ePWM[0] not used)
					&EPwm1Regs,				  
					&EPwm2Regs,
					&EPwm3Regs,
					&EPwm4Regs,
					&EPwm5Regs,
					&EPwm6Regs,
					#if DSP2803x_DEVICE_H
					&EPwm7Regs,
					#endif
				  };

// Used for running BackGround in flash, and ISR in RAM
extern Uint16 *RamfuncsLoadStart, *RamfuncsLoadEnd, *RamfuncsRunStart;


// ---------------------------------- USER -----------------------------------------

int16	AdcNetBus[14];  	// used as consecutive addresses for ADC_Rslts

int16	Vsepic;
int16 	VREFsepic_slewed; 
int16	UOUTsepic;
int16	Vtrgt;
int16	IREF_LED[9];
int16	UOUT_LED[9];
int16	Iset_LED[9];

long	Coef2P2Z_Sepic[7], Coef2P2Z_LED[7];

int16	FaultFlg;
int16	ClearFaultFlg;
int16	SlewError;
int16	Duty[9];

int16	DmaxSepic; 
int16	SlewStepSepic;
int16 	Dmax_LED;
int16	SlewStep_LED;
int16	LEDs_linked;

int 	ChSel[16] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
int		TrigSel[16] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};

// ASM Module Terminal pointers and variables
extern int16	*ADC_Rslt;
extern int16	*Buck_In1, *Buck_In2;
extern int16	*Buck_InA2, *Buck_InB2, *Buck_InA3, *Buck_InB3, *Buck_InA4, *Buck_InB4;
extern int16	*Buck_InA5, *Buck_InB5;

extern int16	*CNTL_2P2Z_Ref1, *CNTL_2P2Z_Out1, *CNTL_2P2Z_Fdbk1;
extern long		*CNTL_2P2Z_Coef1;

extern int16	*CNTL_2P2Z_Ref2, *CNTL_2P2Z_Ref3, *CNTL_2P2Z_Ref4, *CNTL_2P2Z_Ref5;
extern int16	*CNTL_2P2Z_Out2, *CNTL_2P2Z_Out3, *CNTL_2P2Z_Out4, *CNTL_2P2Z_Out5;
extern int16	*CNTL_2P2Z_Fdbk2, *CNTL_2P2Z_Fdbk3, *CNTL_2P2Z_Fdbk4, *CNTL_2P2Z_Fdbk5;
extern long		*CNTL_2P2Z_Coef2, *CNTL_2P2Z_Coef3, *CNTL_2P2Z_Coef4, *CNTL_2P2Z_Coef5;
extern int16	*CNTL_2P2Z_Ref6, *CNTL_2P2Z_Ref7, *CNTL_2P2Z_Ref8, *CNTL_2P2Z_Ref9;
extern int16	*CNTL_2P2Z_Out6, *CNTL_2P2Z_Out7, *CNTL_2P2Z_Out8, *CNTL_2P2Z_Out9;
extern int16	*CNTL_2P2Z_Fdbk6, *CNTL_2P2Z_Fdbk7, *CNTL_2P2Z_Fdbk8, *CNTL_2P2Z_Fdbk9;
extern long		*CNTL_2P2Z_Coef6, *CNTL_2P2Z_Coef7, *CNTL_2P2Z_Coef8, *CNTL_2P2Z_Coef9;



//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
// VARIABLE DECLARATIONS - CCS WatchWindow / GUI support
//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

_sq8 	Gui_Vin;
_sq9 	Gui_Vsepic; 
_sq14	Gui_Isepic;
_sq15	Gui_ILed[9];

// Monitor ("Get")			// Display as:
_sq9	Gui_VsetSepic;		
_sq14	Gui_IsetLed[9];		// Q15 in example, Q14 in GUI and in Watch View


// Configure ("Set") 
// Configure ("Set")
_sq0	ScopeGain;		// "counts" (Q0) 
_sq0	Pgain_Sepic;	// "counts" (Q0) 
_sq0	Igain_Sepic;	// "counts" (Q0) 
_sq0	Dgain_Sepic;	// "counts" (Q0) 

int16	Pgain_LED;
int16	Igain_LED;
int16	Dgain_LED;

//Scaling Constants (values found via spreadsheet; exact value calibrated per board)
_sq15	K_Vsepic = 	_SQ15(0.78);
_sq15	K_Isepic = 	_SQ15(0.50);
_sq15	K_ILed = 	_SQ15(0.66);
_sq15	K_Vin =		_SQ15(0.518);

_sq14	iK_Vsepic =	_SQ14(1.283);
_sq14	iK_ILed = 	_SQ14(1.515);

// Variables for background support only (no need to access)
int16	i,j;							// common use incrementers
int16	HistPtr, temp_Scratch; 			// Temp here means Temporary
int16	temp_ChNum, temp_Iout;

// History arrays are used for Running Average calculation (boxcar filter)
// Used for CCS display and GUI only, not part of control loop processing
int16	VinH[HistorySize];			
int16	VsepicH[HistorySize];			
int16	IsepicH[HistorySize];			
int16	ILedH[9][HistorySize];	
int16	VinH[HistorySize];				

//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
// MAIN CODE - starts here
//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

void main(void)
{
//=================================================================================
//	INITIALISATION - General
//=================================================================================

//-------------------------------- FRAMEWORK --------------------------------------

	DeviceInit();	// Device Life support & GPIO
	InitCommros();		

// Only used if running from FLASH
#ifdef FLASH
// Copy time critical code and Flash setup code to RAM
// The  RamfuncsLoadStart, RamfuncsLoadEnd, and RamfuncsRunStart
// symbols are created by the linker. Refer to the linker files. 
	MemCopy(&RamfuncsLoadStart, &RamfuncsLoadEnd, &RamfuncsRunStart);

// Call Flash Initialization to setup flash waitstates
// This function must reside in RAM
	InitFlash();	// Call the flash wrapper init function
#endif //(FLASH)


// Timing sync for background loops 
// Timer period definitions found in PeripheralHeaderIncludes.h
	CpuTimer0Regs.PRD.all =  mSec1;		// A tasks
	CpuTimer1Regs.PRD.all =  mSec5;		// B tasks
	CpuTimer2Regs.PRD.all =  mSec50;	// C tasks

// Tasks State-machine init
	Alpha_State_Ptr = &A0;
	A_Task_Ptr = &A1;
	B_Task_Ptr = &B1;
	C_Task_Ptr = &C1;

	BlinkStatePtr = 0;
	VTimer0[0] = 0;	
	VTimer1[0] = 0;	
	VTimer1[1] = 0;
	VTimer2[0] = 0;
	HRmode = 1;		// Default to HR mode enabled
	LED_TaskPtr = 0;

// ---------------------------------- USER -----------------------------------------

// PID coefficients & Clamping
	Pgain_Sepic = 900;		
	Igain_Sepic = 20;
	Dgain_Sepic = 10;
	Pgain_LED = 140;
	Igain_LED = 80;
	Dgain_LED = 100;
	DmaxSepic = 900;

// 2 pole / 2 Zero compensator coefficients (B2, B1, B0, A2, A1) are mapped to the
// simpler 3 coefficients P, I, D  to allow for trial & error intuitive tuning via
// CCS WatchWindow or GUI Sliders.  Note: User can modify if needed and assign full
// set of 5 coef.

// Coefficient init for SEPIC Loop
	Coef2P2Z_Sepic[0] = Dgain_Sepic * 67108;											// B2
	Coef2P2Z_Sepic[1] = (Igain_Sepic - Pgain_Sepic - Dgain_Sepic - Dgain_Sepic)*67108;	// B1
	Coef2P2Z_Sepic[2] = (Pgain_Sepic + Igain_Sepic + Dgain_Sepic)*67108;				// B0
	Coef2P2Z_Sepic[3] = 0;																// A2 = 0
	Coef2P2Z_Sepic[4] = 67108864;														// A1 = 1 in Q26 (67108864)
	Coef2P2Z_Sepic[5] = DmaxSepic * 67108;												// Clamp Hi limit (Q26)
	Coef2P2Z_Sepic[6] = 0x00000000;														// Clamp Lo

// Coefficient init for LED Loop
	Coef2P2Z_LED[0] = Dgain_LED * 67108;										// B2
	Coef2P2Z_LED[1] = (Igain_LED - Pgain_LED - Dgain_LED - Dgain_LED)*67108;	// B1
	Coef2P2Z_LED[2] = (Pgain_LED + Igain_LED + Dgain_LED)*67108;				// B0
	Coef2P2Z_LED[3] = 0;														// A2 = 0
	Coef2P2Z_LED[4] = 67108864;													// A1 = 1 in Q26 (67108864)
	Coef2P2Z_LED[5] = Dmax_LED * 67108;											// Clamp Hi limit (Q26)
	Coef2P2Z_LED[6] = 0x00000000;												// Clamp Lo

// Soft start parameter initialisation
	SlewError = 0;

// Init for Sepic control
	VREFsepic_slewed = 0;
	UOUTsepic = 0;
	
// Init for LED control
	for (i = 0; i<9; i++)
	{
		IREF_LED[i] = 0;
		UOUT_LED[i] = 0;
		Iset_LED[i] = 0;
	}
	Vtrgt = 0;

// Misc
	HistPtr = 0;
	LEDs_linked = 0;
	ClearFaultFlg = 0;

//=================================================================================
//	INITIALISATION - GUI connections
//=================================================================================
// Use this section only if you plan to "Instrument" your application using the 
// Microsoft C# freeware GUI Template provided by TI

//	//"Set" variables
//	//---------------------------------------
//	// assign GUI variable Textboxes to desired "setable" parameter addresses
//	varSetTxtList[0] = &LEDs_linked;
//	varSetTxtList[1] = &SlewStep_LED;
//
//	// assign GUI Buttons to desired flag addresses
//	//varSetBtnList[0] = &Var2;
//
//	// assign GUI Sliders to desired "setable" parameter addresses
//	varSetSldrList[0] = &Gui_VsetSepic;
//	//varSetSldrList[1] = &Gui_MaxDutySepic;
//	varSetSldrList[2] = &Gui_IsetLed[1];
//	varSetSldrList[3] = &Gui_IsetLed[2];
//	varSetSldrList[4] = &Gui_IsetLed[3];
//	varSetSldrList[5] = &Gui_IsetLed[4];
//	varSetSldrList[6] = &Gui_IsetLed[5];
//	varSetSldrList[7] = &Gui_IsetLed[6];
//	varSetSldrList[8] = &Gui_IsetLed[7];
//	varSetSldrList[9] = &Gui_IsetLed[8];
//
//	//"Get" variables
//	//---------------------------------------
//	// assign a GUI "getable" parameter address
//	varGetList[0] = &FlashID;
//	varGetList[1] = &Gui_Vin;
//	varGetList[2] = &Gui_Vsepic;
//
//	// assign a GUI "getable" parameter array address
//	arrayGetList[0] = &Gui_ILed[1];  	//only need to set initial position of array,
//										//  program will run through it accordingly

//==================================================================================
//	INCREMENTAL BUILD OPTIONS - NOTE: select via ProjectSettings.h
//==================================================================================

// ---------------------------------- USER -----------------------------------------

//----------------------------------------------------------------------
#if (INCR_BUILD == 1) 	// Open loop Sepic and Open Loop LED Dimming
//----------------------------------------------------------------------

#if (DSP2802x_DEVICE_H || DSP2803x_DEVICE_H)
#define		prd_sepic		600		// Period count = 100 KHz @ 60 MHz
#define		prd_led			3000	// Period count = 20 KHz @ 60 MHz
#endif

// "Raw" (R) ADC measurement name defines
#define		VsepicR	 	AdcResult.ADCRESULT1	//
#define		IsepicR	 	AdcResult.ADCRESULT2	//
#define		ILed2aR		AdcResult.ADCRESULT4	//
#define		ILed2bR		AdcResult.ADCRESULT5	//
#define		ILed3aR		AdcResult.ADCRESULT6	//
#define		ILed3bR		AdcResult.ADCRESULT7	//
#define		ILed4aR		AdcResult.ADCRESULT8	//
#define		ILed4bR		AdcResult.ADCRESULT9	//
#define		ILed5aR		AdcResult.ADCRESULT10	//
#define		ILed5bR		AdcResult.ADCRESULT11	//
#define		VinR	 	AdcResult.ADCRESULT12	//

// Initialize and configure peripherals
//----------------------------------------
	//ADC Configuration
	//  Define ADC sampling order
	#if (DSP2802x_DEVICE_H || DSP2803x_DEVICE_H)
	ChSel[0] = 4;		// A4 - DummyRead to resolve first sample issue (see errata)
	ChSel[1] = 4;		// A4 - VsepicR
	ChSel[2] = 2;		// A2 - IsepicR
	ChSel[3] = 8;		// B0 - DummyRead to resolve first sample issue (see errata)
	ChSel[4] = 8;		// B0 - I-Led String 0 (2a)
	ChSel[5] = 0;		// A0 - I-Led String 1 (2b)
	ChSel[6] = 9;		// B1 - I-Led String 2 (3a)
	ChSel[7] = 10;		// B2 - I-Led String 3 (3b)
	ChSel[8] = 11;		// B3 - I-Led String 4 (4a)
	ChSel[9] = 3;		// A3 - I-Led String 5 (4b)
	ChSel[10] = 12;		// B4 - I-Led String 6 (5a)
	ChSel[11] = 1;		// A1 - I-Led String 7 (5b)
	ChSel[12] = 6;		// A6 - VinR

	//  Select when ADC channel will be sampled
	TrigSel[0] = 5;   //PWM1 SOCA
	TrigSel[1] = 5;   //PWM1 SOCA
	TrigSel[2] = 5;   //PWM1 SOCA
	TrigSel[3] = 7;   //PWM2 SOCA
	TrigSel[4] = 7;   //PWM2 SOCA
	TrigSel[5] = 7;   //PWM2 SOCA
	TrigSel[6] = 7;   //PWM2 SOCA
	TrigSel[7] = 7;   //PWM2 SOCA 
	TrigSel[8] = 7;   //PWM2 SOCA
	TrigSel[9] = 7;	  //PWM2 SOCA
	TrigSel[10] = 7;  //PWM2 SOCA
	TrigSel[11] = 7;  //PWM2 SOCA
	TrigSel[12] = 7;  //PWM2 SOCA
	#endif
	
	//Configure PWM modules
	//  Spread LED power draw over full 50us (1/20kHz);
	BuckSingle_CNF(1, prd_sepic, 1, 0);		// ePWM1,    Period=prd, Master, Phase=Don't Care
	BuckDual_CNF(2, prd_led, 1, 0);			// ePWM2A,B, Period=prd, Master,  Phase=0degrees
	BuckDual_CNF(3, prd_led, 0, 750);		// ePWM3A,B, Period=prd, Slave,  Phase=90degrees
	BuckDual_CNF(4, prd_led, 0, 1500);		// ePWM4A,B, Period=prd, Slave,  Phase=180degrees
	BuckDual_CNF(5, prd_led, 0, 2250);		// ePWM5A,B, Period=prd, Slave,  Phase=270degrees

	#if (DSP2802x_DEVICE_H || DSP2803x_DEVICE_H)
	ADC_CascSeqCNF(ChSel,TrigSel,10,2,1);
	#endif

	EPwm1Regs.CMPB = 20;			// ISR trigger point
	ISR_Init();						// ASM ISR init

	Duty[0] = 0x0;
	Duty[1] = 0x0;
	Duty[2] = 0x0;
	Duty[3] = 0x0;
	Duty[4] = 0x0;
	Duty[5] = 0x0;
	Duty[6] = 0x0;
	Duty[7] = 0x0;
	Duty[8] = 0x0;


// Lib Module connection to "nets" 
//----------------------------------------
// BuckSingle_DRV connections
	Buck_In1 = &Duty[0];	// Sepic DCDC

	Buck_InA2 = &Duty[1];	// LED String 1
	Buck_InB2 = &Duty[2];	// LED String 2
	Buck_InA3 = &Duty[3];	// LED String 3
	Buck_InB3 = &Duty[4];	// LED String 4
	Buck_InA4 = &Duty[5];	// LED String 5
	Buck_InB4 = &Duty[6];	// LED String 6
	Buck_InA5 = &Duty[7];	// LED String 7
	Buck_InB5 = &Duty[8];	// LED String 8

	// ADC_NchDRV connections
	ADC_Rslt = &AdcNetBus[1];


// Configure PWMs to create Start of Conversion (SOC) pulses for the ADC
//----------------------------------------
	EPwm1Regs.ETSEL.bit.SOCASEL = ET_CTRU_CMPB;		// PWM1 SOCA on CMPB event
	EPwm2Regs.ETSEL.bit.SOCASEL = ET_CTRU_CMPA;		// PWM2 SOCA on CMPA event

	EPwm1Regs.ETSEL.bit.SOCAEN = 1;					// Enable PWM1 SOCA
	EPwm2Regs.ETSEL.bit.SOCAEN = 1;					// Enable PWM2 SOCA
	EPwm1Regs.ETPS.bit.SOCAPRD = ET_1ST;			// Trigger on every event
	EPwm2Regs.ETPS.bit.SOCAPRD = ET_1ST;			// Trigger on every event

#endif // (INCR_BUILD == 1)

//----------------------------------------------------------------------
#if (INCR_BUILD == 2) 	// Closed Loop Sepic and Open Loop LED Dimming
//----------------------------------------------------------------------
#if (DSP2802x_DEVICE_H || DSP2803x_DEVICE_H)
#define		prd_sepic	600		// Period count = 100 KHz @ 60 MHz
#define		prd_led		3000	// Period count = 20 KHz @ 60 MHz
#endif

// "Raw" (R) ADC measurement name defines
#define		VsepicR	 	AdcResult.ADCRESULT1	//
#define		IsepicR	 	AdcResult.ADCRESULT2	//
#define		ILed2aR		AdcResult.ADCRESULT4	//
#define		ILed2bR		AdcResult.ADCRESULT5	//
#define		ILed3aR		AdcResult.ADCRESULT6	//
#define		ILed3bR		AdcResult.ADCRESULT7	//
#define		ILed4aR		AdcResult.ADCRESULT8	//
#define		ILed4bR		AdcResult.ADCRESULT9	//
#define		ILed5aR		AdcResult.ADCRESULT10	//
#define		ILed5bR		AdcResult.ADCRESULT11	//
#define		VinR	 	AdcResult.ADCRESULT12	//

	SlewStepSepic = 15;
	SlewStep_LED = 5;
	DmaxSepic = 900;
	Gui_VsetSepic = 0;

	#if (DSP2802x_DEVICE_H || DSP2803x_DEVICE_H)
	ChSel[0] = 4;		// A4 - DummyRead to resolve first sample issue (see errata)
	ChSel[1] = 4;		// A4 - VsepicR
	ChSel[2] = 2;		// A2 - IsepicR
	ChSel[3] = 8;		// B0 - DummyRead to resolve first sample issue (see errata)
	ChSel[4] = 8;		// B0 - I-Led String 0 (2a)
	ChSel[5] = 0;		// A0 - I-Led String 1 (2b)
	ChSel[6] = 9;		// B1 - I-Led String 2 (3a)
	ChSel[7] = 10;		// B2 - I-Led String 3 (3b)
	ChSel[8] = 11;		// B3 - I-Led String 4 (4a)
	ChSel[9] = 3;		// A3 - I-Led String 5 (4b)
	ChSel[10] = 12;		// B4 - I-Led String 6 (5a)
	ChSel[11] = 1;		// A1 - I-Led String 7 (5b)
	ChSel[12] = 6;		// A6 - VinR

	TrigSel[0] = 5;   //PWM1 SOCA
	TrigSel[1] = 5;   //PWM1 SOCA
	TrigSel[2] = 5;   //PWM1 SOCA
	TrigSel[3] = 7;   //PWM2 SOCA
	TrigSel[4] = 7;   //PWM2 SOCA
	TrigSel[5] = 7;   //PWM2 SOCA
	TrigSel[6] = 7;   //PWM2 SOCA
	TrigSel[7] = 7;   //PWM2 SOCA 
	TrigSel[8] = 7;   //PWM2 SOCA
	TrigSel[9] = 7;	  //PWM2 SOCA
	TrigSel[10] = 7;  //PWM2 SOCA
	TrigSel[11] = 7;  //PWM2 SOCA
	TrigSel[12] = 7;  //PWM2 SOCA
	#endif

	BuckSingle_CNF(1, prd_sepic, 1, 0);		// ePWM1,    Period=prd, Master, Phase=Don't Care
	BuckDual_CNF(2, prd_led, 1, 0);			// ePWM2A,B, Period=prd, Master,  Phase=0degrees
	BuckDual_CNF(3, prd_led, 0, 750);		// ePWM3A,B, Period=prd, Slave,  Phase=90degrees
	BuckDual_CNF(4, prd_led, 0, 1500);		// ePWM4A,B, Period=prd, Slave,  Phase=180degrees
	BuckDual_CNF(5, prd_led, 0, 2250);		// ePWM5A,B, Period=prd, Slave,  Phase=270degrees

	#if (DSP2802x_DEVICE_H || DSP2803x_DEVICE_H)
	ADC_CascSeqCNF(ChSel,TrigSel,10,4,0);
	#endif

	EPwm1Regs.CMPB = 20;				// ISR trigger point
	ISR_Init();							// ASM ISR init


// ADC_NchDRV connections
	ADC_Rslt = &AdcNetBus[1];
// CNTL_2P2Z connections
	CNTL_2P2Z_Ref1 = &VREFsepic_slewed;		// point to Vref from SlewRate limiter
	CNTL_2P2Z_Out1 = &UOUTsepic;		// point to Uout
	CNTL_2P2Z_Fdbk1 = &AdcNetBus[2];	// point to Sepic output voltage value
	CNTL_2P2Z_Coef1 = &Coef2P2Z_Sepic[0];	// point to first coeff of Sepic Loop
// BuckSingle_DRV connections
	Buck_In1 = &UOUTsepic;

	Duty[1] = 0x0;
	Duty[2] = 0x0;
	Duty[3] = 0x0;
	Duty[4] = 0x0;
	Duty[5] = 0x0;
	Duty[6] = 0x0;
	Duty[7] = 0x0;
	Duty[8] = 0x0;

// Lib Module connection to "nets" 
//----------------------------------------
	Buck_InA2 = &Duty[1];	// LED String 1
	Buck_InB2 = &Duty[2];	// LED String 2
	Buck_InA3 = &Duty[3];	// LED String 3
	Buck_InB3 = &Duty[4];	// LED String 4
	Buck_InA4 = &Duty[5];	// LED String 5
	Buck_InB4 = &Duty[6];	// LED String 6
	Buck_InA5 = &Duty[7];	// LED String 7
	Buck_InB5 = &Duty[8];	// LED String 8


// Configure PWMs to create Start of Conversion (SOC) pulses for the ADC
//----------------------------------------
	EPwm1Regs.ETSEL.bit.SOCASEL = ET_CTRU_CMPB;		// PWM1 SOCA on CMPB event
	EPwm2Regs.ETSEL.bit.SOCASEL = ET_CTRU_CMPA;		// PWM2 SOCA on CMPA event

	EPwm1Regs.ETSEL.bit.SOCAEN = 1;					// Enable PWM1 SOCA
	EPwm2Regs.ETSEL.bit.SOCAEN = 1;					// Enable PWM2 SOCA
	EPwm1Regs.ETPS.bit.SOCAPRD = ET_1ST;			// Trigger on every event
	EPwm2Regs.ETPS.bit.SOCAPRD = ET_1ST;			// Trigger on every event

#endif // (INCR_BUILD == 2)


//----------------------------------------------------------------------
#if (INCR_BUILD == 3) 	// Closed Loop SEPIC + Closed Loop LED Strings
//----------------------------------------------------------------------
#if (DSP2802x_DEVICE_H || DSP2803x_DEVICE_H)
#define		prd_sepic		600		// Period count = 100 KHz @ 60 MHz
#define		prd_led		 	3000	// Period count = 20 KHz @ 60 MHz
#endif

// "Raw" (R) ADC measurement name defines
#define		VsepicR	 	AdcResult.ADCRESULT1	//
#define		IsepicR	 	AdcResult.ADCRESULT2	//
#define		ILed2aR		AdcResult.ADCRESULT4	//
#define		ILed2bR		AdcResult.ADCRESULT5	//
#define		ILed3aR		AdcResult.ADCRESULT6	//
#define		ILed3bR		AdcResult.ADCRESULT7	//
#define		ILed4aR		AdcResult.ADCRESULT8	//
#define		ILed4bR		AdcResult.ADCRESULT9	//
#define		ILed5aR		AdcResult.ADCRESULT10	//
#define		ILed5bR		AdcResult.ADCRESULT11	//
#define		VinR	 	AdcResult.ADCRESULT12	//


	SlewStepSepic = 15;
	SlewStep_LED = 5;
	DmaxSepic = 900;	//90% duty cycle
	Dmax_LED = 900;		//90% duty cycle
	Gui_VsetSepic = 0;

	for(i = 0; i<9; i++) 
	{
		Iset_LED[i] = 0;
		Gui_IsetLed[i] = 0;
	}

	#if (DSP2802x_DEVICE_H || DSP2803x_DEVICE_H)
	ChSel[0] = 4;		// A4 - DummyRead to resolve first sample issue (see errata)
	ChSel[1] = 4;		// A4 - VsepicR
	ChSel[2] = 2;		// A2 - IsepicR
	ChSel[3] = 8;		// B0 - DummyRead to resolve first sample issue (see errata)
	ChSel[4] = 8;		// B0 - I-Led String 0 (2a)
	ChSel[5] = 0;		// A0 - I-Led String 1 (2b)
	ChSel[6] = 9;		// B1 - I-Led String 2 (3a)
	ChSel[7] = 10;		// B2 - I-Led String 3 (3b)
	ChSel[8] = 11;		// B3 - I-Led String 4 (4a)
	ChSel[9] = 3;		// A3 - I-Led String 5 (4b)
	ChSel[10] = 12;		// B4 - I-Led String 6 (5a)
	ChSel[11] = 1;		// A1 - I-Led String 7 (5b)
	ChSel[12] = 6;		// A6 - VinR

	TrigSel[0] = 5;   //PWM1 SOCA
	TrigSel[1] = 5;   //PWM1 SOCA
	TrigSel[2] = 5;   //PWM1 SOCA
	TrigSel[3] = 7;   //PWM2 SOCA
	TrigSel[4] = 7;   //PWM2 SOCA
	TrigSel[5] = 7;   //PWM2 SOCA
	TrigSel[6] = 7;   //PWM2 SOCA
	TrigSel[7] = 7;   //PWM2 SOCA 
	TrigSel[8] = 7;   //PWM2 SOCA
	TrigSel[9] = 7;	  //PWM2 SOCA
	TrigSel[10] = 7;  //PWM2 SOCA
	TrigSel[11] = 7;  //PWM2 SOCA
	TrigSel[12] = 7;  //PWM2 SOCA
	#endif

	BuckSingle_CNF(1, prd_sepic, 1, 0);		// ePWM1A    Period=prd, Master, Phase=Don't Care

	// Spread LED power draw over full 50us (1/20kHz);
	BuckDual_CNF(2, prd_led, 1, 0);			// ePWM2A,B, Period=prd, Master,  Phase =  0degrees
	BuckDual_CNF(3, prd_led, 0, 750);		// ePWM3A,B, Period=prd, Slave,  Phase =  90degrees
	BuckDual_CNF(4, prd_led, 0, 1500);		// ePWM4A,B, Period=prd, Slave,  Phase = 180degrees
	BuckDual_CNF(5, prd_led, 0, 2250);		// ePWM5A,B, Period=prd, Slave,  Phase = 270degrees


	#if (DSP2802x_DEVICE_H || DSP2803x_DEVICE_H)
	ADC_CascSeqCNF(ChSel,TrigSel,10,4,0);
	AdcRegs.SOCPRICTL.bit.SOCPRIORITY = 0x03;  //SOCs 0-2 are high priority
	#endif

	EPwm1Regs.CMPB = 20;				// ISR trigger point
	ISR_Init();							// ASM ISR init


// Lib Module connection to "nets" 
//----------------------------------------
// *SEPIC
// ADC_NchDRV connections
	ADC_Rslt = &AdcNetBus[1];
// CNTL_2P2Z connections
	CNTL_2P2Z_Ref1 = &VREFsepic_slewed;		// point to Vref
	CNTL_2P2Z_Out1 = &UOUTsepic;		// point to Uout
	CNTL_2P2Z_Fdbk1 = &AdcNetBus[2];	// point to Sepic output voltage value
	CNTL_2P2Z_Coef1 = &Coef2P2Z_Sepic[0];	// point to first coeff of Sepic Loop


// *LED strings
// CNTL_2P2Z connections
	//LED String 1 - LED2A
	CNTL_2P2Z_Ref2 = &IREF_LED[1];		// point to Iref from SlewRate limiter
	CNTL_2P2Z_Out2 = &UOUT_LED[1];		// point to Uout
	CNTL_2P2Z_Fdbk2 = &AdcNetBus[5];	// point to Sepic output voltage value
	CNTL_2P2Z_Coef2 = &Coef2P2Z_LED[0];	// point to first coeff of Sepic Loop

	//LED String 2 - LED2B
	CNTL_2P2Z_Ref3 = &IREF_LED[2];		// point to Iref from SlewRate limiter
	CNTL_2P2Z_Out3 = &UOUT_LED[2];		// point to Uout
	CNTL_2P2Z_Fdbk3 = &AdcNetBus[6];	// point to Sepic output voltage value
	CNTL_2P2Z_Coef3 = &Coef2P2Z_LED[0];	// point to first coeff of Sepic Loop

	//LED String 3 - LED3A
	CNTL_2P2Z_Ref4 = &IREF_LED[3];		// point to Iref from SlewRate limiter
	CNTL_2P2Z_Out4 = &UOUT_LED[3];		// point to Uout
	CNTL_2P2Z_Fdbk4 = &AdcNetBus[7];	// point to Sepic output voltage value
	CNTL_2P2Z_Coef4 = &Coef2P2Z_LED[0];	// point to first coeff of Sepic Loop

	//LED String 4 - LED3B
	CNTL_2P2Z_Ref5 = &IREF_LED[4];		// point to Iref from SlewRate limiter
	CNTL_2P2Z_Out5 = &UOUT_LED[4];		// point to Uout
	CNTL_2P2Z_Fdbk5 = &AdcNetBus[8];	// point to Sepic output voltage value
	CNTL_2P2Z_Coef5 = &Coef2P2Z_LED[0];	// point to first coeff of Sepic Loop

	//LED String 5 - LED4A
	CNTL_2P2Z_Ref6 = &IREF_LED[5];		// point to Iref from SlewRate limiter
	CNTL_2P2Z_Out6 = &UOUT_LED[5];		// point to Uout
	CNTL_2P2Z_Fdbk6 = &AdcNetBus[9];	// point to Sepic output voltage value
	CNTL_2P2Z_Coef6 = &Coef2P2Z_LED[0];	// point to first coeff of Sepic Loop

	//LED String 6 - LED4B
	CNTL_2P2Z_Ref7 = &IREF_LED[6];		// point to Iref from SlewRate limiter
	CNTL_2P2Z_Out7 = &UOUT_LED[6];		// point to Uout
	CNTL_2P2Z_Fdbk7 = &AdcNetBus[10];	// point to Sepic output voltage value
	CNTL_2P2Z_Coef7 = &Coef2P2Z_LED[0];	// point to first coeff of Sepic Loop

	//LED String 7 - LED5A
	CNTL_2P2Z_Ref8 = &IREF_LED[7];		// point to Iref from SlewRate limiter
	CNTL_2P2Z_Out8 = &UOUT_LED[7];		// point to Uout
	CNTL_2P2Z_Fdbk8 = &AdcNetBus[11];	// point to Sepic output voltage value
	CNTL_2P2Z_Coef8 = &Coef2P2Z_LED[0];	// point to first coeff of Sepic Loop

	//LED String 8 - LED5B
	CNTL_2P2Z_Ref9 = &IREF_LED[8];		// point to Iref from SlewRate limiter
	CNTL_2P2Z_Out9 = &UOUT_LED[8];		// point to Uout
	CNTL_2P2Z_Fdbk9 = &AdcNetBus[12];	// point to Sepic output voltage value
	CNTL_2P2Z_Coef9 = &Coef2P2Z_LED[0];	// point to first coeff of Sepic Loop


	// BuckSingle_DRV connections
	Buck_In1 = &UOUTsepic;

	Buck_InA2 = &UOUT_LED[1];	// LED String 1
	Buck_InB2 = &UOUT_LED[2];	// LED String 2
	Buck_InA3 = &UOUT_LED[3];	// LED String 3
	Buck_InB3 = &UOUT_LED[4];	// LED String 4
	Buck_InA4 = &UOUT_LED[5];	// LED String 5
	Buck_InB4 = &UOUT_LED[6];	// LED String 6
	Buck_InA5 = &UOUT_LED[7];	// LED String 7
	Buck_InB5 = &UOUT_LED[8];	// LED String 8


// Configure PWMs to create Start of Conversion (SOC) pulses for the ADC
//----------------------------------------
	EPwm1Regs.ETSEL.bit.SOCASEL = ET_CTRU_CMPB;		// PWM1 SOCA on CMPB event
	EPwm2Regs.ETSEL.bit.SOCASEL = ET_CTRU_CMPA;		// PWM2 SOCA on CMPA event

	EPwm1Regs.ETSEL.bit.SOCAEN = 1;					// Enable PWM1 SOCA
	EPwm2Regs.ETSEL.bit.SOCAEN = 1;					// Enable PWM2 SOCA
	EPwm1Regs.ETPS.bit.SOCAPRD = ET_1ST;			// Trigger on every event
	EPwm2Regs.ETPS.bit.SOCAPRD = ET_1ST;			// Trigger on every event
#endif // (INCR_BUILD == 3)



//=================================================================================
//	CONFIGURE COMPARATORS/TRIP ZONE CIRCUITRY FOR SYSTEM PROTECTION
//=================================================================================
	EALLOW;
	Comp1Regs.COMPCTL.bit.SYNCSEL = 0;		// asynchronous
	Comp1Regs.COMPCTL.bit.CMPINV = 1;		// invert output
	Comp1Regs.COMPCTL.bit.COMPSOURCE = 0;	// use internal DAC
	Comp1Regs.COMPCTL.bit.COMPDACEN = 1;	// enable comparator
	Comp1Regs.DACVAL.bit.DACVAL = 764;  	//number is in Q5    (764=2.463V =>3A)

	Comp2Regs.COMPCTL.bit.SYNCSEL = 0;		// asynchronous
	Comp2Regs.COMPCTL.bit.CMPINV = 1;		// invert output
	Comp2Regs.COMPCTL.bit.COMPSOURCE = 0;	// use internal DAC
	Comp2Regs.COMPCTL.bit.COMPDACEN = 1;	// enable comparator
	Comp2Regs.DACVAL.bit.DACVAL = 986;  	// number is in Q5    (986=3.18V =>48V)

	EPwm1Regs.DCTRIPSEL.bit.DCAHCOMPSEL = DC_COMP1OUT;   // DCAH = Comparator 1 output
    EPwm1Regs.DCTRIPSEL.bit.DCBHCOMPSEL = DC_COMP2OUT;        // DCAL = TZ2
    EPwm1Regs.TZDCSEL.bit.DCAEVT1 = TZ_DCAH_LOW; // DCAEVT1 =  DCAH low(will become active as Comparator1 output goes low)
    EPwm1Regs.TZDCSEL.bit.DCBEVT1 = TZ_DCBH_LOW; // DCAEVT1 =  DCAH low(will become active as Comparator2 output goes low) 
    EPwm1Regs.DCACTL.bit.EVT1SRCSEL = DC_EVT1;           // DCAEVT1 = DCAEVT1 (not filtered)
	EPwm1Regs.DCBCTL.bit.EVT1SRCSEL = DC_EVT1;           // DCAEVT1 = DCAEVT1 (not filtered)
    EPwm1Regs.DCACTL.bit.EVT1FRCSYNCSEL = DC_EVT_ASYNC;  // Take async path
	EPwm1Regs.DCBCTL.bit.EVT1FRCSYNCSEL = DC_EVT_ASYNC;  // Take async path
         
    EPwm1Regs.TZSEL.bit.DCAEVT1 = 1;
    EPwm1Regs.TZSEL.bit.DCBEVT1 = 1;
	
    // What do we want the DCAEVT1 and DCBEVT1 events to do? - DCAEVTx events can force EPWMxA  
    EPwm1Regs.TZCTL.bit.TZA = TZ_FORCE_LO;           // EPWM1A will go low
	EDIS;


//=================================================================================
//	INTERRUPT & ISR INITIALISATION (best to run this section after other initialisation)
//=================================================================================
	EALLOW;
	PieVectTable.EPWM1_INT = &ISR_Run;			// Map Interrupt
	EDIS;
	PieCtrlRegs.PIEIER3.bit.INTx1 = 1;			// PIE level enable, Grp3 / Int1
	EPwm1Regs.ETSEL.bit.INTSEL = ET_CTR_ZERO;	// INT on Zero event
	EPwm1Regs.ETSEL.bit.INTEN = 1;				// Enable INT
	EPwm1Regs.ETPS.bit.INTPRD = ET_1ST;			// Generate INT on every event

// Enable Peripheral, global Ints and higher priority real-time debug events:
	IER |= M_INT3;	// Enable CPU INT3 connected to EPWM1-6 INTs:
	EINT;   		// Enable Global interrupt INTM
	ERTM;   		// Enable Global realtime interrupt DBGM

//=================================================================================
//	BACKGROUND (BG) LOOP
//=================================================================================

//--------------------------------- FRAMEWORK -------------------------------------
	for(;;)
	{
		// State machine entry & exit point
		//===========================================================
		(*Alpha_State_Ptr)();	// jump to an Alpha state (A0,B0,...)
		//===========================================================
		ServiceRoutine(&commros);

	}
} //END MAIN CODE


//=================================================================================
//	STATE-MACHINE SEQUENCING AND SYNCRONIZATION
//=================================================================================

//--------------------------------- FRAMEWORK -------------------------------------

void A0(void)
{
	// loop rate synchronizer for A-tasks
	if(CpuTimer0Regs.TCR.bit.TIF == 1)
	{
		CpuTimer0Regs.TCR.bit.TIF = 1;	// clear flag
		//-----------------------------------------------------------
		(*A_Task_Ptr)();		// jump to an A Task (A1,A2,A3,...)
		//-----------------------------------------------------------
		VTimer0[0]++;			// virtual timer 0, instance 0 (spare		
	}
	Alpha_State_Ptr = &B0;		// Comment out to allow only A tasks
}

void B0(void)
{
	// loop rate synchronizer for B-tasks
	if(CpuTimer1Regs.TCR.bit.TIF == 1)
	{
		CpuTimer1Regs.TCR.bit.TIF = 1;				// clear flag
		//-----------------------------------------------------------
		(*B_Task_Ptr)();		// jump to a B Task (B1,B2,B3,...)
		//-----------------------------------------------------------
		VTimer1[0]++;			// virtual timer 1, instance 0 (used to control SPI LEDs
	}
	Alpha_State_Ptr = &C0;	
}

void C0(void)
{
	// loop rate synchronizer for C-tasks
	if(CpuTimer2Regs.TCR.bit.TIF == 1)
	{
		CpuTimer2Regs.TCR.bit.TIF = 1;				// clear flag
		//-----------------------------------------------------------
		(*C_Task_Ptr)();		// jump to a C Task (C1,C2,C3,...)
		//-----------------------------------------------------------
		VTimer2[0]++;			// virtual timer 2, instance 0 (spare)	
	}
	Alpha_State_Ptr = &A0;		// Back to State A0
}


//=================================================================================
//	A - TASKS
//=================================================================================
//--------------------------------------------------------
void A1(void) // Slewrates (soft-start) of references (Sepic & LED string current 1-4)
//--------------------------------------------------------
// Task runs every 2ms (CpuTimer0 period [1ms] * 2 "A" tasks active)
// Slows the changing of a controller's reference by SlewStep_Sepic or SlewStep_LED
{
	//  Dmax Clamping
	Coef2P2Z_Sepic[5] = DmaxSepic * 67108;
	Coef2P2Z_LED[5] = Dmax_LED * 67108;


	//---------------------------------------------------------
	SlewError = (VREFsepic_slewed - Vtrgt);
	if (SlewError > SlewStepSepic)		VREFsepic_slewed = VREFsepic_slewed - SlewStepSepic;
	else
	if (SlewError < (-SlewStepSepic) )	VREFsepic_slewed = VREFsepic_slewed + SlewStepSepic;
	else								VREFsepic_slewed = Vtrgt;
	//---------------------------------------------------------


	//---------------------------------------------------------
	SlewError = (IREF_LED[1] - Iset_LED[1]);
	if (SlewError > SlewStep_LED)			IREF_LED[1] = IREF_LED[1] - SlewStep_LED;
	else if (SlewError < (-SlewStep_LED) )	IREF_LED[1] = IREF_LED[1] + SlewStep_LED;
	else									IREF_LED[1] = Iset_LED[1];
	//---------------------------------------------------------
	SlewError = (IREF_LED[2] - Iset_LED[2]);
	if (SlewError > SlewStep_LED)			IREF_LED[2] = IREF_LED[2] - SlewStep_LED;
	else if (SlewError < (-SlewStep_LED) )	IREF_LED[2] = IREF_LED[2] + SlewStep_LED;
	else									IREF_LED[2] = Iset_LED[2];
	//---------------------------------------------------------
	SlewError = (IREF_LED[3] - Iset_LED[3]);
	if (SlewError > SlewStep_LED)			IREF_LED[3] = IREF_LED[3] - SlewStep_LED;
	else if (SlewError < (-SlewStep_LED) )	IREF_LED[3] = IREF_LED[3] + SlewStep_LED;
	else									IREF_LED[3] = Iset_LED[3];
	//---------------------------------------------------------
	SlewError = (IREF_LED[4] - Iset_LED[4]);
	if (SlewError > SlewStep_LED)			IREF_LED[4] = IREF_LED[4] - SlewStep_LED;
	else if (SlewError < (-SlewStep_LED) )	IREF_LED[4] = IREF_LED[4] + SlewStep_LED;
	else									IREF_LED[4] = Iset_LED[4];
	//---------------------------------------------------------

	//-------------------
	A_Task_Ptr = &A2;
	//-------------------
}

//-----------------------------------------------------------------
void A2(void) // Slewrates (soft-start) of references (LED string current 5-8); UART comms
//-----------------------------------------------------------------
// Task runs every 2ms (CpuTimer0 period [1ms] * 2 "A" tasks active) 
// Slows the changing of a controller's reference by SlewStep_LED
// SerialHostComms is a function that provides communication with a GUI host if one is connected
{

	if(ClearFaultFlg == 1)
	{
		UOUTsepic = 0;
		VREFsepic_slewed = 0;
		
		for (i=0; i<9; i++)
		{
			IREF_LED[i] = 0;
		}
	
		EALLOW;
		EPwm1Regs.TZCLR.bit.OST = 1;
		EPwm2Regs.TZCLR.bit.OST = 1;
		EPwm3Regs.TZCLR.bit.OST = 1;
		EPwm4Regs.TZCLR.bit.OST = 1;
		EPwm5Regs.TZCLR.bit.OST = 1;
		EPwm6Regs.TZCLR.bit.OST = 1;
		EDIS;
	
		ClearFaultFlg = 0;
	}

	//---------------------------------------------------------
	SlewError = (IREF_LED[5] - Iset_LED[5]);
	if (SlewError > SlewStep_LED)			IREF_LED[5] = IREF_LED[5] - SlewStep_LED;
	else if (SlewError < (-SlewStep_LED) )	IREF_LED[5] = IREF_LED[5] + SlewStep_LED;
	else									IREF_LED[5] = Iset_LED[5];
	//---------------------------------------------------------
	SlewError = (IREF_LED[6] - Iset_LED[6]);
	if (SlewError > SlewStep_LED)			IREF_LED[6] = IREF_LED[6] - SlewStep_LED;
	else if (SlewError < (-SlewStep_LED) )	IREF_LED[6] = IREF_LED[6] + SlewStep_LED;
	else									IREF_LED[6] = Iset_LED[6];
	//---------------------------------------------------------
	SlewError = (IREF_LED[7] - Iset_LED[7]);
	if (SlewError > SlewStep_LED)			IREF_LED[7] = IREF_LED[7] - SlewStep_LED;
	else if (SlewError < (-SlewStep_LED) )	IREF_LED[7] = IREF_LED[7] + SlewStep_LED;
	else									IREF_LED[7] = Iset_LED[7];
	//---------------------------------------------------------
	SlewError = (IREF_LED[8] - Iset_LED[8]);
	if (SlewError > SlewStep_LED)			IREF_LED[8] = IREF_LED[8] - SlewStep_LED;
	else if (SlewError < (-SlewStep_LED) )	IREF_LED[8] = IREF_LED[8] + SlewStep_LED;
	else									IREF_LED[8] = Iset_LED[8];
	//---------------------------------------------------------


	//SerialHostComms();	//found in SciCommsGui.c

	//-------------------
	A_Task_Ptr = &A1;	// To make task A3 active, change &A1 to &A3
	//-------------------
}

//-----------------------------------------
void A3(void) // SPARE (not active)
//-----------------------------------------
{

	//-----------------
	A_Task_Ptr = &A1;	// To make task A4 active, change &A1 to &A4
	//-----------------
}


//=================================================================================
//	B - TASKS
//=================================================================================

//----------------------------------- USER ----------------------------------------
//----------------------------------------------------
void B1(void) // SEPIC & Vin Dashboard measurements
//----------------------------------------------------
// Task runs every 10ms (CpuTimer1 period [5ms] * 2 "B" tasks active)
// ADC samples are averaged over "HistPtr" samples and then scaled to real-world units
//   see LedCalculations.xls for how scaling was derived. 
{
// Measurement code expandable to multiple channels, defined by "NumChannels"
// Voltage measurement calculated by:
//	Gui_Vin = VinAvg * K_Vin, where VinAvg = sum of 8 VinR samples
//	Gui_Vout = VoutAvg * K_Vout, where VoutAvg = sum of 8 VoutR samples
//
	HistPtr++;
	if (HistPtr >= HistorySize)	HistPtr = 0;

	// BoxCar Averages - Input Raw samples into BoxCar arrays
	//----------------------------------------------------------------

	VsepicH[HistPtr]   = VsepicR;
	IsepicH[HistPtr]   = IsepicR;


	VinH[HistPtr]   = VinR;

	// Sepic Measurements
	//----------------------------------------------------------------
	// view following variables in Watch Window as:
	//		Gui_Vin = Q8
	//		Gui_Vsepic = Q9
	//		Gui_Isepic = Q14

	temp_Scratch=0;
	for(i=0; i<8; i++)	temp_Scratch = temp_Scratch + VsepicH[i];
	Gui_Vsepic = ( (long) temp_Scratch * (long) K_Vsepic ) >> 15;

	temp_Scratch=0;
	for(i=0; i<8; i++)	temp_Scratch = temp_Scratch + IsepicH[i];
	Gui_Isepic = ( (long) temp_Scratch * (long) K_Isepic ) >> 15;

	temp_Scratch=0;
	for(i=0; i<8; i++)	temp_Scratch = temp_Scratch + VinH[i];
	Gui_Vin = ( (long) temp_Scratch * (long) K_Vin ) >> 15;



	Vtrgt = ( (long) Gui_VsetSepic * (long) iK_Vsepic ) >> 14;


	//-----------------
	B_Task_Ptr = &B2;	
	//-----------------
}

//------------------------------------------------------------
void B2(void) // LED Dashboard measurements
//------------------------------------------------------------
// Task runs every 10ms (CpuTimer1 period [5ms] * 2 "B" tasks active)
// ADC samples are averaged over "HistPtr" samples and then scaled to real-world units
//   see LedCalculations.xls for how scaling was derived.
{

	// LED String Measurements
	//----------------------------------------------------------------
	// view following variables in Watch Window as:
	//		Gui_ILed[j] = Q15

	ILedH[1][HistPtr]   = ILed2aR;
	ILedH[2][HistPtr]   = ILed2bR;
	ILedH[3][HistPtr]	= ILed3aR;
	ILedH[4][HistPtr]	= ILed3bR;
	ILedH[5][HistPtr]   = ILed4aR;
	ILedH[6][HistPtr]   = ILed4bR;
	ILedH[7][HistPtr]	= ILed5aR;
	ILedH[8][HistPtr]	= ILed5bR;

	for(j = 0; j<9; j++)
	{
		temp_Scratch=0;
		for(i=0; i<HistorySize; i++)	temp_Scratch = temp_Scratch + ILedH[j][i];
		Gui_ILed[j] = ( (long) temp_Scratch * (long) K_ILed ) >> 15;

		Iset_LED[j] = ( (long) Gui_IsetLed[j] * (long) iK_ILed ) >> 14;
	}


	//-----------------
	B_Task_Ptr = &B1;	
	//-----------------
}

//------------------------------------------------------------
void B3(void) //  SPARE (not active)
//------------------------------------------------------------
{

	//-----------------
	B_Task_Ptr = &B1;
	//-----------------
}


//=================================================================================
//	C - TASKS
//=================================================================================

//----------------------------- USER ----------------------------------------------

//--------------------------------------------------------------
void C1(void) // Coefficient update / control
//--------------------------------------------------------------
// Task runs every 100ms (CpuTimer2 period [50ms] * 2 "C" tasks active) 
// Allows user to dynamically change P, I, and D gains while the system is running.
{
// Coefficient Update for 1st (Ch1) Loop
	Coef2P2Z_Sepic[0] = Dgain_Sepic * 67108;							// B2
	Coef2P2Z_Sepic[1] = (Igain_Sepic - Pgain_Sepic - Dgain_Sepic - Dgain_Sepic)*67108;	// B1
	Coef2P2Z_Sepic[2] = (Pgain_Sepic + Igain_Sepic + Dgain_Sepic)*67108;			// B0

	Coef2P2Z_LED[0] = Dgain_LED * 67108;							// B2
	Coef2P2Z_LED[1] = (Igain_LED - Pgain_LED - Dgain_LED - Dgain_LED)*67108;	// B1
	Coef2P2Z_LED[2] = (Pgain_LED + Igain_LED + Dgain_LED)*67108;			// B0
	 

	if (HRmode ==1)			EPwm1Regs.HRCNFG.bit.EDGMODE = HR_FEP;	// Enable
	else					EPwm1Regs.HRCNFG.bit.EDGMODE = HR_DISABLE;
	EDIS;

	//-----------------
	C_Task_Ptr = &C2;	
	//-----------------
}

//----------------------------------------
void C2(void) // Merging of LED controls & LED "life"-pulse
//----------------------------------------
// Task runs every 100ms (CpuTimer2 period [50ms] * 2 "C" tasks active) 
// Blinks an LED and allows all LED controls to be controlled individually or
//   as a group
{
	// toggle an LED on the controlCARD
	GpioDataRegs.GPBTOGGLE.bit.GPIO34 = 1;	// 1/(2*100ms) = 500Hz blink rate

	if (LEDs_linked == 1)
	{
		for (i=1; i<9; i++)
		{
			Gui_IsetLed[i] = Gui_IsetLed[1];
		}
	}
	
	if ( (*ePWM[1]).TZFLG.bit.OST == 1 )
	{	
		FaultFlg = 1;
	}
	else
	{
		FaultFlg = 0;
	}

	//-----------------
	C_Task_Ptr = &C1;	
	//-----------------
}

//--------------------------------------
void C3(void) //  SPARE (not active)
//--------------------------------------
{
	
	//-----------------
	C_Task_Ptr = &C1;	
	//-----------------
}

