/*
 * Copyright (c) 2020 Texas Instruments Incorporated - http://www.ti.com
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "board.h"

//*****************************************************************************
//
// Board Configurations
// Initializes the rest of the modules.
// Call this function in your application if you wish to do all module
// initialization.
// If you wish to not use some of the initializations, instead of the
// Board_init use the individual Module_inits
//
//*****************************************************************************
void Board_init()
{
	PinMux_init();
	PERCONFIG_init();
	GPIO_init();
	MCAN_SYSCFG_init();
	INTERRUPT_init();
}

//*****************************************************************************
//
// PINMUX Configurations
//
//*****************************************************************************
void PinMux_init()
{
	//
	// PinMux for modules assigned to CPU1
	//

	//
	// MCANA -> myMCAN0 Pinmux
	//
	GPIO_setPinConfig(myMCAN0_MCANRX_PIN_CONFIG);
	GPIO_setPadConfig(myMCAN0_MCANRX_GPIO, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(myMCAN0_MCANRX_GPIO, GPIO_QUAL_ASYNC);

	GPIO_setPinConfig(myMCAN0_MCANTX_PIN_CONFIG);
	GPIO_setPadConfig(myMCAN0_MCANTX_GPIO, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(myMCAN0_MCANTX_GPIO, GPIO_QUAL_ASYNC);


}

//*****************************************************************************
//
// Peripheral Configurations
//
//*****************************************************************************
void PERCONFIG_init()
{
	//
	// Configuration for modules assigned to CPU1
	//
    SysCtl_selectCPUForPeripheral(SYSCTL_PERIPH_CONFIG_MCANA, SYSCTL_CPUSEL_CPU1);
    SysCtl_selectFrameForPeripheral(SYSCTL_PERIPH_CONFIG_MCANA, SYSCTL_FRAMESEL_0);

	//
	// Inserting some delay for frame updates to take effect
	//
    SysCtl_delay(100);
}


//*****************************************************************************
//
// RTDMA1 MPU Configurations
//
//*****************************************************************************
void RTDMA1_MPU_init(){
    DMA_MPUConfigParams configParams;
}

//*****************************************************************************
//
// RTDMA2 MPU Configurations
//
//*****************************************************************************
void RTDMA2_MPU_init(){
    DMA_MPUConfigParams configParams;
}


//*****************************************************************************
//
// GPIO Configurations
//
//*****************************************************************************
void GPIO_init(){
}


//*****************************************************************************
//
// INTERRUPT Configurations
//
//*****************************************************************************
void INTERRUPT_init(){
	Interrupt_setThreshold(0);
    Interrupt_setGroupMask(0xFF);
    Interrupt_setRTINTSPWarning(15, 0);

	// Interrupt Settings for myMCAN0_INT1 -> INT_MCANA_1
	Interrupt_register(myMCAN0_INT1, &MCANIntr1ISR);
	Interrupt_enable(myMCAN0_INT1);
	Interrupt_setPriority(myMCAN0_INT1, 255);
	Interrupt_setContextID(myMCAN0_INT1, INTERRUPT_CONTEXTID_0);

	// Interrupt Settings for myMCAN0_INT0 -> INT_MCANA_0
	Interrupt_register(myMCAN0_INT0, &MCANIntr0ISR);
	Interrupt_enable(myMCAN0_INT0);
	Interrupt_setPriority(myMCAN0_INT0, 255);
	Interrupt_setContextID(myMCAN0_INT0, INTERRUPT_CONTEXTID_0);
}
//*****************************************************************************
//
// MCAN Configurations
//
//*****************************************************************************
void MCAN_SYSCFG_init(){
    myMCAN0_init();
}

void myMCAN0_init(){
    MCAN_RevisionId revid_myMCAN0;
    MCAN_InitParams initParams_myMCAN0;
    MCAN_ConfigParams configParams_myMCAN0;
    MCAN_StdMsgIDFilterElement stdFiltelem_myMCAN0;
    MCAN_MsgRAMConfigParams    msgRAMConfigParams_myMCAN0;
    MCAN_BitTimingParams       bitTimes_myMCAN0;
    //
    // Initialize MCAN Init parameters.
    //
    initParams_myMCAN0.fdMode            = true;
    initParams_myMCAN0.brsEnable         = true;
    initParams_myMCAN0.txpEnable         = false;
    initParams_myMCAN0.efbi              = false;
    initParams_myMCAN0.pxhddisable       = false;
    initParams_myMCAN0.darEnable         = false;
    initParams_myMCAN0.wkupReqEnable     = false;
    initParams_myMCAN0.autoWkupEnable    = false;
    initParams_myMCAN0.emulationEnable   = false;
    initParams_myMCAN0.tdcEnable         = false;
    initParams_myMCAN0.wdcPreload        = 255;
    //
    // Transmitter Delay Compensation parameters.
    //
    initParams_myMCAN0.tdcConfig.tdcf    = 10;
    initParams_myMCAN0.tdcConfig.tdco    = 6;
    //
    // Initialize MCAN Config parameters.
    //
    configParams_myMCAN0.monEnable         = false;
    configParams_myMCAN0.asmEnable         = false;
    configParams_myMCAN0.tsPrescalar       = 15;
    configParams_myMCAN0.tsSelect          = 0;
    configParams_myMCAN0.timeoutSelect     = MCAN_TIMEOUT_SELECT_CONT;
    configParams_myMCAN0.timeoutPreload    = 65535;
    configParams_myMCAN0.timeoutCntEnable  = false;
    configParams_myMCAN0.filterConfig.rrfs = false;
    configParams_myMCAN0.filterConfig.rrfe = false;
    configParams_myMCAN0.filterConfig.anfe = 0;
    configParams_myMCAN0.filterConfig.anfs = 0;
    //
    // Initialize Message RAM Sections Configuration Parameters.
    //
    msgRAMConfigParams_myMCAN0.flssa                = myMCAN0_MCAN_STD_ID_FILT_START_ADDR;
    //
    // Standard ID Filter List Start Address.
    //
    msgRAMConfigParams_myMCAN0.lss                  = myMCAN0_MCAN_STD_ID_FILTER_NUM;
    //
    // List Size: Standard ID.
    //
    msgRAMConfigParams_myMCAN0.flesa                = myMCAN0_MCAN_EXT_ID_FILT_START_ADDR;
    //
    // Extended ID Filter List Start Address.
    //
    msgRAMConfigParams_myMCAN0.lse                  = myMCAN0_MCAN_EXT_ID_FILTER_NUM;
    //
    // List Size: Extended ID.
    //
    msgRAMConfigParams_myMCAN0.txStartAddr          = myMCAN0_MCAN_TX_BUFF_START_ADDR;
    //
    // Tx Buffers Start Address.
    //
    msgRAMConfigParams_myMCAN0.txBufNum             = myMCAN0_MCAN_TX_BUFF_SIZE;
    //
    // Number of Dedicated Transmit Buffers.
    //
    msgRAMConfigParams_myMCAN0.txFIFOSize           = 0;
    msgRAMConfigParams_myMCAN0.txBufMode            = 0;
    msgRAMConfigParams_myMCAN0.txBufElemSize        = MCAN_ELEM_SIZE_64BYTES;
    //
    // Tx Buffer Element Size.
    //
    msgRAMConfigParams_myMCAN0.txEventFIFOStartAddr = myMCAN0_MCAN_TX_EVENT_START_ADDR;
    //
    // Tx Event FIFO Start Address.
    //
    msgRAMConfigParams_myMCAN0.txEventFIFOSize      = myMCAN0_MCAN_TX_EVENT_SIZE;
    //
    // Event FIFO Size.
    //
    msgRAMConfigParams_myMCAN0.txEventFIFOWaterMark = 3;
    //
    // Level for Tx Event FIFO watermark interrupt.
    //
    msgRAMConfigParams_myMCAN0.rxFIFO0startAddr     = myMCAN0_MCAN_FIFO_0_START_ADDR;
    //
    // Rx FIFO0 Start Address.
    //
    msgRAMConfigParams_myMCAN0.rxFIFO0size          = myMCAN0_MCAN_FIFO_0_NUM;
    //
    // Number of Rx FIFO elements.
    //
    msgRAMConfigParams_myMCAN0.rxFIFO0waterMark     = 3; // Rx FIFO0 Watermark.
    msgRAMConfigParams_myMCAN0.rxFIFO0OpMode        = 0;
    msgRAMConfigParams_myMCAN0.rxFIFO1startAddr     = myMCAN0_MCAN_FIFO_1_START_ADDR;
    //
    // Rx FIFO1 Start Address.
    //
    msgRAMConfigParams_myMCAN0.rxFIFO1size          = myMCAN0_MCAN_FIFO_1_NUM;
    //
    // Number of Rx FIFO elements.
    //
    msgRAMConfigParams_myMCAN0.rxFIFO1waterMark     = 3; // Level for Rx FIFO 1
                                                  // watermark interrupt.
    msgRAMConfigParams_myMCAN0.rxFIFO1OpMode        = 0; // FIFO blocking mode.
    msgRAMConfigParams_myMCAN0.rxBufStartAddr       = myMCAN0_MCAN_RX_BUFF_START_ADDR;
    //
    // Rx Buffer Start Address.
    //
    msgRAMConfigParams_myMCAN0.rxBufElemSize        = MCAN_ELEM_SIZE_64BYTES;
    //
    // Rx Buffer Element Size.
    //
    msgRAMConfigParams_myMCAN0.rxFIFO0ElemSize      = MCAN_ELEM_SIZE_64BYTES;
    //
    // Rx FIFO0 Element Size.
    //
    msgRAMConfigParams_myMCAN0.rxFIFO1ElemSize      = MCAN_ELEM_SIZE_64BYTES;
    //
    // Rx FIFO1 Element Size.
    //
    //
    // Initialize bit timings.
    //
    bitTimes_myMCAN0.nomRatePrescalar   = 7; // Nominal Baud Rate Pre-scaler.
    bitTimes_myMCAN0.nomTimeSeg1        = 4; // Nominal Time segment before sample point.
    bitTimes_myMCAN0.nomTimeSeg2        = 3; // Nominal Time segment after sample point.
    bitTimes_myMCAN0.nomSynchJumpWidth  = 3; // Nominal (Re)Synchronization Jump Width Range.
    bitTimes_myMCAN0.dataRatePrescalar  = 3; // Data Baud Rate Pre-scaler.
    bitTimes_myMCAN0.dataTimeSeg1       = 4; // Data Time segment before sample point.
    bitTimes_myMCAN0.dataTimeSeg2       = 3; // Data Time segment after sample point.
    bitTimes_myMCAN0.dataSynchJumpWidth = 3; // Data (Re)Synchronization Jump Width.
    //
    // Get MCANSS Revision ID.
    //
    MCAN_getRevisionId(myMCAN0_BASE, &revid_myMCAN0);
    //
    // Wait for Memory initialization to be completed.
    //
    while(0 == MCAN_isMemInitDone(myMCAN0_BASE));
    //
    // Put MCAN in SW initialization mode.
    //
    MCAN_setOpMode(myMCAN0_BASE, MCAN_OPERATION_MODE_SW_INIT);
    //
    // Wait till MCAN is not initialized.
    //
    while (MCAN_OPERATION_MODE_SW_INIT != MCAN_getOpMode(myMCAN0_BASE));
    //
    // Initialize MCAN module.
    //
    MCAN_init(myMCAN0_BASE, &initParams_myMCAN0);
    //
    // Configure MCAN module.
    //
    MCAN_config(myMCAN0_BASE, &configParams_myMCAN0);
    //
    // Configure Bit timings.
    //
    MCAN_setBitTime(myMCAN0_BASE, &bitTimes_myMCAN0);
    //
    // Configure Message RAM Sections
    //
    MCAN_msgRAMConfig(myMCAN0_BASE, &msgRAMConfigParams_myMCAN0);

    stdFiltelem_myMCAN0.sft = MCAN_STDFILT_RANGE;
    stdFiltelem_myMCAN0.sfec = MCAN_STDFILTEC_RXBUFF;
    stdFiltelem_myMCAN0.sfid1 = 4;
    stdFiltelem_myMCAN0.sfid2 = 0;

    //
    // Configure Standard ID filter element 0
    //
    MCAN_addStdMsgIDFilter(myMCAN0_BASE, 0U, &stdFiltelem_myMCAN0);
    //
    // Internal loopback mode
    //
    MCAN_lpbkModeEnable(myMCAN0_BASE, MCAN_LPBK_MODE_EXTERNAL, true);
    //
    // Take MCAN out of the SW initialization mode
    //
    MCAN_setOpMode(myMCAN0_BASE, MCAN_OPERATION_MODE_NORMAL);
    while (MCAN_OPERATION_MODE_NORMAL != MCAN_getOpMode(myMCAN0_BASE));
    //
    // Enable MCAN Interrupts
    //
    MCAN_enableIntr(myMCAN0_BASE, MCAN_INTR_MASK_ALL, 1U);
    MCAN_selectIntrLine(myMCAN0_BASE, MCAN_INTR_MASK_ALL, MCAN_INTR_LINE_NUM_1);
    MCAN_enableIntrLine(myMCAN0_BASE, MCAN_INTR_LINE_NUM_1, 1U);
}

