//#############################################################################
//
// FILE:   mcan_ex4_transmit.c
//
// TITLE:  MCAN External Transmit using Tx Buffer
//
//! \addtogroup driver_example_list
//! <h1> MCAN External Transmit using Tx Buffer </h1>
//!
//! This example demonstrates the MCAN External Transmit function. External
//! communication is done between two CAN nodes. The receiving node could be
//! another MCU or a CAN bus analysis tool capable of Receiving/ACKnowledging
//! transmitted frames. The transmit and receive pins of the MCAN module
//! should be connected to a CAN Transceiver. Nominal Bit Rate of 1 mbps and
//! Data bit rate of 2 Mbps is used.
//! Standard Identifier (STD ID) 0x4 is transmitted with 64 data bytes.
//! #defines that are not required for this test case have been commented out.
//! However, they have been left in the code should the scope of this code
//! be expanded to include Receive and FIFO functions.
//!
//! If another C2000 MCU is used as the receiver, mcan_ex5_receive.c can be
//! run on it for the receive function.
//!
//! NOTE -  Select XTAL clock source by doing the following change in device.h file:
//!         #define USE_PLL_SRC_XTAL
//!         //#define USE_PLL_SRC_INTOSC
//!
//! \b Hardware \b Required \n
//!  - A C2000 board with CAN transceiver
//!
//! \b External \b Connections \n
//!  Both nodes should communicate through CAN FD capable transceivers.
//!
//!  - MCAN is on GPIO_65_MCANA_RX (MCANRXA)
//!  - and GPIO_64_MCANA_TX (MCANTXA)
//!
//! \b Watch \b Variables \n
//!  - txMsg
//!
//
//#############################################################################

//
// Include Files
//
#include "inc/stw_types.h"
#include "device.h"
#include "driverlib.h"

//
// MCANE_RX - GPIO Settings
//
#define GPIO_PIN_MCANE_RX 47
#define myMCAN0_MCANRX_GPIO 47
#define myMCAN0_MCANRX_PIN_CONFIG GPIO_47_MCANE_RX
//
// MCANE_TX - GPIO Settings
//
#define GPIO_PIN_MCANE_TX 46
#define myMCAN0_MCANTX_GPIO 46
#define myMCAN0_MCANTX_PIN_CONFIG GPIO_46_MCANE_TX

#define myMCAN0_BASE MCANE_DRIVER_BASE

//
// GPIO23 - GPIO Settings
//
#define LEDBlink_GPIO_PIN_CONFIG GPIO_23_GPIO23

#define LEDBlink 23
void LEDBlink_init();

//
// Defines.
//
#define NUM_OF_MSG                      (3U)
#define MCAN_STD_ID_FILTER_NUM          (0U)
#define MCAN_EXT_ID_FILTER_NUM          (0U)
#define MCAN_FIFO_0_NUM                 (0U)
#define MCAN_FIFO_0_ELEM_SIZE           (MCAN_ELEM_SIZE_64BYTES)
#define MCAN_FIFO_1_NUM                 (0U)
#define MCAN_FIFO_1_ELEM_SIZE           (MCAN_ELEM_SIZE_64BYTES)
#define MCAN_RX_BUFF_NUM                (0U)
#define MCAN_RX_BUFF_ELEM_SIZE          (MCAN_ELEM_SIZE_64BYTES)
#define MCAN_TX_BUFF_SIZE               (NUM_OF_MSG)
#define MCAN_TX_FQ_SIZE                 (0U)
#define MCAN_TX_BUFF_ELEM_SIZE          (MCAN_ELEM_SIZE_64BYTES)
#define MCAN_TX_EVENT_SIZE              (0U)

//
//  Defining Starting Addresses for Message RAM Sections,
//  (Calculated from Macros based on User defined configuration above)
//
#define MCAN_STD_ID_FILT_START_ADDR     (0x0U)
#define MCAN_EXT_ID_FILT_START_ADDR     (MCAN_STD_ID_FILT_START_ADDR + ((MCAN_STD_ID_FILTER_NUM * MCANSS_STD_ID_FILTER_SIZE_WORDS * 4U)))
#define MCAN_FIFO_0_START_ADDR          (MCAN_EXT_ID_FILT_START_ADDR + ((MCAN_EXT_ID_FILTER_NUM * MCANSS_EXT_ID_FILTER_SIZE_WORDS * 4U)))
#define MCAN_FIFO_1_START_ADDR          (MCAN_FIFO_0_START_ADDR + (MCAN_getMsgObjSize(MCAN_FIFO_0_ELEM_SIZE) * 4U * MCAN_FIFO_0_NUM))
#define MCAN_RX_BUFF_START_ADDR         (MCAN_FIFO_1_START_ADDR + (MCAN_getMsgObjSize(MCAN_FIFO_1_ELEM_SIZE) * 4U * MCAN_FIFO_1_NUM))
#define MCAN_TX_BUFF_START_ADDR         (MCAN_RX_BUFF_START_ADDR + (MCAN_getMsgObjSize(MCAN_RX_BUFF_ELEM_SIZE) * 4U * MCAN_RX_BUFF_NUM))
#define MCAN_TX_EVENT_START_ADDR        (MCAN_TX_BUFF_START_ADDR + (MCAN_getMsgObjSize(MCAN_TX_BUFF_ELEM_SIZE) * 4U * (MCAN_TX_BUFF_SIZE + MCAN_TX_FQ_SIZE)))


//
// Global Variables.
//
MCAN_TxBufElement    txMsg[3];
MCAN_RxBufElement    rxMsg[3];
MCAN_RxFIFOStatus    newData_FIFOStatus;
uint32_t            loopCnt = 0U;

//
// Function Prototype.
//
static void MCANConfig(void);

int main(void)
{
    uint32_t    i = 0;
    uint32_t    dataBytes = 64;

    //
    // Initialize device clock and peripherals
    //
    Device_init();

    //
    // Initialize GPIO and unlock the GPIO configuration registers
    //
    Device_initGPIO();

    //
    // Configure the divisor for the MCAN bit-clock
    // -> MCAN_CLOCK = SYSPLLCLK/10 = 200/20 = 10 MHz
    //
    SysCtl_setMCANClock(
                SYSCTL_MCAN_E,
                SYSCTL_MCANCLK_SOURCE_SYS,
                SYSCTL_MCANCLK_DIV_20);

    LEDBlink_init();

	//
	// MCANE -> myMCAN0 Pinmux
	//
	GPIO_setPinConfig(myMCAN0_MCANRX_PIN_CONFIG);
	GPIO_setPadConfig(myMCAN0_MCANRX_GPIO, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(myMCAN0_MCANRX_GPIO, GPIO_QUAL_ASYNC);

	GPIO_setPinConfig(myMCAN0_MCANTX_PIN_CONFIG);
	GPIO_setPadConfig(myMCAN0_MCANTX_GPIO, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(myMCAN0_MCANTX_GPIO, GPIO_QUAL_ASYNC);

    //
    // Initialize message to transmit.
    //
    // ********* Msg 1 *********
    txMsg[0].id       = ((uint32_t)(0x7F2)) << 18U; // Identifier Value.
    txMsg[0].rtr      = 0U; // Transmit data frame.
    txMsg[0].xtd      = 0U; // 11-bit standard identifier.
    txMsg[0].esi      = 0U; // ESI bit in CAN FD format depends only on error
                         // passive flag.
    txMsg[0].dlc      = 8U; // CAN + CAN FD: transmit frame has 0-8 data bytes.
    txMsg[0].brs      = 0; // CAN FD frames transmitted with bit rate
                         // switching.
    txMsg[0].fdf      = 0U; // Frame transmitted in CAN FD format.
    txMsg[0].efc      = 1U; // Store Tx events.
    txMsg[0].mm       = 0xAAU; // Message Marker.

    //
    // Data bytes.
    //
    txMsg[0].data[0]  = 0x12;
    txMsg[0].data[1]  = 0x34;
    txMsg[0].data[2]  = 0x56;
    txMsg[0].data[3]  = 0x78;
    txMsg[0].data[4]  = 0x90;
    txMsg[0].data[5]  = 0x12;
    txMsg[0].data[6]  = 0x34;
    txMsg[0].data[7]  = 0x56;

    // ********* Msg 2 *********
    txMsg[1].id       = ((uint32_t)(1598)) << 18U; // Identifier Value.
    txMsg[1].rtr      = 0U; // Transmit data frame.
    txMsg[1].xtd      = 0U; // 11-bit standard identifier.
    txMsg[1].esi      = 0U; // ESI bit in CAN FD format depends only on error
                         // passive flag.
    txMsg[1].dlc      = 8U; // CAN + CAN FD: transmit frame has 0-8 data bytes.
    txMsg[1].brs      = 0; // CAN FD frames transmitted with bit rate
                         // switching.
    txMsg[1].fdf      = 0U; // Frame transmitted in CAN FD format.
    txMsg[1].efc      = 1U; // Store Tx events.
    txMsg[1].mm       = 0xAAU; // Message Marker.

    //
    // Data bytes.
    //
    txMsg[1].data[0]  = 0x22;
    txMsg[1].data[1]  = 0x22;
    txMsg[1].data[2]  = 0x22;
    txMsg[1].data[3]  = 0x22;
    txMsg[1].data[4]  = 0x22;
    txMsg[1].data[5]  = 0x22;
    txMsg[1].data[6]  = 0x22;
    txMsg[1].data[7]  = 0x22;

    // ********* Msg 3 *********
    txMsg[2].id       = ((uint32_t)(1739)) << 18U; // Identifier Value.
    txMsg[2].rtr      = 0U; // Transmit data frame.
    txMsg[2].xtd      = 0U; // 11-bit standard identifier.
    txMsg[2].esi      = 0U; // ESI bit in CAN FD format depends only on error
                         // passive flag.
    txMsg[2].dlc      = 8U; // CAN + CAN FD: transmit frame has 0-8 data bytes.
    txMsg[2].brs      = 0U; // CAN FD frames transmitted with bit rate
                         // switching.
    txMsg[2].fdf      = 0U; // Frame transmitted in CAN FD format.
    txMsg[2].efc      = 1U; // Store Tx events.
    txMsg[2].mm       = 0xAAU; // Message Marker.

    //
    // Data bytes.
    //
    txMsg[2].data[0]  = 0x33;
    txMsg[2].data[1]  = 0x33;
    txMsg[2].data[2]  = 0x33;
    txMsg[2].data[3]  = 0x33;
    txMsg[2].data[4]  = 0x33;
    txMsg[2].data[5]  = 0x33;
    txMsg[2].data[6]  = 0x33;
    txMsg[2].data[7]  = 0x33;

    i = 0;

    //
    // Configure the MCAN Module.
    //
    MCANConfig();

    newData_FIFOStatus.num = MCAN_RX_FIFO_NUM_1;

    uint32_t num = 0;
    uint32_t Status_Transmission = 0;

    //
    // Wait for the isrIntr1Flag to be reset.
    //
    while(1)
    {
        // Infinite Loop to be able to send Msg If needed

        //
        // Write Tx Message to the Message RAM.
        //
        MCAN_writeMsgRam(myMCAN0_BASE, MCAN_MEM_TYPE_BUF, (uint32_t)(num), &txMsg[num]);

        //
        // Enable Transmission interrupt.
        //
        MCAN_txBufTransIntrEnable(myMCAN0_BASE, (uint32_t)(num),1U);

        //
        // Add request for transmission.
        //
        MCAN_txBufAddReq(myMCAN0_BASE, (uint32_t)(num));

//        while((MCAN_getTxBufReqPend(myMCAN0_BASE)) && (count_Pending != 0))
//        {
//            Status_Transmission = MCAN_getTxBufTransmissionStatus(myMCAN0_BASE);
//            count_Pending--;
//        }

//        count_Pending = 1000;

        num++;

        if(num>2){ num = 0;}

        DEVICE_DELAY_US(1000000);
        GPIO_togglePin(LEDBlink);

    }

    //
    // Stop Application.
    //
    ESTOP0;
}

static void MCANConfig(void)
{
    MCAN_InitParams         initParams;
    MCAN_MsgRAMConfigParams msgRAMConfigParams;
    MCAN_BitTimingParams    bitTimes;

    //
    //  Initializing all structs to zero to prevent stray values
    //
    memset(&initParams, 0, sizeof(initParams));
    memset(&msgRAMConfigParams, 0, sizeof(msgRAMConfigParams));
    memset(&bitTimes, 0, sizeof(bitTimes));

    //
    // Initialize MCAN Init parameters.
    //
    initParams.fdMode            = false;
    initParams.brsEnable         = false;
    initParams.txpEnable         = false;
    initParams.efbi              = false;
    initParams.pxhddisable       = false;
    initParams.darEnable         = true;
    initParams.wkupReqEnable     = false;
    initParams.autoWkupEnable    = false;
    initParams.emulationEnable   = false;
    initParams.tdcEnable         = false;
    initParams.wdcPreload        = 255;

    //
    // Transmitter Delay Compensation parameters.
    //
    initParams.tdcConfig.tdcf           = 0x0U;
    initParams.tdcConfig.tdco           = 0x0U;

    //
    // Initialize Message RAM Sections Configuration Parameters
    //
    msgRAMConfigParams.txStartAddr      = MCAN_TX_BUFF_START_ADDR;
    // Tx Buffers Start Address.

    msgRAMConfigParams.txBufNum         = MCAN_TX_BUFF_SIZE;
    // Number of Dedicated Transmit Buffers.

    msgRAMConfigParams.txBufMode        = 0U;

    msgRAMConfigParams.txFIFOSize       = MCAN_TX_FQ_SIZE;
    // Number of Tx FIFO or Tx Queue Elements

    msgRAMConfigParams.txBufElemSize    = MCAN_TX_BUFF_ELEM_SIZE;
    // Tx Buffer Element Size.

    //
    // Initialize bit timings.
    // -> NOM_BITRATE = (MCAN_CLOCK/(bitTimes.nomRatePrescalar + 1))/(bitTimes.nomTimeSeg1 + bitTimes.nomTimeSeg2 + 3)
    //                = (10/(1 + 1))/(8 + 9 + 3) = 5/20 = 250 kbps
    //
    bitTimes.nomRatePrescalar   = 0x1U; // Nominal Baud Rate Pre-scaler
    bitTimes.nomTimeSeg1        = 0x9U; // Nominal Time segment before SP
    bitTimes.nomTimeSeg2        = 0x8U; // Nominal Time segment after SP
    bitTimes.nomSynchJumpWidth  = 0x8U; // Nominal SJW
    bitTimes.dataRatePrescalar  = 0x0U; // Data Baud Rate Pre-scaler
    bitTimes.dataTimeSeg1       = 0x0U; // Data Time segment before SP
    bitTimes.dataTimeSeg2       = 0x0U; // Data Time segment after SP
    bitTimes.dataSynchJumpWidth = 0x0U; // Data SJW
    //
    // Wait for memory initialization to happen.
    //
    while(FALSE == MCAN_isMemInitDone(myMCAN0_BASE))
    {
        ;
    }

    //
    // Put MCAN in SW initialization mode.
    //
    MCAN_setOpMode(myMCAN0_BASE, MCAN_OPERATION_MODE_SW_INIT);

    //
    // Wait till MCAN is not initialized.
    //
    while (MCAN_OPERATION_MODE_SW_INIT != MCAN_getOpMode(myMCAN0_BASE))
    {
        ;
    }

    //
    // Initialize MCAN module.
    //
    MCAN_init(myMCAN0_BASE, &initParams);

    //
    // Configure Bit timings.
    //
    MCAN_setBitTime(myMCAN0_BASE, &bitTimes);

    //
    // Configure Message RAM Sections
    //
    MCAN_msgRAMConfig(myMCAN0_BASE, &msgRAMConfigParams);

    //
    // Take MCAN out of the SW initialization mode
    //
    MCAN_setOpMode(myMCAN0_BASE, MCAN_OPERATION_MODE_NORMAL);

    while(MCAN_OPERATION_MODE_NORMAL != MCAN_getOpMode(myMCAN0_BASE))
    {
        ;
    }
}

//*****************************************************************************
//
// GPIO Configurations
//
//*****************************************************************************
void LEDBlink_init(){
    GPIO_writePin(LEDBlink, 0);
    GPIO_setPadConfig(LEDBlink, GPIO_PIN_TYPE_STD);
    GPIO_setQualificationMode(LEDBlink, GPIO_QUAL_SYNC);
    GPIO_setDirectionMode(LEDBlink, GPIO_DIR_MODE_OUT);
    GPIO_setControllerCore(LEDBlink, GPIO_CORE_CPU1);
}
