//----------------------------------------------------------------------------------
//	FILE:			PWM_MicroInvLineClamp_CNF.c
//
//	Description:	PWM for line clamped inverter
//
//	Version: 		2.0
//
//  Target:  		TMS320F2802x,
//					TMS320F2803x,
//
// The function call is:
//
// 		PWM_1ch_UpDwnCnt_CNF(int16 n, int16 period, int16 mode, int16 phase)
//
// Function arguments defined as:
//-------------------------------
// n = 		Target ePWM module, 1,2,...16.  e.g. if n=2, then target is ePWM2
// period = PWM period in Sysclks
// mode =	Master/Slave mode, e.g. mode=1 for master, mode=0 for slave
// phase =	phase offset from upstream master in Sysclks,
//			applicable only if mode=0, i.e. slave
//--------------------------------------------------------------------------------
#include "DSP2803x_Device.h"
#include "DSP2803x_EPWM_defines.h"

extern volatile struct EPWM_REGS *ePWM[];


void PWM_MicroInvLineClamp_CNF(int n,int period,int deadband_rising,int deadband_falling)
{
	EALLOW;
	/* 	---------------------- Init EPWM n ------------------------------------------------ */
	(*ePWM[n]).TBPRD 		   			= period/2;     		/* Set timer period */
	(*ePWM[n]).TBCTL.bit.CTRMODE   		= TB_COUNT_UPDOWN;	/* Symmetric wave*/
	(*ePWM[n]).TBCTL.bit.PHSEN     		= TB_DISABLE;      	/* Disable phase loading*/
	(*ePWM[n]).TBCTL.bit.HSPCLKDIV 		= 0;          		/* Clock ratio to SYSCLKOUT*/
	(*ePWM[n]).TBCTL.bit.CLKDIV    		= 0;				/* SYSCLKOUT / 1*/
	(*ePWM[n]).TBCTL.bit.SYNCOSEL  		= TB_CTR_ZERO;		/* Sync out when CTR =0*/
	(*ePWM[n]).TBCTL.bit.PRDLD 			= TB_SHADOW;		/* Shadow load for PRD*/
	/* Setup shadowing*/
	(*ePWM[n]).CMPCTL.bit.SHDWAMODE 	= CC_SHADOW;		/* Shadow Mode*/
	(*ePWM[n]).CMPCTL.bit.SHDWBMODE 	= CC_SHADOW;		/* Shadow Mode*/
	(*ePWM[n]).CMPCTL.bit.LOADAMODE 	= CC_CTR_ZERO;  	/* Load A on Zero*/
	(*ePWM[n]).CMPCTL.bit.LOADBMODE 	= CC_CTR_ZERO; 		/* Load B on Zero*/
	/* Set Actions */
	(*ePWM[n]).AQCTLA.bit.CAD 			= AQ_SET;
	(*ePWM[n]).AQCTLA.bit.CAU			= AQ_CLEAR;
	(*ePWM[n]).AQCTLA.bit.ZRO 			= AQ_NO_ACTION;
	/* Set Actions */
	(*ePWM[n]).AQCTLB.bit.CBD 			= AQ_SET;
	(*ePWM[n]).AQCTLB.bit.CBU			= AQ_CLEAR;
	(*ePWM[n]).AQCTLB.bit.ZRO 			= AQ_NO_ACTION;
	/* Set Dead Band*/
	(*ePWM[n]).DBCTL.bit.OUT_MODE 		= DB_DISABLE;	    /* Dead band is dsiabled*/
	/* 	---------------------- End Init EPWM n --------------------------------------------*/
	/* 	---------------------- Init EPWM n+1 ----------------------------------------------*/
	(*ePWM[n+1]).TBPRD 		   			= period/2;     		/* Set timer period */
	(*ePWM[n+1]).TBCTL.bit.CTRMODE  	= TB_COUNT_UPDOWN;	/* Symmetric wave*/
	(*ePWM[n+1]).TBCTL.bit.PHSEN    	= TB_ENABLE;      	/* Enable Phase loading*/
	(*ePWM[n+1]).TBCTL.bit.HSPCLKDIV 	= 0;          		/* Clock ratio to SYSCLKOUT*/
	(*ePWM[n+1]).TBCTL.bit.CLKDIV    	= 0;				/* SYSCLKOUT / 1*/
	(*ePWM[n+1]).TBCTL.bit.SYNCOSEL  	= TB_SYNC_IN;		/* Sync out the sync in signal */
	(*ePWM[n+1]).TBCTL.bit.PRDLD 		= TB_SHADOW;		/* Shadow load for PRD*/
	(*ePWM[n+1]).TBCTL.bit.PHSDIR       = TB_DOWN;			/* ON sync start counting down */
	(*ePWM[n+1]).TBPHS.half.TBPHS       = 50;        /* Add phase shift to ensure symmetric waveforms when crossing zero */
	/* Setup shadowing*/
	(*ePWM[n+1]).CMPCTL.bit.SHDWAMODE 	= CC_SHADOW;		/* Shadow Mode*/
	(*ePWM[n+1]).CMPCTL.bit.SHDWBMODE 	= CC_SHADOW;		/* Shadow Mode*/
	(*ePWM[n+1]).CMPCTL.bit.LOADAMODE 	= CC_CTR_ZERO;      /* Load A on PRD*/
	(*ePWM[n+1]).CMPCTL.bit.LOADBMODE 	= CC_CTR_ZERO; 		/* Load B on Zero*/
	/* Set Actions */
	(*ePWM[n+1]).AQCTLA.bit.CAD 		= AQ_SET;
	(*ePWM[n+1]).AQCTLA.bit.CAU			= AQ_CLEAR;
	(*ePWM[n+1]).AQCTLA.bit.ZRO 		= AQ_NO_ACTION;
	/* Set Dead Band for PWM1*/
	(*ePWM[n+1]).DBCTL.bit.OUT_MODE 	= DB_FULL_ENABLE;	/* Rising Delay on 1A & Falling Delay on 1B*/
	(*ePWM[n+1]).DBCTL.bit.POLSEL 		= DB_ACTV_HIC;		/* Active high complementary mode (EPWMxA is inverted)*/
	(*ePWM[n+1]).DBCTL.bit.IN_MODE 		= DBA_ALL; 			/* 1A for Rising& Falling*/
	(*ePWM[n+1]).DBRED 					= deadband_rising;	/* Delay at Rising edge*/
	(*ePWM[n+1]).DBFED 					= deadband_falling;	/* Delay at Falling edge*/
	/* 	---------------------- End Init EPWM n+1 ----------------------------------------*/
	/* Clear TB counter*/
	(*ePWM[n]).TBCTR            		= 0x0;         		/* Clear counter*/
	(*ePWM[n+1]).TBCTR            		= 0x0;         		/* Clear counter*/
	/* ADC SOC for Inverter Control*/
	(*ePWM[n]).ETSEL.bit.SOCAEN	= 1;						/* Enable SOC on A group*/
	(*ePWM[n]).ETSEL.bit.SOCASEL	= ET_CTR_ZERO ;			/* Select SOC from counter at ctr = 0*/
	(*ePWM[n]).ETPS.bit.SOCAPRD 	= ET_1ST;				/* Generate pulse on 1st even*/
	(*ePWM[n+1]).ETSEL.bit.SOCAEN	= 1;					/* Enable SOC on A group*/
	(*ePWM[n+1]).ETSEL.bit.SOCASEL	= ET_CTR_ZERO ;	/* Select SOC from counter at ctr = 0*/
	(*ePWM[n+1]).ETPS.bit.SOCAPRD 	= ET_1ST;			/* Generate pulse on 1st event*/
	EDIS;
}
