/*
 * NG_Converter_support.c
 *
 *  Created on: 29 Apr 2019
 *      Author: Ed
 *  Edited  on: 16 Jun 2019
 *      Author: Davide
 */

#include "NG_Converter_support.h"
#include <DCLF32.h> // Digital Control Lib
#include "utils.h"

volatile struct TRIPS_STRUCT TRIPS; // struct defines custom data type that can hold multiple variables of different types, volatile means the elements/contents of this structure may change asynchronously, perhaps due to external hardware or interrupt handlers modifying its values.

float ADC_LOWPASS;              // Declares the variable as a float
float ADC_READINGS[26];         // Declares variable as an array of 26 elements

float ADC_OFFSET[26]={
                      32832.8477,  // ADC_VA_VNX
                      32789.5469,
                      32860.4844,
                      32847.543 ,
                      32843.5117,
                      32844.4805,
                      32892.0,
                      32855.6055,
                      32848.3281,
                      32748.9961,
                      32382.4023,
                      32721.2832,
                      33100.2656,
                      32758.3691,
                      49169.3633,
                      49157.7305,
                      49213.1211,
                      32420.5801,
                      35283, //TEMP1
                      30811, //TEMP2
                      32727.498 ,
                      32848.5156,
                      32807.5586,
                      32767.5176,
                      32778.0078,
                      32782.5469,    // ADC_x_1A_C


                      };                // Declares 26 elements of the array

float ADC_GAIN[26]={

                    0.0096151717,   // ADC_VA_VNX
                    //0.0095680384,   // Just added
                    0.0095680384,
                    0.0095680384,
                    0.04032258064,  //0.000999418669
                    0.00975939911,
                    0.00975939911,
                    0.0096151717,
                    0.0096151717,
                    -0.00623538578,
                    -0.00623538578,
                    -0.00623538578,
                    -0.00623538578,
                    -0.00623538578,
                    0.00773993786,
                    0.00773993786,
                    0.00773993786,
                    0.00773993786,
                    -0.00623538578,
                    0.00205353089,  //TEMP1
                    -0.00205353089, //TEMP2
                    0.00649000006,
                    0.00649000006,
                    0.00649000006,
                    7.8505007e-05,
                    7.8505007e-05,
                    7.8505007e-05, // ADC_x_1A_C

                    };              // Declares 26 elements of the array

Uint16 ADC_UP_LIMITS[26]={65535,65535,65535,65535,65535,65535,65535,65535,65535,65535,65535,65535,65535,65535,65535,65535,65535,65535,65535,65535,65535,65535,65535,65535,65535,65535}; // ADC Array that can hold 26 elements of unsigned 16-bit integer data type
Uint16 ADC_LOW_LIMITS[26]={0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0}; // ADC Array that can hold 26 elements of unsigned 16-bit integer data type

float curr_FSW=0; //Global Store of current switching frequency

int Initialise_FPGA(void)
{

    *(Uint16 *)FPGA_REG_01=1;          //check connection with FPGA
    if(*(Uint16 *)FPGA_REG_01==1)
    {
        RESET_FPGA();


        *(Uint16 *)FPGA_REG_12 = 0x0000;       //Enable / Disable Trips - Group A
        *(Uint16 *)FPGA_REG_13 = 0x0000;       //Enable / Disable Trips - Group B
        *(Uint16 *)FPGA_REG_14 = 0x0000;       //Enable / Disable Trips - Group C
        *(Uint16 *)FPGA_REG_15 = 0x0000;       //Enable / Disable Trips - Group D

        SET_SWITCHING_FREQUENCY (Para_Float[10]);  //PF10 Para[60] is in 100's hz

        *(Uint16 *)FPGA_REG_04 |= 75;         // Phase A,B,C and N - Load deadtime value (8bit 7:0) = DEAD_TIME / 50MHz (bitwise OR operation)
        *(Uint16 *)FPGA_REG_04 |= 75<<8;      // AUX1 and AUX2 - Load deadtime value (8bit 15:8) = DEAD_TIME / 50MHz (bitwise OR operation)

        *(Uint16 *)FPGA_REG_03 |= 0x0010;       // Enable interrupt
        return 0;
    }
    else
    {
        return 1;       //cannot communicate with FPGA
    }

}


void Enable_Converter(void)
{
    Relay_control(0, 1); //switch on contactor and pass the 2 arguments in the brackets (0 represents the the contactor to be controlled and 1 represents ON state
    GpioDataRegs.GPADAT.bit.GPIO0 = 1;      // Set LED to show PWM status

    PWM_En = 1;
    *(Uint16 *)FPGA_REG_08 = 0;//0xFFFF;       // Enable pwm output
    *(Uint16 *)FPGA_REG_08 |= 1<<5;            // performs a bitwise OR operation to set the 6th bit (1<<5) of the value stored at FPGA_REG_08 to 1.

    *(Uint16 *)FPGA_REG_08 |= 1<<8;     //phase A // sets the 9th bit (1<<8) of the value stored at FPGA_REG_08 to 1
    *(Uint16 *)FPGA_REG_08 |= 1<<9;     //phase b // sets the 10th bit (1<<9) of the value stored at FPGA_REG_08 to 1
    *(Uint16 *)FPGA_REG_08 |= 1<<10;    //phase c // sets the 11th bit (1<<10) of the value stored at FPGA_REG_08 to 1

}

extern DCL_PI PI_Id,PI_Iq,PI_PLL;  // Defined in DCL.h

void Disable_Converter(void)
{
    Relay_control(0, 0); //switch off contactor
    GpioDataRegs.GPADAT.bit.GPIO0 = 0;    // Set LED to show PWM status
    PWM_En = 0;
    *(Uint16 *)FPGA_REG_08 = 0x0000;       // Disable pwm output

    //change parameters: TRIPS
    setTrip(ADC_IA,     Para_Float[61-50], -Para_Float[61-50]);            // ADC Channel, upper limit(Para_Float[11]), Lower limit(-Para_Float[11])
    setTrip(ADC_IB,     Para_Float[62-50], -Para_Float[62-50]);           // ADC Channel, upper limit(Para_Float[12]), Lower limit(-Para_Float[12])
    setTrip(ADC_IC,     Para_Float[63-50], -Para_Float[63-50]);          // ADC Channel, upper limit(Para_Float[13]), Lower limit(-Para_Float[13])etc
    setTrip(ADC_IAUX1,  Para_Float[64-50], -Para_Float[64-50]);
    setTrip(ADC_IAUX2,  Para_Float[65-50], -Para_Float[65-50]);
    setTrip(ADC_IN,     Para_Float[66-50], -Para_Float[66-50]);

    setTrip(ADC_VHTP_VHTN,  Para_Float[67-50], Para_Float[68-50]);
    setTrip(ADC_IHTP,       Para_Float[69-50], Para_Float[70-50]);

    //change parameters: CONROLLERS
 /*   PI_Id.Kp=Para_Float[75-50];
    PI_Id.Kp=Para_Float[76-50];
    PI_Iq.Ki=Para_Float[77-50];
    PI_Iq.Ki=Para_Float[78-50];
    PI_PLL.Kp=Para_Float[79-50];
    PI_PLL.Ki=Para_Float[80-50];*/


    if(curr_FSW != Para_Float[10])  //Detect FSW change
        Change_switching_frequency();
}

void Change_switching_frequency(void)
{
    *(Uint16 *)FPGA_REG_03 = 0x0000;       // Disable interrupt
    *(Uint16 *)FPGA_REG_08 = 0x0000;        //Disable all outputs
    *(Uint16 *)FPGA_REG_03 |= 0x0002;       //Reset PWM unit // Sets the 2nd bit (0x0002) of the value stored at FPGA_REG_03 to 1
    *(Uint16 *)FPGA_REG_03 &= 0xFFFD;       //Clear PWM reset line // performs a bitwise AND operation to clear the 2nd bit (0x0002) of the value stored at FPGA_REG_03

  //  CONFIG_TRIP_EN_DIS(); // Reset Trip state back to default

    SET_SWITCHING_FREQUENCY (Para_Float[10]);  //PF10 Para[60] is in 100's hz

    *(Uint16 *)FPGA_REG_03 |= 0x0010;          //Enable interrupt // performs a bitwise OR operation to set the 5th bit (0x0010) of the value stored at FPGA_REG_03 to 1
}

void CONFIG_TRIP_EN_DIS(void)
{
    /* Enable / Disable individual trips */

    /*
     * GROUP A:
     * A0:      Watchdog
     * Others:  unused
     *
     * GROUP B:
     * B0-15 => connected to register FPGA_REG_12 0-15
     * useful to trip the fpga using the dsp
     *
     * GROUP C:
     * C0-15 => ADC trips: ADC0-15
     *
     * GROUP D:
     * D0-9  => ADC trips: ADC16-26
     * Others: unused
     *
     */

    *(Uint16 *)FPGA_REG_12 = 0x0001;       //Enable / Disable Trips - Group A
    *(Uint16 *)FPGA_REG_13 = 0xFFFF;       //Enable / Disable Trips - Group B
    *(Uint16 *)FPGA_REG_14 = 0;            //Enable / Disable Trips - Group C
    *(Uint16 *)FPGA_REG_15 = 0;            //Enable / Disable Trips - Group D
}



void Reset_Trips(void)
{
    Uint16 i=1000;

    *(Uint16 *)FPGA_REG_01 |= 0x0100;       //Reset Trips


    while(i!=0)
        i--;

    *(Uint16 *)FPGA_REG_01 &= 0xFEFF;       //Release from Trip reset


}

void Close_Precharge_Relay(void)
{
    GpioDataRegs.GPBDAT.bit.GPIO33=1;

    Relay_control(3, 1);                    // Precharge Relay 3 is turned 0N (1)
}

void Open_Precharge_Relay(void)
{
    //GpioDataRegs.GPBDAT.bit.GPIO33=0;

    GpioDataRegs.GPBDAT.bit.GPIO33= 1;       // Relay HE1AN is now latched to ON

    Relay_control(3, 0);                    // Precharge Relay 3 is turned 0FF (0)

}

void Fan_Control(void)
{
    GpioDataRegs.GPADAT.bit.GPIO6=1;   //heatsink right fan
    GpioDataRegs.GPADAT.bit.GPIO7=1;   //precharge res. fan
    GpioDataRegs.GPADAT.bit.GPIO8=1;   //heatsink left fan
    GpioDataRegs.GPADAT.bit.GPIO9=1;   //Gate drivers fan
}

void Relay_control(Uint16 relay, bool state)
{
    /*
     * changes relay state by setting or clearing the corresponding bit in FPGA_REG_06
     */
    switch(relay)
    {
    case 0:     //relay A
        if(state) *(Uint16 *)FPGA_REG_06 |= 0b0000000000000001;     // If state is true, the 1st bit of FPGA_REG_06 is set to 1
        else      *(Uint16 *)FPGA_REG_06 &= 0b1111111111111110;     // If false, the 1st bit of FPGA_REG_06 is cleared to 0
    break;
    case 1:     //relay B
        if(state) *(Uint16 *)FPGA_REG_06 |= 0b0000000000000010;     // If state is true, the 2nd bit of FPGA_REG_06 is set to 1
        else      *(Uint16 *)FPGA_REG_06 &= 0b1111111111111101;     // If false, the 2nd bit of FPGA_REG_06 is cleared to 0
    break;
    case 2:     //relay C
        if(state) *(Uint16 *)FPGA_REG_06 |= 0b0000000000000100;     // If state is true, the 3rd bit of FPGA_REG_06 is set to 1
        else      *(Uint16 *)FPGA_REG_06 &= 0b1111111111111011;     // If false, the 3rd bit of FPGA_REG_06 is cleared to 0
    break;
    case 3:     //relay D
        if(state) *(Uint16 *)FPGA_REG_06 |= 0b0000000000001000;     // If state is true, the 4th bit of FPGA_REG_06 is set to 1
        else      *(Uint16 *)FPGA_REG_06 &= 0b1111111111110111;     // If false, the 4th bit of FPGA_REG_06 is cleared to 0
    break;
    }
}

void RESET_FPGA(void)
{
    *(Uint16 *)FPGA_REG_03 |= 0x0001;      // Reset FPGA clears all registers, enable lines and resets PWM
    DELAY_US(500);

    *(Uint16 *)FPGA_REG_03 |= 0x0002;      // Reset PWM unit


    *(Uint16 *)FPGA_REG_03 &= 0xFFFD;      // Clear PWM reset line
    *(Uint16 *)FPGA_REG_08 = 0x0000;       // Disable pwm output
}


void SET_SWITCHING_FREQUENCY (float FSW)
{
//    Uint16 per_tmp=0;
    float ftmp = 0;
    Uint16 itmp = 0;

    curr_FSW = FSW;  // curr_FSW is assigned the value of FSW, storing the current switching

    if(FSW>=1)
    {
        ftmp = ((float)FPGA_PWM_CLK / (1000*FSW));
        itmp = ftmp;

        *(Uint16 *)FPGA_REG_02 = (0xFFFF - (itmp-1)); //0xEC78;       // 0xFFFF - ((1/fsw)/(1/50Mhz)-1)   10khzm = EC78 Load PWM switching period, CPU interrupt timing (16bit 15:0)
        curr_FSW = FSW;
    }

    SET_WATCH_DOG(itmp);

}

void SET_WATCH_DOG(Uint16 SWPER)
{
    int itmp=0;
    itmp = 1.5 * SWPER;
    *(Uint16 *)FPGA_REG_05 = itmp;       // Load watchdog period (16bit 15:0) 1.5 x 10khz

    *(Uint16 *)FPGA_REG_12 |= 0x0001;       //Enable WatchDog Trip
    *(Uint16 *)FPGA_REG_00 |= 0x0002;       //Enable Watchdog
}

void Modulator(float DCYa,float DCYb,float DCYc,float DCYn,float DCYaux1,float DCYaux2) // Represent duty cycle values for different channels or phases of the modulator.
{
    float halfPeriod=0;     // represents half of the modulation period
    int tX;

    halfPeriod = ((float)FPGA_PWM_CLK / (1000*curr_FSW))/2;

    if(DCYa>0.95) DCYa=0.95;   // Duty Cycle values checked and adjusted
    if(DCYa<0.05) DCYa=0.05;
    tX = halfPeriod*DCYa;
    if(tX<=0)           tX=1;
    if(tX>=halfPeriod)  tX=halfPeriod-1;

    *(Uint16 *)FPGA_REG_16    = (Uint16)(0xFFFF-halfPeriod-tX); //tON   // values are written to the respective registers
    *(Uint16 *)FPGA_REG_17    = (Uint16)(0xFFFF-halfPeriod+tX); //tOFF

    if(DCYb>0.95) DCYb=0.95;
    if(DCYb<0.05) DCYb=0.05;
    tX = halfPeriod*DCYb;
    if(tX<=0)           tX=1;
    if(tX>=halfPeriod)  tX=halfPeriod-1;

    *(Uint16 *)FPGA_REG_18    = (Uint16)(0xFFFF-halfPeriod-tX);
    *(Uint16 *)FPGA_REG_19    = (Uint16)(0xFFFF-halfPeriod+tX);

    if(DCYc>0.95) DCYc=0.95;
    if(DCYc<0.05) DCYc=0.05;
    tX = halfPeriod*DCYc;
    if(tX<=0)           tX=1;
    if(tX>=halfPeriod)  tX=halfPeriod-1;

    *(Uint16 *)FPGA_REG_20    = (Uint16)(0xFFFF-halfPeriod-tX);
    *(Uint16 *)FPGA_REG_21    = (Uint16)(0xFFFF-halfPeriod+tX);

    if(DCYn>0.95) DCYn=0.95;
    if(DCYn<0.05) DCYn=0.05;
    tX = halfPeriod*DCYn;
    if(tX<=0)           tX=1;
    if(tX>=halfPeriod)  tX=halfPeriod-1;

    *(Uint16 *)FPGA_REG_22    = (Uint16)(0xFFFF-halfPeriod-tX);
    *(Uint16 *)FPGA_REG_23    = (Uint16)(0xFFFF-halfPeriod+tX);

    if(DCYaux1>0.95) DCYaux1=0.95;
    if(DCYaux1<0.05) DCYaux1=0.05;
    tX = halfPeriod*DCYaux1;
    if(tX<=0)           tX=1;
    if(tX>=halfPeriod)  tX=halfPeriod-1;

    *(Uint16 *)FPGA_REG_24    = (Uint16)(0xFFFF-halfPeriod-tX);
    *(Uint16 *)FPGA_REG_25    = (Uint16)(0xFFFF-halfPeriod+tX);

    if(DCYaux2>0.95) DCYaux2=0.95;
    if(DCYaux2<0.05) DCYaux2=0.05;
    tX = halfPeriod*DCYaux2;
    if(tX<=0)           tX=1;
    if(tX>=halfPeriod)  tX=halfPeriod-1;

    *(Uint16 *)FPGA_REG_26    = (Uint16)(0xFFFF-halfPeriod-tX);
    *(Uint16 *)FPGA_REG_27    = (Uint16)(0xFFFF-halfPeriod+tX);
}


//ADC

void readTripsStatus(void) // responsible for reading the status of various trips from specific registers in the FPGA
{
    TRIPS.STATUS=*(Uint16 *)FPGA_REG_32;   // The function begins by reading the value from the register FPGA_REG_32 and storing it in the TRIPS.STATUS variable
    TRIPS.TRIP_A=*(Uint16 *)FPGA_REG_60;  //  Function reads value from FPGA_REG_60 and stores it in and assigns them to the corresponding member of the TRIPS struct: TRIPS.TRIP_A
    TRIPS.TRIP_B=*(Uint16 *)FPGA_REG_61;
    TRIPS.TRIP_C=*(Uint16 *)FPGA_REG_62;
    TRIPS.TRIP_D=*(Uint16 *)FPGA_REG_63;
}


Uint16 readFpgaReg(Uint16 channel)
{
    if(channel<=127)
        return *(Uint16 *)(FPGA_REG_00+channel); // The expression (FPGA_REG_00 + channel) calculates the memory address offset by adding the value of FPGA_REG_00 to the value of channel
    else
        return 0;
}

// The readFpgaReg function is used to read the value stored in a specific FPGA register based on the provided channel

//It performs a range check to ensure the channel is within a valid range, and if so, it retrieves the value from the corresponding register

//Otherwise, it returns 0 as a default value.


void writeFpgaReg(Uint16 value, Uint16 channel)
{
    if(channel<=127)
        *(Uint16 *)(FPGA_REG_00+channel)=value; // The expression (FPGA_REG_00 + channel) calculates the memory address offset by adding the value of FPGA_REG_00 to the value of channel. The (Uint16 *) cast is used to interpret this memory location as a pointer to a 16-bit unsigned integer (Uint16).
}

float ADCprocess3(Uint16 channel)
{
    return (((float)ADC_READ(channel))-ADC_OFFSET[channel])*ADC_GAIN[channel];
}


Uint16 OLD_CURRENTS[6]={500,500,500,500,500,500}; //  Represents the initial values for the old/currents
float ADC_CURRENTS_OFFSET[6]={
                              4981.12256,
                              5012.92627,
                              4977.46826,
                              4936.40167,
                              4997.48749,
                              5062.19086
                              };
float ADC_CURRENTS_GAINS[6]={
                             -0.03,
                             -0.03,
                             -0.03,
                             -0.03,
                             -0.03,
                             -0.03
                              };


void read_currents(float*I1, float*I2, float*I3, float*I4, float*I5, float*I6, float fsw) //  Takes 7 float pointers as parameters, representing the variables where the calculated current & fsw values will be stored
{
    Uint16 temp;

    temp=ADC_READ(ADC_IA); // Each channel is read using the ADC_READ function, and the obtained ADC value is stored in the temp variable
    //temp=*(Uint16 *) FPGA_REG_20;
    *I1=(((float)(temp-OLD_CURRENTS[0]))*fsw-ADC_CURRENTS_OFFSET[0])*ADC_CURRENTS_GAINS[0]; // For each channel, the function calculates the current value using the formula
    OLD_CURRENTS[0]=temp;   // Array is updated with the current ADC reading (temp) for each channel

    temp=ADC_READ(ADC_IB);
    //temp=*(Uint16 *) FPGA_REG_21;
    *I2=(((float)(temp-OLD_CURRENTS[1]))*fsw-ADC_CURRENTS_OFFSET[1])*ADC_CURRENTS_GAINS[1];
    OLD_CURRENTS[1]=temp;

    temp=ADC_READ(ADC_IC);
    //temp=*(Uint16 *) FPGA_REG_22;
    *I3=(((float)(temp-OLD_CURRENTS[2]))*fsw-ADC_CURRENTS_OFFSET[2])*ADC_CURRENTS_GAINS[2];
    OLD_CURRENTS[2]=temp;

    temp=ADC_READ(ADC_IN);
    //temp=*(Uint16 *) FPGA_REG_23;
    *I4=(((float)(temp-OLD_CURRENTS[3]))*fsw-ADC_CURRENTS_OFFSET[3])*ADC_CURRENTS_GAINS[3];
    OLD_CURRENTS[3]=temp;

    temp=ADC_READ(ADC_IAUX1);
    //temp=*(Uint16 *) FPGA_REG_24;
    *I5=(((float)(temp-OLD_CURRENTS[4]))*fsw-ADC_CURRENTS_OFFSET[4])*ADC_CURRENTS_GAINS[4];
    OLD_CURRENTS[4]=temp;

    temp=ADC_READ(ADC_IAUX2);
    //temp=*(Uint16 *) FPGA_REG_25;
    *I6=(((float)(temp-OLD_CURRENTS[5]))*fsw-ADC_CURRENTS_OFFSET[5])*ADC_CURRENTS_GAINS[5];
    OLD_CURRENTS[5]=temp;
}

void setTrip(Uint16 channel, float up, float low){
    /*
     * Sets upper and lower trip limits, taking care of the conversion
     */
    float upper,lower,temp;

    upper=up/ADC_GAIN[channel]+(float)ADC_OFFSET[channel];
    if(upper>0xFFFF) upper=0xFFFF;      // Ensures that the calculated upper values are within the valid ADC range (0 to 0xFFFF)
    if(upper<0)      upper=0;

    lower=low/ADC_GAIN[channel]+(float)ADC_OFFSET[channel];
    if(lower>0xFFFF) lower=0xFFFF;     // Ensures that the calculated lower values are within the valid ADC range (0 to 0xFFFF)
    if(lower<0)      lower=0;

    if(lower>upper){ // can happen if gain < 0  // // Swap the values of upper and lower using a temporary variable
        temp=upper;
        upper=lower;
        lower=temp; // The purpose of using temp is to avoid losing the original value of upper when it is assigned the value of lower. By temporarily storing the value of upper in temp, the value can later be assigned to lower without any data loss. After the swapping operation is complete, temp is no longer needed, and its value will be discarded.
    }

    if(channel<26){
        ADC_LOW_LIMIT_FPGA(channel) = (Uint16) lower;
        *(Uint16 *)(ASRAM_CS3_START_ADDR+64+channel)  = (Uint16) upper;         // ASRAM (an address in memory)

        if(channel<16)
            *(Uint16 *)FPGA_REG_14 |=(1<<channel); // (1 << channel) creates a bitmask with only the bit corresponding to the channel set to 1. The OR operation sets that bit in the register while preserving the states of other bits
        else
            *(Uint16 *)FPGA_REG_15 |=(1<<(channel-16));
    }
}

void lowpass(float X, float*Y, float tau)
{
    *Y+=tau*(X-*Y);   // Y =  Output , X = Input, tau = 1/(2*pi*fc)
}


//STARTUP PROCEDURE

extern float Wn;                             //grid pulsation used for PLL
float tauSequence=1e-3,omegaSequence=0;      // fc = 159.15Hz

Uint16 converterStart(void)
{
    //const float CapChargeTime=  6200;       //capacitor charge time in ms

    const float CapChargeTime=  6000;       //capacitor charge time in ms to correspond with the timer relay
    Uint16 i;
    float VdSequence,VqSequence,foo,oldTheta;


    GpioDataRegs.GPBDAT.bit.GPIO36= 1;      // enable diff transmitters


    for(i=0; i<1000;i++) DELAY_US(1000);   // Delay in microseconds = 1ms


    Open_Precharge_Relay();

    GpioDataRegs.GPBDAT.bit.GPIO36= 1;      // enable diff transmitters

    Fan_Control();   //turns on fans

    GpioDataRegs.GPBDAT.bit.GPIO36= 1;      // enable diff transmitters

    DELAY_US(500);                          // 0.5ms
    Initialise_FPGA();
    Disable_Converter();

    //setup trips
    CONFIG_TRIP_EN_DIS();                   //enable watchdog trip, disables ADC trips

    writeFpgaReg(7, 9);                     //set trips counter threshold to 7

    setTrip(ADC_IA,     25, -25);           //Upper and Lower trip limits
    setTrip(ADC_IB,     25, -25);
    setTrip(ADC_IC,     25, -25);
    setTrip(ADC_IAUX1,  10, -10);
    setTrip(ADC_IAUX2,  10, -10);
    setTrip(ADC_IN,     10, -10);


    setTrip(ADC_VHTP_VHTN,     750, -10);

    //setTrip(ADC_TEMP1,  -10, 50);
    //setTrip(ADC_TEMP2,  -10, 50);
    /*
     * DO NOT SET FPGA TRIPS ON TEMPS MEASUREMENTS!!!!
     *
     * Gets really noisy when converter in operation
     *
     */

    /*.
     * TURN ON CONTACTOR FOR MAIN POWER
     */

    Relay_control(0, 1);                    // Relay 0 is turned 0N (1)

    /*
     * WAIT FOR DC LINK CAPACITOR CHARGE
     */
    for(i=0; i<CapChargeTime; i++) DELAY_US(1000);  // Delay in microseconds



    Close_Precharge_Relay();

    /*
     * DETERMINE IF GRID SEQUENCE IS DIRECT OR INVERSE
     */
    for(i=0;i<10000;i++){

        dq0(
            ADCprocess3(ADC_VA_VNX),  // Reads and processes the analog input from ADC channel ADC_VA_VNX (Voltage Phase A to Neutral).
            ADCprocess3(ADC_VB_VNX),  // Reads and processes the analog input from ADC channel ADC_VB_VNX (Voltage Phase B to Neutral).
            ADCprocess3(ADC_VC_VNX),  // Reads and processes the analog input from ADC channel ADC_VC_VNX (Voltage Phase C to Neutral).
            &VdSequence,              // Pointer to a variable where the dq0 function will store the D-component of the voltage.
            &VqSequence,              // Pointer to a variable where the dq0 function will store the Q-component of the voltage.
            NULL,
            1, 0                       // 1 indicates that the inputs to dq0 are in per-unit values. 0 indicates that the transformation is performed for a grid voltage
            );

        foo=atan2(VqSequence,VdSequence); // Angle

        if(foo>oldTheta) omegaSequence++; // Grid sequence is direct, so omegaSequence is incremented (omegaSequence++).
        else omegaSequence--;             // Grid sequence is inverse, so omegaSequence is decremented (omegaSequence--).

        oldTheta=foo;

        DELAY_US(100);
    }

    if(omegaSequence>0) Wn=2*piFloat*50;   //direct sequence
    else                Wn=-2*piFloat*50;  //inverse sequence

    /*
     * Wait for PLL to lock
     */


    Reset_Trips();
    //Enable_Converter();                  // This enables the PWM automatically instead on expressions

    return 0;
}



