//###########################################################################
//
// FILE:   Example_2806xEPwmUpDownAQ.c
//
// TITLE:  ePWM Action Qualifier Module using up/down count Example
//
//! \addtogroup f2806x_example_list
//! <h1>ePWM Action Qualifier Module using up/down count (epwm_updown_aq)</h1>
//!
//! This example configures ePWM1, ePWM2, ePWM3 to produce an waveform with
//! independent modulation on EPWMxA and EPWMxB. The compare values CMPA
//! and CMPB are modified within the ePWM's ISR. The TB counter is in up/down
//! count mode for this example.
//!
//! Monitor ePWM1-ePWM3 pins on an oscilloscope as described
//!
//! \b External \b Connections \n
//!  - EPWM1A is on GPIO0
//!  - EPWM1B is on GPIO1
//!  - EPWM2A is on GPIO2
//!  - EPWM2B is on GPIO3
//!  - EPWM3A is on GPIO4
//!  - EPWM3B is on GPIO5
//
//###########################################################################
// $TI Release:  $
// $Release Date:  $
// $Copyright:
// Copyright (C) 2009-2024 Texas Instruments Incorporated - http://www.ti.com/
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
//
//   Redistributions of source code must retain the above copyright
//   notice, this list of conditions and the following disclaimer.
//
//   Redistributions in binary form must reproduce the above copyright
//   notice, this list of conditions and the following disclaimer in the
//   documentation and/or other materials provided with the
//   distribution.
//
//   Neither the name of Texas Instruments Incorporated nor the names of
//   its contributors may be used to endorse or promote products derived
//   from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// $
//###########################################################################

//
// Included Files
//
#include "DSP28x_Project.h"     // Device Headerfile and Examples Include File
#include "string.h"

//
// Typedef
//
typedef struct
{
    volatile struct EPWM_REGS *EPwmRegHandle;
    Uint16 EPwmSampleCount;
    Uint16 NK_Tsw;
    Uint16 NK_Tsw_1_third;
    Uint16 NK_Tsw_2_third;
    Uint16 SIN_i_Count;

//    Uint16 EPwmMinCMPA;
//    Uint16 EPwmMaxCMPA;
} EPWM_INFO;

//
// Function Prototypes
//
void InitEPwm1Example(Uint16);
void InitEPwm2Example(Uint16);
void InitEPwm3Example(Uint16);
void InitEPwm4Example(Uint16);

#pragma CODE_SECTION(epwm1_isr, "ramfuncs");
#pragma CODE_SECTION(update_compare, "ramfuncs");
#pragma CODE_SECTION(update_freq, "ramfuncs");

__interrupt void epwm1_isr(void);


void update_compare(EPWM_INFO*);
void update_freq(EPWM_INFO *info1, Uint16 n);


//
// Globals
//
EPWM_INFO epwm1_info;



Uint16 const N    = 256;      // total number of COS table
const float sin_table256[] =  {
                               1.0000000, 0.9996988, 0.9987955, 0.9972905, 0.9951847, 0.9924795, 0.9891765, 0.9852776,
                               0.9807853, 0.9757021, 0.9700313, 0.9637761, 0.9569403, 0.9495282, 0.9415441, 0.9329928,
                               0.9238795, 0.9142098, 0.9039893, 0.8932243, 0.8819213, 0.8700870, 0.8577286, 0.8448536,
                               0.8314696, 0.8175848, 0.8032075, 0.7883464, 0.7730105, 0.7572088, 0.7409511, 0.7242471,
                               0.7071068, 0.6895405, 0.6715590, 0.6531728, 0.6343933, 0.6152316, 0.5956993, 0.5758082,
                               0.5555702, 0.5349976, 0.5141027, 0.4928982, 0.4713967, 0.4496113, 0.4275551, 0.4052413,
                               0.3826834, 0.3598950, 0.3368899, 0.3136817, 0.2902847, 0.2667128, 0.2429802, 0.2191012,
                               0.1950903, 0.1709619, 0.1467305, 0.1224107, 0.0980171, 0.0735646, 0.0490677, 0.0245412,

                               0.0000000, -0.0245412, -0.0490677, -0.0735646, -0.0980171, -0.1224107, -0.1467305, -0.1709619,
                               -0.1950903, -0.2191012, -0.2429802, -0.2667128, -0.2902847, -0.3136817, -0.3368899, -0.3598950,
                               -0.3826834, -0.4052413, -0.4275551, -0.4496113, -0.4713967, -0.4928982, -0.5141027, -0.5349976,
                               -0.5555702, -0.5758082, -0.5956993, -0.6152316, -0.6343933, -0.6531728, -0.6715590, -0.6895405,
                               -0.7071068, -0.7242471, -0.7409511, -0.7572088, -0.7730105, -0.7883464, -0.8032075, -0.8175848,
                               -0.8314696, -0.8448536, -0.8577286, -0.8700870, -0.8819213, -0.8932243, -0.9039893, -0.9142098,
                               -0.9238795, -0.9329928, -0.9415441, -0.9495282, -0.9569403, -0.9637761, -0.9700313, -0.9757021,
                               -0.9807853, -0.9852776, -0.9891765, -0.9924795, -0.9951847, -0.9972905, -0.9987955, -0.9996988,

                               -1.0000000, -0.9996988, -0.9987955, -0.9972905, -0.9951847, -0.9924795, -0.9891765, -0.9852776,
                               -0.9807853, -0.9757021, -0.9700313, -0.9637761, -0.9569403, -0.9495282, -0.9415441, -0.9329928,
                               -0.9238795, -0.9142098, -0.9039893, -0.8932243, -0.8819213, -0.8700870, -0.8577286, -0.8448536,
                               -0.8314696, -0.8175848, -0.8032075, -0.7883464, -0.7730105, -0.7572088, -0.7409511, -0.7242471,
                               -0.7071068, -0.6895405, -0.6715590, -0.6531728, -0.6343933, -0.6152316, -0.5956993, -0.5758082,
                               -0.5555702, -0.5349976, -0.5141027, -0.4928982, -0.4713967, -0.4496113, -0.4275551, -0.4052413,
                               -0.3826834, -0.3598950, -0.3368899, -0.3136817, -0.2902847, -0.2667128, -0.2429802, -0.2191012,
                               -0.1950903, -0.1709619, -0.1467305, -0.1224107, -0.0980171, -0.0735646, -0.0490677, -0.0245412,

                               0.0000000, 0.0245412, 0.0490677, 0.0735646, 0.0980171, 0.1224107, 0.1467305, 0.1709619,
                               0.1950903, 0.2191012, 0.2429802, 0.2667128, 0.2902847, 0.3136817, 0.3368899, 0.3598950,
                               0.3826834, 0.4052413, 0.4275551, 0.4496113, 0.4713967, 0.4928982, 0.5141027, 0.5349976,
                               0.5555702, 0.5758082, 0.5956993, 0.6152316, 0.6343933, 0.6531728, 0.6715590, 0.6895405,
                               0.7071068, 0.7242471, 0.7409511, 0.7572088, 0.7730105, 0.7883464, 0.8032075, 0.8175848,
                               0.8314696, 0.8448536, 0.8577286, 0.8700870, 0.8819213, 0.8932243, 0.9039893, 0.9142098,
                               0.9238795, 0.9329928, 0.9415441, 0.9495282, 0.9569403, 0.9637761, 0.9700313, 0.9757021,
                               0.9807853, 0.9852776, 0.9891765, 0.9924795, 0.9951847, 0.9972905, 0.9987955, 0.9996988
                            };

Uint16 sin_table_PRD[256];

Uint16 adc_data = 6311;
Uint16 sin_index_A, sin_index_B, sin_index_C;

int flagA, flagB, flagC = 0;
float temp2=23.33;
float temp3=1.55;
float temp4;
//
// Defines that configure the period for each timer
//
/*
#define EPWM1_TIMER_TBPRD  2246  // Period register
#define EPWM1_MAX_CMPA     2246
#define EPWM1_MIN_CMPA     0
#define EPWM1_MAX_CMPB     2246
#define EPWM1_MIN_CMPB     0
*/

//
// Defines to keep track of which way the compare value is moving
//
#define EPWM_CMP_UP   1
#define EPWM_CMP_DOWN 0
#define EPWM_dead_band 45    //45 = 500ns

//extern Uint16 Cla1funcsLoadStart;
//extern Uint16 Cla1funcsLoadEnd;
//extern Uint16 Cla1funcsRunStart;
//extern Uint16 Cla1funcsLoadSize;

//
// Main
//
void main(void)
{
    //
    // Step 1. Initialize System Control:
    // PLL, WatchDog, enable Peripheral Clocks
    // This example function is found in the F2806x_SysCtrl.c file.
    //
    InitSysCtrl();

    //
    // Step 2. Initalize GPIO:
    // This example function is found in the F2806x_Gpio.c file and
    // illustrates how to set the GPIO to it's default state.
    //
    //InitGpio_QZ();

    //
    // For this case just init GPIO pins for ePWM1, ePWM2, ePWM3
    // These functions are in the F2806x_EPwm.c file


    memcpy(&RamfuncsRunStart, &RamfuncsLoadStart, (Uint32)&RamfuncsLoadSize);

    //
    // Call Flash Initialization to setup flash waitstates
    // This function must reside in RAM
    //

    InitFlash();

    InitEPwm1Gpio();
    InitEPwm2Gpio();
    InitEPwm3Gpio();
    InitEPwm4Gpio();

    //
    // Step 3. Clear all interrupts and initialize PIE vector table:
    // Disable CPU interrupts
    //
    DINT;

    //
    // Initialize the PIE control registers to their default state.
    // The default state is all PIE interrupts disabled and flags
    // are cleared.
    // This function is found in the F2806x_PieCtrl.c file.
    //
    InitPieCtrl();

    //
    // Disable CPU interrupts and clear all CPU interrupt flags
    //
    IER = 0x0000;
    IFR = 0x0000;

    //
    // Initialize the PIE vector table with pointers to the shell Interrupt
    // Service Routines (ISR).
    // This will populate the entire table, even if the interrupt
    // is not used in this example.  This is useful for debug purposes.
    // The shell ISR routines are found in F2806x_DefaultIsr.c.
    // This function is found in F2806x_PieVect.c.
    //
    InitPieVectTable();

    //
    // Interrupts that are used in this example are re-mapped to
    // ISR functions found within this file.
    //
    EALLOW;  // This is needed to write to EALLOW protected PieVectTable.EPWM1_INT
    PieVectTable.EPWM1_INT = &epwm1_isr;
//    PieVectTable.EPWM2_INT = &epwm2_isr;
//   PieVectTable.EPWM3_INT = &epwm3_isr;
//   PieVectTable.EPWM4_INT = &epwm4_isr;
    EDIS;    // This is needed to disable write to EALLOW protected registers

    //
    // Step 4. Initialize all the Device Peripherals:
    // This function is found in F2806x_InitPeripherals.c
    //
    // InitPeripherals();  // Not required for this example

    //
    // For this example, only initialize the ePWM
    //
    EALLOW;
    SysCtrlRegs.PCLKCR0.bit.TBCLKSYNC = 0;
    EDIS;

    Uint16 NN, KN_Tsw;
    Uint32 i=0;

    float TB_PRD=1123.0f;

    float freq, f_carrier, tempf, tempf2;     //
    float M = 50.0f;     //modulation index in percentage, 0-100:0%-100%
    freq = 70.3001;   // Fundamental frequency
    if (freq<70.3)
    {
        f_carrier = freq*256.0f/1000.0f;
    }
    else
    {
        f_carrier = 18.0f;       // carrier frequency, switching frequency kHz
    }
    TB_PRD = (45000.0/f_carrier);

    KN_Tsw = (Uint16) (1000*f_carrier/freq);   // cycles of switching cycle within a fundamental cycle


    update_freq(&epwm1_info, KN_Tsw);    // CHANGE freq


    for (i=0;i<N;i++)
        {
           tempf =  (1.0 + M*sin_table256[i]/100.0)/2.0;
           tempf *= TB_PRD;
           sin_table_PRD[i]= (Uint16) tempf;
//        sin_table_PRD[i]= 2;       /////////reduce from 50 until PWM disrupt ///////////TB_PRD=100///////////////
        }


    InitEPwm1Example((Uint16) (TB_PRD));
    InitEPwm2Example((Uint16) (TB_PRD));
    InitEPwm3Example((Uint16) (TB_PRD));
    InitEPwm4Example((Uint16) (TB_PRD));

    EALLOW;
    SysCtrlRegs.PCLKCR0.bit.TBCLKSYNC = 1;
    EDIS;

    //
    // Step 5. User specific code, enable interrupts
    //

    //
    // Enable CPU INT3 which is connected to EPWM1-3 INT
    //
    IER |= M_INT3;

    //
    // Enable EPWM INTn in the PIE: Group 3 interrupt 1-3
    //
    PieCtrlRegs.PIEIER3.bit.INTx1 = 1;
//   PieCtrlRegs.PIEIER3.bit.INTx2 = 1;
//   PieCtrlRegs.PIEIER3.bit.INTx3 = 1;
//   PieCtrlRegs.PIEIER3.bit.INTx4 = 1;

    //
    // Enable global Interrupts and higher priority real-time debug events
    //
    EINT;   // Enable Global interrupt INTM
    ERTM;   // Enable Global realtime interrupt DBGM

    //
    // Step 6. IDLE loop. Just sit and loop forever (optional)
    //
    for(;;)
    {
        __asm("          NOP");
    }
}

//
// epwm1_isr -
//
__interrupt void
epwm1_isr(void)
{
    //
    // Update the CMPA and CMPB values
    update_compare(&epwm1_info);

    // Clear INT flag for this timer
    EPwm1Regs.ETCLR.bit.INT = 1;

    // Acknowledge this interrupt to receive more interrupts from group 3
    //
    PieCtrlRegs.PIEACK.all = PIEACK_GROUP3;
}

//
//
// InitEPwm1Example -
//
void
InitEPwm1Example(Uint16 TB_peroid)
{
    //
    // Setup TBCLK
    //
    EPwm1Regs.TBPRD = TB_peroid;     // Set timer period 801 TBCLKs
    EPwm1Regs.TBPHS.half.TBPHS = 0x0000;     // Phase is 0
    EPwm1Regs.TBCTR = 0x0000;                // Clear counter

    //
    // Set Compare values
    //
    EPwm1Regs.CMPA.half.CMPA = 1123;     // Set compare A value
    EPwm1Regs.CMPB = 1123;               // Set Compare B value

    //
    // Setup counter mode
    //
    EPwm1Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN; // Count up & down
    EPwm1Regs.TBCTL.bit.PHSEN = TB_DISABLE;        // Disable phase loading
    EPwm1Regs.TBCTL.bit.SYNCOSEL = TB_CTR_ZERO; // Sync down-stream module
    EPwm1Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1;       // Clock ratio to SYSCLKOUT
    EPwm1Regs.TBCTL.bit.CLKDIV = TB_DIV1;

    //
    // Setup shadowing
    //
    EPwm1Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;
    EPwm1Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;
    EPwm1Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO;  // Load on Zero
    EPwm1Regs.CMPCTL.bit.LOADBMODE = CC_CTR_ZERO;

    //
    // Set actions
    //   duty = (CMPA*2+1)/(TBPRD*2+1)
    EPwm1Regs.AQCTLA.bit.CAU = AQ_CLEAR;    // Set PWM1A on event A, up count
    EPwm1Regs.AQCTLA.bit.CAD = AQ_SET;  // Clear PWM1A on event A, down count

    //EPwm1Regs.AQCTLB.bit.CBU = AQ_SET;    // Set PWM1B on event B, up count
    //EPwm1Regs.AQCTLB.bit.CBD = AQ_CLEAR;  // Clear PWM1B on event B, down count

    //
    // set deadband
/*    EPwm1Regs.DBCTL.bit.OUT_MODE = DB_FULL_ENABLE;
    EPwm1Regs.DBCTL.bit.POLSEL = DB_ACTV_HIC;
    EPwm1Regs.DBCTL.bit.IN_MODE = DBA_ALL;     // A for all
    EPwm1Regs.DBRED = 45;             //dead band  = 45*11.1ns = 500ns, OFFtime = 170ns
    EPwm1Regs.DBFED = 45;             //dead band  = 45*11.1ns = 500ns, ONtime  = 200ns
*/
    //
    // Interrupt where we will change the Compare Values
    //
    EPwm1Regs.ETSEL.bit.INTSEL = ET_CTR_ZERO;      // Select INT on Zero event
    EPwm1Regs.ETSEL.bit.INTEN = 1;                 // Enable INT
    EPwm1Regs.ETPS.bit.INTPRD = ET_1ST;            // Generate INT on 1ST event


    epwm1_info.EPwmSampleCount = 0;         // 240 degree phase
//    epwm1_info.NK_Tsw = 256;
//   epwm1_info.SIN_i_Count = 0;

    //
    // Set the pointer to the ePWM module
    //
    epwm1_info.EPwmRegHandle = &EPwm1Regs;
}

//
// InitEPwm2Example -
//
void
InitEPwm2Example(Uint16 TB_peroid)
{
    //
    // Setup TBCLK
    //
    EPwm2Regs.TBPRD = TB_peroid;         // Set timer period 801 TBCLKs
    EPwm2Regs.TBPHS.half.TBPHS = 0;         // Phase is 0
    EPwm2Regs.TBCTR = 0;                    // Clear counter

    //
    // Set Compare values
    //
    EPwm2Regs.CMPA.half.CMPA = 1123;     // Set compare A value
    EPwm2Regs.CMPB = 1123;               // Set Compare B value

    //
    // Setup counter mode
    //
    EPwm2Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN; // Count up
    EPwm2Regs.TBCTL.bit.PHSEN = TB_ENABLE;        // Enable phase loading
    EPwm2Regs.TBCTL.bit.SYNCOSEL =  TB_SYNC_IN; // sync flow-through
    EPwm2Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1;       // Clock ratio to SYSCLKOUT
    EPwm2Regs.TBCTL.bit.CLKDIV = TB_DIV1;

    //
    // Setup shadowing
    //
    EPwm2Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;
    EPwm2Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;
    EPwm2Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO;  // Load on Zero
    EPwm2Regs.CMPCTL.bit.LOADBMODE = CC_CTR_ZERO;

    //
    // Set actions
    //
    EPwm2Regs.AQCTLA.bit.CAU = AQ_CLEAR;    // Set PWM1A on event A, up count
    EPwm2Regs.AQCTLA.bit.CAD = AQ_SET;  // Clear PWM1A on event A, down count

    //EPwm2Regs.AQCTLB.bit.CBU = AQ_SET;    // Set PWM1B on event B, up count
    //EPwm2Regs.AQCTLB.bit.CBD = AQ_CLEAR;  // Clear PWM1B on event B, down count
    //
    //
    // set deadband
    EPwm2Regs.DBCTL.bit.OUT_MODE = DB_FULL_ENABLE;
    EPwm2Regs.DBCTL.bit.POLSEL = DB_ACTV_HIC;
    EPwm2Regs.DBCTL.bit.IN_MODE = DBA_ALL;     // A for all
    EPwm2Regs.DBRED = EPWM_dead_band;             //dead band  = 45*11.1ns = 500ns, OFFtime = 170ns
    EPwm2Regs.DBFED = EPWM_dead_band;             //dead band  = 45*11.1ns = 500ns, ONtime  = 200ns

    // Interrupt where we will change the Compare Values
    //
    EPwm2Regs.ETSEL.bit.INTSEL = ET_CTR_ZERO;     // Select INT on Zero event
    EPwm2Regs.ETSEL.bit.INTEN = 0;                // Enable INT
    EPwm2Regs.ETPS.bit.INTPRD = ET_1ST;           // Generate INT on 2Nd event


//   epwm2_info.NK_Tsw = 256;
//   epwm2_info.SIN_i_Count = 85;
    //
    // Set the pointer to the ePWM module
    //

}

//
// InitEPwm3Example -
//
void
InitEPwm3Example(Uint16 TB_peroid)
{
    //
    // Setup TBCLK
    //
    EPwm3Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN;// Count up/down
    EPwm3Regs.TBPRD = TB_peroid;          // Set timer period
    EPwm3Regs.TBPHS.half.TBPHS = 0;         // Phase is 0
    EPwm3Regs.TBCTL.bit.PHSEN = TB_ENABLE;       // Enable phase loading
    EPwm3Regs.TBCTL.bit.SYNCOSEL =  TB_SYNC_IN; // sync flow-through
    EPwm3Regs.TBCTR = 0x0000;                     // Clear counter
    EPwm3Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1;      // Clock ratio to SYSCLKOUT
    EPwm3Regs.TBCTL.bit.CLKDIV = TB_DIV1;

    //
    // Setup shadow register load on ZERO
    //
    EPwm3Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;
    EPwm3Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;
    EPwm3Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO;
    EPwm3Regs.CMPCTL.bit.LOADBMODE = CC_CTR_ZERO;

    //
    // Set Compare values
    //
    EPwm3Regs.CMPA.half.CMPA = 1123;    // Set compare A value
    EPwm3Regs.CMPB = 1123;              // Set Compare B value

    //
    // Set Actions
    //
    EPwm3Regs.AQCTLA.bit.CAU = AQ_CLEAR;    // Set PWM1A on event A, up count
    EPwm3Regs.AQCTLA.bit.CAD = AQ_SET;  // Clear PWM1A on event A, down count

    //EPwm3Regs.AQCTLB.bit.CBU = AQ_SET;    // Set PWM1B on event B, up count
    //EPwm3Regs.AQCTLB.bit.CBD = AQ_CLEAR;  // Clear PWM1B on event B, down count

    //
    // set deadband
    EPwm3Regs.DBCTL.bit.OUT_MODE = DB_FULL_ENABLE;
    EPwm3Regs.DBCTL.bit.POLSEL = DB_ACTV_HIC;
    EPwm3Regs.DBCTL.bit.IN_MODE = DBA_ALL;     // A for all
    EPwm3Regs.DBRED = EPWM_dead_band;             //dead band  = 45*11.1ns = 500ns, OFFtime = 170ns
    EPwm3Regs.DBFED = EPWM_dead_band;             //dead band  = 45*11.1ns = 500ns, ONtime  = 200ns

    //
    // Interrupt where we will change the Compare Values
    //
    EPwm3Regs.ETSEL.bit.INTSEL = ET_CTR_ZERO;     // Select INT on Zero event
    EPwm3Regs.ETSEL.bit.INTEN = 0;                // Enable INT
    EPwm3Regs.ETPS.bit.INTPRD = ET_1ST;           // Generate INT on 3rd event


//   epwm3_info.NK_Tsw = 256;
//   epwm3_info.SIN_i_Count = 170;
    //
    // Set the pointer to the ePWM module
    //
}

//
// InitEPwm3Example -
//
void
InitEPwm4Example(Uint16 TB_peroid)
{
    //
    // Setup TBCLK
    //
    EPwm4Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN;// Count up
    EPwm4Regs.TBPRD = TB_peroid;          // Set timer period  = 1s
    EPwm4Regs.TBPHS.half.TBPHS = 0;         // Phase is 0
    EPwm4Regs.TBCTL.bit.PHSEN = TB_ENABLE;       // enable phase loading
    EPwm4Regs.TBCTL.bit.SYNCOSEL =  TB_SYNC_IN; // sync flow-through
    EPwm4Regs.TBCTR = 0x0000;                     // Clear counter
    EPwm4Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1;      // Clock ratio to SYSCLKOUT
    EPwm4Regs.TBCTL.bit.CLKDIV = TB_DIV1;         //

    //
    // Setup shadow register load on ZERO
    //
    EPwm4Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;
    EPwm4Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;
    EPwm4Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO;
    EPwm4Regs.CMPCTL.bit.LOADBMODE = CC_CTR_ZERO;

    //
    // Set Compare values
    //
    EPwm4Regs.CMPA.half.CMPA = sin_table_PRD[1];    // Set compare A value
    EPwm4Regs.CMPB = sin_table_PRD[1];              // Set Compare B value

    //
    // Set Actions
    //
    EPwm4Regs.AQCTLA.bit.CAU = AQ_CLEAR;    // Set PWM1A on event A, up count
    EPwm4Regs.AQCTLA.bit.CAD = AQ_SET;  // Clear PWM1A on event A, down count

    // set deadband
    EPwm4Regs.DBCTL.bit.OUT_MODE = DB_FULL_ENABLE;
    EPwm4Regs.DBCTL.bit.POLSEL = DB_ACTV_HIC;
    EPwm4Regs.DBCTL.bit.IN_MODE = DBA_ALL;     // A for all
    EPwm4Regs.DBRED = EPWM_dead_band;             //dead band  = 45*11.1ns = 500ns, OFFtime = 170ns
    EPwm4Regs.DBFED = EPWM_dead_band;             //dead band  = 45*11.1ns = 500ns, ONtime  = 200ns

    //
    // Interrupt where we will change the Compare Values
    //
    EPwm4Regs.ETSEL.bit.INTSEL = ET_CTR_ZERO;     // Select INT on Zero event
    EPwm4Regs.ETSEL.bit.INTEN = 0;                // disable INT
    //EPwm4Regs.ETPS.bit.INTPRD = ET_1ST;           // Generate INT on 1st event

}
//
// update_compare -
//

void
update_compare(EPWM_INFO *epwm_info)
{
    epwm_info->SIN_i_Count = epwm_info->EPwmSampleCount<<8;
    epwm_info->SIN_i_Count /= epwm_info->NK_Tsw;

    sin_index_A = epwm_info->SIN_i_Count;
    sin_index_B = epwm_info->NK_Tsw_1_third + epwm_info->SIN_i_Count;
    sin_index_B = sin_index_B & 0xFF;

    sin_index_C = epwm_info->NK_Tsw_2_third + epwm_info->SIN_i_Count;
    sin_index_C = sin_index_C & 0xFF;

    EPwm1Regs.CMPA.half.CMPA = sin_table_PRD[sin_index_A];
    EPwm2Regs.CMPA.half.CMPA = sin_table_PRD[sin_index_A];
    EPwm3Regs.CMPA.half.CMPA = sin_table_PRD[sin_index_B];
    EPwm4Regs.CMPA.half.CMPA = sin_table_PRD[sin_index_C];
    //epwm_info->EPwmRegHandle->CMPA.half.CMPA = sin_table_PRD[1];    // Set compare A value
    //epwm_info->EPwmRegHandle->CMPB = sin_table[epwm_info->EPwmSampleCount];    // Set compare B value

    // Every NK_Tsw'th interrupt, start a new cycle,   EPwmSampleCount=[0 ~ NK_Tsw-1]
    if(epwm_info->EPwmSampleCount == epwm_info->NK_Tsw-1)
    {
        epwm_info->EPwmSampleCount = 0;
        //GpioDataRegs.GPATOGGLE.bit.GPIO16 = 1;
        //GpioDataRegs.GPATOGGLE.bit.GPIO17 = 1;
    }
    else
    {
        epwm_info->EPwmSampleCount++;
    }
    return;
}



void
update_freq(EPWM_INFO *info1, Uint16 n)
{
    info1->NK_Tsw = n;
    info1->NK_Tsw_1_third = n/3;
    info1->NK_Tsw_2_third = 2*n/3;
    info1->EPwmSampleCount = info1->NK_Tsw_2_third;     // 240 degree phase

    return;
}

//
// End of File
//

