// TI File $Revision: /main/3 $
// Check in $Date: October 6, 2010   14:42:42 $
//###########################################################################
//
// FILE:    Example_2802xFlash.c
//
// TITLE:   DSP2802x EPwm Timer Interrupt From Flash Example.
//
// ASSUMPTIONS:
//
//    This program requires the DSP2802x header files.
//
//    As supplied, this project is configured for "boot to FLASH"
//    operation.  The 2802x Boot Mode table is shown below.
//    For information on configuring the boot mode of an eZdsp,
//    please refer to the documentation included with the eZdsp,
//
//    $Boot_Table
//    While an emulator is connected to your device, the TRSTn pin = 1,
//    which sets the device into EMU_BOOT boot mode. In this mode, the
//    peripheral boot modes are as follows:
//
//      Boot Mode:   EMU_KEY        EMU_BMODE
//                   (0xD00)         (0xD01)
//      ---------------------------------------
//      Wait         !=0x55AA        X
//      I/O          0x55AA          0x0000
//      SCI          0x55AA          0x0001
//      Wait         0x55AA          0x0002
//      Get_Mode     0x55AA          0x0003
//      SPI          0x55AA          0x0004
//      I2C          0x55AA          0x0005
//      OTP          0x55AA          0x0006
//      Wait         0x55AA          0x0007
//      Wait         0x55AA          0x0008
//      SARAM        0x55AA          0x000A
//      Flash        0x55AA          0x000B   <-- "Boot to Flash"
//      Wait         0x55AA          Other
//
//   Write EMU_KEY to 0xD00 and EMU_BMODE to 0xD01 via the debugger
//   according to the Boot Mode Table above. Build/Load project,
//   Reset the device, and Run example
//
//   $End_Boot_Table
//
//    The program must first be compiled and then programmed into the
//    flash.
//
//
// DESCRIPTION:
//
//    This example runs the EPwm interrupt example from flash.
//
//    1) Build the project
//    2) Flash the .out file into the device.
//    3) Set the hardware jumpers to boot to Flash
//    4) Use the included GEL file to load the project, symbols
//       defined within the project and the variables into the watch
//       window.
//
//    Steps that were taken to convert the EPwm example from RAM
//    to Flash execution:
//
//    - Change the linker cmd file to reflect the flash memory map.
//    - Make sure any initialized sections are mapped to Flash.
//      In SDFlash utility this can be checked by the View->Coff/Hex
//      status utility. Any section marked as "load" should be
//      allocated to Flash.
//    - Make sure there is a branch instruction from the entry to Flash
//      at 0x3F7FF6 to the beginning of code execution. This example
//      uses the DSP0x_CodeStartBranch.asm file to accomplish this.
//    - Set boot mode Jumpers to "boot to Flash"
//    - For best performance from the flash, modify the waitstates
//      and enable the flash pipeline as shown in this example.
//      Note: any code that manipulates the flash waitstate and pipeline
//      control must be run from RAM. Thus these functions are located
//      in their own memory section called ramfuncs.
//
//
//    EPwm1 Interrupt will run from RAM and puts the flash into sleep mode
//    EPwm2 Interrupt will run from RAM and puts the flash into standby mode
//    EPwm3 Interrupt will run from FLASH
//
//    As supplied:
//
//    All timers have the same period
//    The timers are started sync'ed
//    An interrupt is taken on a zero event for each EPwm timer
//
//       EPwm1: takes an interrupt every event
//       EPwm2: takes an interrupt every 2nd event
//       EPwm3: takes an interrupt every 3rd event
//
//    Thus the Interrupt count for EPwm1, EPwm4-EPwm6 should be equal
//    The interrupt count for EPwm2 should be about half that of EPwm1
//    and the interrupt count for EPwm3 should be about 1/3 that of EPwm1
//
//          Watch Variables:
//                 EPwm1TimerIntCount
//                 EPwm2TimerIntCount
//                 EPwm3TimerIntCount
//
//                 Toggle GPIO34 while in the background loop.
//
//###########################################################################
// $TI Release: 2802x C/C++ Header Files and Peripheral Examples V1.29 $
// $Release Date: January 11, 2011 $
//###########################################################################

#include "DSP28x_Project.h"     // Device Headerfile and Examples Include File

// Configure which EPwm timer interrupts are enabled at the PIE level:
// 1 = enabled,  0 = disabled
#define PWM1_INT_ENABLE  1
//#define PWM2_INT_ENABLE  1
//#define PWM3_INT_ENABLE  1

// Global variables used in this example:
   Uint16 LoopCount;
   Uint16 ConversionCount;
   Uint16 Voltage1[10];
   Uint16 Voltage2[10];
   float V1,V2,Va,Vb,Vc,Vd,sum = 0; //anup
   #define period 10000;
   #define duty_cycle 6000;



// Configure the period for each timer
#define EPWM1_TIMER_TBPRD  300  // Period register
//#define PWM2_TIMER_TBPRD   0x1FFF
//#define PWM3_TIMER_TBPRD   0x1FFF

// Make this long enough so that we can see an LED toggle
//#define DELAY 1000000L

// Functions that will be run from RAM need to be assigned to
// a different section.  This section will then be mapped using
// the linker cmd file.
#pragma CODE_SECTION(EPwm1_timer_isr, "ramfuncs");
//#pragma CODE_SECTION(EPwm2_timer_isr, "ramfuncs");

// Prototype statements for functions found within this file.
interrupt void EPwm1_timer_isr(void);
interrupt void adc_isr(void);
void ConfigADC(void);

//interrupt void EPwm2_timer_isr(void);
//interrupt void EPwm3_timer_isr(void);
void InitEPwmTimer(void);

// Global variables used in this example
Uint32  EPwm1TimerIntCount;
//Uint32  EPwm2TimerIntCount;
//Uint32  EPwm3TimerIntCount;


// These are defined by the linker (see F2808.cmd)
extern Uint16 RamfuncsLoadStart;
extern Uint16 RamfuncsLoadEnd;
extern Uint16 RamfuncsRunStart;

void main(void)
{

// Step 1. Initialize System Control:
// PLL, WatchDog, enable Peripheral Clocks
// This example function is found in the DSP2802x_SysCtrl.c file.
   InitSysCtrl();

// Step 2. Initalize GPIO:
// This example function is found in the DSP2802x_Gpio.c file and
// illustrates how to set the GPIO to it's default state.
// InitGpio();  // Skipped for this example
   InitEPwm1Gpio();
  // InitEPwm2Gpio();
// Step 3. Clear all interrupts and initialize PIE vector table:
// Disable CPU interrupts
   DINT;

// Initialize the PIE control registers to their default state.
// The default state is all PIE interrupts disabled and flags
// are cleared.
// This function is found in the DSP2802x_PieCtrl.c file.
   InitPieCtrl();

// Disable CPU interrupts and clear all CPU interrupt flags:
   IER = 0x0000;
   IFR = 0x0000;

// Initialize the PIE vector table with pointers to the shell Interrupt
// Service Routines (ISR).
// This will populate the entire table, even if the interrupt
// is not used in this example.  This is useful for debug purposes.
// The shell ISR routines are found in DSP2802x_DefaultIsr.c.
// This function is found in DSP2802x_PieVect.c.
   InitPieVectTable();

// Interrupts that are used in this example are re-mapped to
// ISR functions found within this file.
   EALLOW;  // This is needed to write to EALLOW protected registers
   PieVectTable.EPWM1_INT = &EPwm1_timer_isr;
   PieVectTable.ADCINT1 = &adc_isr;
   EDIS;    // This is needed to disable write to EALLOW protected registers

// Step 4. Initialize all the Device Peripherals:
// This function is found in DSP2802x_InitPeripherals.c
// InitPeripherals();  // Not required for this example

   InitAdc();

   // For this example, only initialize the ePWM

   EALLOW;
   SysCtrlRegs.PCLKCR0.bit.TBCLKSYNC = 0;
   EDIS;

   InitEPwmTimer();    // For this example, only initialize the EPwm Timers

   EALLOW;
   SysCtrlRegs.PCLKCR0.bit.TBCLKSYNC = 1;
   EDIS;

// Step 5. User specific code, enable interrupts:

// Copy time critical code and Flash setup code to RAM
// This includes the following ISR functions: EPwm1_timer_isr(), EPwm2_timer_isr()
// EPwm3_timer_isr and and InitFlash();
// The  RamfuncsLoadStart, RamfuncsLoadEnd, and RamfuncsRunStart
// symbols are created by the linker. Refer to the F2808.cmd file.
   MemCopy(&RamfuncsLoadStart, &RamfuncsLoadEnd, &RamfuncsRunStart);

// Call Flash Initialization to setup flash waitstates
// This function must reside in RAM
   InitFlash();

// Initalize counters:
   EPwm1TimerIntCount = 0;
 //  EPwm2TimerIntCount = 0;
//   EPwm3TimerIntCount = 0;

// Enable CPU INT3 which is connected to EPwm1-3 INT:
   IER |= M_INT3;

// Enable EPwm INTn in the PIE: Group 3 interrupt 1-3
   PieCtrlRegs.PIEIER3.bit.INTx1 = PWM1_INT_ENABLE;
   PieCtrlRegs.PIEIER1.bit.INTx1 = 1;	// Enable INT 1.1 in the PIE
   IER |= M_INT1;
 //  PieCtrlRegs.PIEIER3.bit.INTx2 = PWM2_INT_ENABLE;
 //  PieCtrlRegs.PIEIER3.bit.INTx3 = PWM3_INT_ENABLE;

// Enable global Interrupts and higher priority real-time debug events:
   EINT;   // Enable Global interrupt INTM
   ERTM;   // Enable Global real time interrupt DBGM


   LoopCount = 0;
   ConversionCount = 0;

   // Configure ADC

    ConfigADC();


// Step 6. IDLE loop. Just sit and loop forever (optional):
   EALLOW;
   GpioCtrlRegs.GPBMUX1.bit.GPIO34 = 0;
   GpioCtrlRegs.GPBDIR.bit.GPIO34 = 1;
   EDIS;

   for(;;)
   {

	   LoopCount++;
   }

}


void  InitEPwmTimer()
{

	// Setup TBCLK
	   EPwm1Regs.TBPRD = EPWM1_TIMER_TBPRD;           // Set timer period 801 TBCLKs // 2000  Total Time period // 15kHz
	   EPwm1Regs.TBPHS.half.TBPHS = 0x0000;           // Phase is 0
	   EPwm1Regs.TBCTR = 0x0000;                      // Clear counter

	   // Set Compare values
	   EPwm1Regs.CMPA.half.CMPA = 150;     //anup

	   // Setup counter mode
	   EPwm1Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN; // Count up
	   EPwm1Regs.TBCTL.bit.PHSEN = TB_DISABLE;        // Disable phase loading
	   EPwm1Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1;       // Clock ratio to SYSCLKOUT
	   EPwm1Regs.TBCTL.bit.CLKDIV = TB_DIV1;

	   // Setup shadowing
	   EPwm1Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;
	   EPwm1Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO;  // Load on Zero


	   // Set actions
	   EPwm1Regs.AQCTLA.bit.CAU = AQ_SET;             // Set PWM1A on event A, up count
	   EPwm1Regs.AQCTLA.bit.CAD = AQ_CLEAR;           // Clear PWM1A on event A, down count


}
// This ISR MUST be executed from RAM as it will put the Flash into Sleep
// Interrupt routines uses in this example:
interrupt void EPwm1_timer_isr(void)
{

   // Put the Flash to sleep
   FlashRegs.FPWR.bit.PWR = FLASH_SLEEP;

   EPwm1TimerIntCount++;

   // Clear INT flag for this timer
   EPwm1Regs.ETCLR.bit.INT = 1;

   // Acknowledge this interrupt to receive more interrupts from group 3
   PieCtrlRegs.PIEACK.all = PIEACK_GROUP3;
}

interrupt void  adc_isr(void)
{

  Voltage1[ConversionCount] = AdcResult.ADCRESULT1;  //discard ADCRESULT0 as part of the workaround to the 1st sample errata for rev0
  Voltage2[ConversionCount] = AdcResult.ADCRESULT2;
  sum += Voltage1[ConversionCount];
  // If 25 conversions have been logged, start over
  if(ConversionCount == 29)
  {
	  V1 = sum/ConversionCount;
   //   V1 = 3999;
	  Va = (V1 * 3.3)/(4096) ;
	  if (V1 >= 4000)
	 	  {
	 		  EPwm1Regs.AQCTLA.bit.CAU = AQ_CLEAR;             // Set PWM1A on event A, up count
	 		  EPwm1Regs.AQCTLA.bit.CAD = AQ_CLEAR;           // Clear PWM1A on event A, down count
	 		  sum=0;
	 	  }
	  sum=0;
      ConversionCount = 0;
  }


  else ConversionCount++;

  AdcRegs.ADCINTFLGCLR.bit.ADCINT1 = 1;		//Clear ADCINT1 flag reinitialize for next SOC
  PieCtrlRegs.PIEACK.all = PIEACK_GROUP1;   // Acknowledge interrupt to PIE
  return;
}

void ConfigADC()
{
//Note: Channel ADCINA4  will be double sampled to workaround the ADC 1st sample issue for rev0 silicon errata

	EALLOW;
		AdcRegs.ADCCTL1.bit.INTPULSEPOS	= 1;	//ADCINT1 trips after AdcResults latch
		AdcRegs.INTSEL1N2.bit.INT1E     = 1;	//Enabled ADCINT1
		AdcRegs.INTSEL1N2.bit.INT1CONT  = 0;	//Disable ADCINT1 Continuous mode
		AdcRegs.INTSEL1N2.bit.INT1SEL	= 2;	//setup EOC2 to trigger ADCINT1 to fire
		AdcRegs.ADCSOC0CTL.bit.CHSEL 	= 4;	//set SOC0 channel select to ADCINA4
		AdcRegs.ADCSOC1CTL.bit.CHSEL 	= 4;	//set SOC1 channel select to ADCINA4
		AdcRegs.ADCSOC2CTL.bit.CHSEL 	= 2;	//set SOC1 channel select to ADCINA2
		AdcRegs.ADCSOC0CTL.bit.TRIGSEL 	= 7;	//set SOC0 start trigger on EPWM1A, due to round-robin SOC0 converts first then SOC1
		AdcRegs.ADCSOC1CTL.bit.TRIGSEL 	= 7;	//set SOC1 start trigger on EPWM1A, due to round-robin SOC0 converts first then SOC1
		AdcRegs.ADCSOC2CTL.bit.TRIGSEL 	= 7;	//set SOC2 start trigger on EPWM1A, due to round-robin SOC0 converts first then SOC1, then SOC2
		AdcRegs.ADCSOC0CTL.bit.ACQPS 	= 9;	//set SOC0 S/H Window to 7 ADC Clock Cycles, (6 ACQPS plus 1)
		AdcRegs.ADCSOC1CTL.bit.ACQPS 	= 9;	//set SOC1 S/H Window to 7 ADC Clock Cycles, (6 ACQPS plus 1)
		AdcRegs.ADCSOC2CTL.bit.ACQPS 	= 9;	//set SOC2 S/H Window to 7 ADC Clock Cycles, (6 ACQPS plus 1)
		EDIS;

	// Assumes ePWM1 clock is already enabled in InitSysCtrl();
	   EPwm2Regs.ETSEL.bit.SOCAEN	= 1;		// Enable SOC on A group
	   EPwm2Regs.ETSEL.bit.SOCASEL	= 4;		// Select SOC from from CPMA on up count
	   EPwm2Regs.ETPS.bit.SOCAPRD 	= 1;		// Generate pulse on 1st event
	   EPwm2Regs.CMPA.half.CMPA	    = duty_cycle;	// Set compare A value
	   EPwm2Regs.TBPRD 				= period;	// Set period for ePWM1
	   EPwm2Regs.TBCTL.bit.CTRMODE 	= 0;		// count up and start

}



//===========================================================================
// No more.
//===========================================================================
