/*
 * Drivers.c
 *
 *  Created on: Aug 17, 2022
 *      Author: 210055498
 */
#include <F2837xD_device.h>
#include <F2837xD_EPwm_defines.h>
#include <F2837xD_epwm.h>
#include <F2837xD_Examples.h>
#include "Func.h"
#include "C1_Def.h"
#include "C1_Proto.h"
#include "C1_Struct.h"
#include "C1_Vars.h"

//
// InitPeripheralClocksCustom - Initializes the clocks for the peripherals.
//
// Note: In order to reduce power consumption, turn off the clocks to any
// peripheral that is not specified for your part-number or is not used in the
// application
//
void InitPeripheralClocksCustom(void)
{
    EALLOW;

    CpuSysRegs.PCLKCR0.bit.CLA1 = 0;
    CpuSysRegs.PCLKCR0.bit.DMA = 1;
    CpuSysRegs.PCLKCR0.bit.CPUTIMER0 = 0;
    CpuSysRegs.PCLKCR0.bit.CPUTIMER1 = 0;
    CpuSysRegs.PCLKCR0.bit.CPUTIMER2 = 0;

#ifdef CPU1
    CpuSysRegs.PCLKCR0.bit.HRPWM = 1;
#endif

    CpuSysRegs.PCLKCR0.bit.TBCLKSYNC = 1;

#ifdef CPU1
    CpuSysRegs.PCLKCR1.bit.EMIF1 = 1;
    CpuSysRegs.PCLKCR1.bit.EMIF2 = 0;
#endif

    CpuSysRegs.PCLKCR2.bit.EPWM1 = 1;
    CpuSysRegs.PCLKCR2.bit.EPWM2 = 1;
    CpuSysRegs.PCLKCR2.bit.EPWM3 = 1;
    CpuSysRegs.PCLKCR2.bit.EPWM4 = 1;
    CpuSysRegs.PCLKCR2.bit.EPWM5 = 0;
    CpuSysRegs.PCLKCR2.bit.EPWM6 = 0;
    CpuSysRegs.PCLKCR2.bit.EPWM7 = 0;
    CpuSysRegs.PCLKCR2.bit.EPWM8 = 0;
    CpuSysRegs.PCLKCR2.bit.EPWM9 = 01;
    CpuSysRegs.PCLKCR2.bit.EPWM10 = 0;
    CpuSysRegs.PCLKCR2.bit.EPWM11 = 0;
    CpuSysRegs.PCLKCR2.bit.EPWM12 = 0;

    CpuSysRegs.PCLKCR3.bit.ECAP1 = 1;
    CpuSysRegs.PCLKCR3.bit.ECAP2 = 1;
    CpuSysRegs.PCLKCR3.bit.ECAP3 = 1;
    CpuSysRegs.PCLKCR3.bit.ECAP4 = 0;
    CpuSysRegs.PCLKCR3.bit.ECAP5 = 0;
    CpuSysRegs.PCLKCR3.bit.ECAP6 = 0;

    CpuSysRegs.PCLKCR4.bit.EQEP1 = 0;
    CpuSysRegs.PCLKCR4.bit.EQEP2 = 0;
    CpuSysRegs.PCLKCR4.bit.EQEP3 = 0;

    CpuSysRegs.PCLKCR6.bit.SD1 = 1;
    CpuSysRegs.PCLKCR6.bit.SD2 = 1;

    CpuSysRegs.PCLKCR7.bit.SCI_A = 0;
    CpuSysRegs.PCLKCR7.bit.SCI_B = 0;
    CpuSysRegs.PCLKCR7.bit.SCI_C = 0;
    CpuSysRegs.PCLKCR7.bit.SCI_D = 0;

    CpuSysRegs.PCLKCR8.bit.SPI_A = 0;
    CpuSysRegs.PCLKCR8.bit.SPI_B = 1;
    CpuSysRegs.PCLKCR8.bit.SPI_C = 0;

    CpuSysRegs.PCLKCR9.bit.I2C_A = 0;
    CpuSysRegs.PCLKCR9.bit.I2C_B = 0;

    CpuSysRegs.PCLKCR10.bit.CAN_A = 1;
    CpuSysRegs.PCLKCR10.bit.CAN_B = 1;

    CpuSysRegs.PCLKCR11.bit.McBSP_A = 0;
    CpuSysRegs.PCLKCR11.bit.McBSP_B = 0;

#ifdef CPU1
    CpuSysRegs.PCLKCR11.bit.USB_A = 0;

    CpuSysRegs.PCLKCR12.bit.uPP_A = 1;
#endif

    CpuSysRegs.PCLKCR13.bit.ADC_A = 1;
    CpuSysRegs.PCLKCR13.bit.ADC_B = 1;
    CpuSysRegs.PCLKCR13.bit.ADC_C = 1;
    CpuSysRegs.PCLKCR13.bit.ADC_D = 1;

    CpuSysRegs.PCLKCR14.bit.CMPSS1 = 0;
    CpuSysRegs.PCLKCR14.bit.CMPSS2 = 0;
    CpuSysRegs.PCLKCR14.bit.CMPSS3 = 0;
    CpuSysRegs.PCLKCR14.bit.CMPSS4 = 0;
    CpuSysRegs.PCLKCR14.bit.CMPSS5 = 0;
    CpuSysRegs.PCLKCR14.bit.CMPSS6 = 0;
    CpuSysRegs.PCLKCR14.bit.CMPSS7 = 0;
    CpuSysRegs.PCLKCR14.bit.CMPSS8 = 0;

    CpuSysRegs.PCLKCR16.bit.DAC_A = 1;
    CpuSysRegs.PCLKCR16.bit.DAC_B = 1;
    CpuSysRegs.PCLKCR16.bit.DAC_C = 1;

    EDIS;
}

//***********************************************************************************************************************************************
//
//                                   Table 3-2. PIE Channel Mapping
//        INTx.1  INTx.2  INTx.3  INTx.4  INTx.5   INTx.6   INTx.7   INTx.8   INTx.9  INTx.10  INTx.11   INTx.12  NTx.13 INTx.14 INTx.15 INTx.16
//INT1.y  ADCA1   ADCB1   ADCC1   XINT1   XINT2    ADCD1    TIMER0   WAKE     -       -         -        -        IPC0   IPC1    IPC2    IPC3
//INT2.y  EPWM1TZ EPWM2TZ EPWM3TZ EPWM4TZ EPWM5TZ  EPWM6TZ  EPWM7TZ  EPWM8TZ  EPWM9TZ EPWM10TZ EPWM11TZ  EPWM12TZ -      -       -       -
//INT3.y  EPWM1   EPWM2   EPWM3   EPWM4   EPWM5    EPWM6    EPWM7    EPWM8    EPWM9   EPWM10   EPWM11    EPWM12   -      -       -       -
//INT4.y  ECAP1   ECAP2   ECAP3   ECAP4   ECAP5    ECAP6    -        -        -       -        -         -        -      -       -       -
//INT5.y  EQEP1   EQEP2   EQEP3   -       CLB1     CLB2     CLB3     CLB4     SD1     SD2      -         -        -      -       -       -
//INT6.y  SPIARX  SPIATX  SPIBRX  SPIBTX  MCBSPARX MCBSPATX MCBSPBRX MCBSPBTX SPICRX  SPICTX   -         -        -      -       -       -
//INT7.y  DMACH1  DMACH2  DMACH3  DMACH4  DMACH5   DMACH6   -        -        -       -        -         -        -      -       -       -
//INT8.y  I2CA   I2CAFIFO I2CB   I2CBFIFO SCIC_RX  SCIC_TX  SCID_RX  SCID_TX  -       -        -         -        -      -       UPPA    -
//INT9.y  SCIARX  SCIATX  SCIBRX  SCIBTX  CANA_0   CANA_1   CANB_0   CANB_1   -       -        -         -        -      -       USBA    -
//INT10.y ADCAEVT ADCA2   ADCA3   ADCA4   ADCBEVT  ADCB2    ADCB3    ADCB4   ADCCEVT  ADCC2    ADCC3     ADCC4    ADCDEVT ADCD2  ADCD3   ADCD4
//INT11.y CLA1_1  CLA1_2  CLA1_3  CLA1_4  CLA1_5   CLA1_6   CLA1_7   CLA1_8  -        -        -         -        -       -       -      -
//INT12.y XINT3   XINT4   XINT5    -      -        VCU      FPUOVRF  FPUUNDF EMIFERR  RAMCERR  FLSHCERR  RAMACCV  SYSPLL  AUXPLL  CLAOVRF CLAUNDRF
//
// In Wolfspeed application enabled Pie interrupts are INT1.1 (ADCA1), and INT3.1 (ePWM1)
// Note that USER1 is not PIE Mapped.
//***********************************************************************************************************************************************
void InitPie()
{
   // Install custom interrupts by replacing default TI ISR addresses
   EALLOW;
   PieVectTable.EPWM1_INT = &EPWM1_Interrupt;
   PieVectTable.ADCA1_INT = &ADC_A1_Interrupt;
//   PieVectTable.DMA_CH1_INT = &DMA_CH1_Interrupt;
//   PieVectTable.USER1_INT = &USER_Interrupt;

   // Reset value of all PIEIERx registers is 0x0000 (everything disabled)
   PieCtrlRegs.PIEIER1.bit.INTx1 = 1;     // Enable INT1.1 (ADCA1)
   PieCtrlRegs.PIEIER3.bit.INTx1 = 1;     // Enable INT3.1 (EPWM1)
//   PieCtrlRegs.PIEIER7.bit.INTx1 = 1;     // enable INT7.1 (DMACH1)

   IER |= (M_INT3 | M_INT1);     // | M_INT7 ); Enable group interrupts in IER

   // Enable the PIE
   PieCtrlRegs.PIECTRL.bit.ENPIE = 1;
   // Enables PIE to drive a pulse into the CPU
   PieCtrlRegs.PIEACK.all = 0xFFFF;
   EDIS;
}

void InitGpio()
{
   EALLOW;

   // Set all MUXes to zero
   GpioCtrlRegs.GPAGMUX1.all = 0;
   GpioCtrlRegs.GPBGMUX1.all = 0;
   GpioCtrlRegs.GPCGMUX1.all = 0;
   GpioCtrlRegs.GPDGMUX1.all = 0;
   GpioCtrlRegs.GPEGMUX1.all = 0;
   GpioCtrlRegs.GPFGMUX1.all = 0;
   GpioCtrlRegs.GPAMUX1.all = 0;
   GpioCtrlRegs.GPBMUX1.all = 0;
   GpioCtrlRegs.GPCMUX1.all = 0;
   GpioCtrlRegs.GPDMUX1.all = 0;
   GpioCtrlRegs.GPEMUX1.all = 0;
   GpioCtrlRegs.GPFMUX1.all = 0;

   // PORT A - GPIO0 to GPIO31
   // GPIO0 -  EPWM1A,
   // GPIO1 -  EPWM1B,
   // GPIO2 -  EPWM2A,
   // GPIO3 -  EPWM2B,
   // GPIO4 -  EPWM3A,
   // GPIO5 -  EPWM3B,
   // GPIO6 -  A_Fault Input
   // GPIO7 -  B_Fault Input,
   // GPIO8 -  C_Fault Input,
   // GPIO9 -  EPWM5B - 50% debug pulse,
   // GPIO10 - Case LED1 output,
   // GPIO11 - Case LED2 output,
   // GPIO12 - tristate not used,
   // GPIO13 - tristate not used,
   // GPIO14 - C_PSDIS_Output,
   // GPIO15 - Global Fault input,

   // GPIO16 - tristate not used,
   // GPIO17 - tristate not used,
   // GPIO18 - CANA_RX,
   // GPIO19 - CANA_TX,
   // GPIO20 - tristate not used,
   // GPIO21 - tristate not used,
   // GPIO22 - tristate not used,
   // GPIO23 - tristate not used,
   // GPIO24 - tristate not used,
   // GPIO25 - A_OCEN_Output,
   // GPIO26 - B_OCEN_Output,
   // GPIO27 - A_LEN_Output,
   // GPIO28 - tristate not used,
   // GPIO29 - tristate not used,
   // GPIO30 - tristate not used,
   // GPIO31 - tristate not used

   // disable pullups on all EPWM pins
   GpioCtrlRegs.GPAPUD.all = 0x0000003F;
   // Enable EPWMs and CAN, the rest is GPIO
   GpioCtrlRegs.GPAMUX1.all = 0x00040555;
   GpioCtrlRegs.GPAMUX2.all = 0x000000F0;
   // Write appropriate values into GPO before setting direction of digital outputs
   EN_PS_PH_C;
   RST_FLT_PH_A;
   RST_FLT_PH_B;
   DIS_PWM_PH_A;
   GpioCtrlRegs.GPADIR.all = 0x0E004C00;

   // PORT B - GPIO32 to GPIO63
   // GPIO32 - tristate not used
   // GPIO33 - tristate not used
   // GPIO34 - tristate not used
   // GPIO35 - tristate not used
   // GPIO36 - tristate not used
   // GPIO37 - tristate not used
   // GPIO38 - tristate not used
   // GPIO39 - tristate not used
   // GPIO40 - tristate not used
   // GPIO41 - LED_G Output
   // GPIO42 - tristate not used
   // GPIO43 - tristate not used
   // GPIO44 - tristate not used
   // GPIO45 - tristate not used
   // GPIO46 - tristate not used
   // GPIO47 - tristate not used

   // GPIO48 - tristate not used
   // GPIO49 - tristate not used
   // GPIO50 - tristate not used
   // GPIO51 - tristate not used
   // GPIO52 - LED_Y output
   // GPIO53 - tristate not used
   // GPIO54 - tristate not used
   // GPIO55 - tristate not used
   // GPIO56 - tristate not used
   // GPIO57 - tristate not used
   // GPIO58 - tristate not used - SPIA_MOSI
   // GPIO59 - tristate not used - SPIA_MISO
   // GPIO60 - tristate not used - SPIA_CLK
   // GPIO61 - tristate not used - nSPIA_CS
   // GPIO62 - tristate not used
   // GPIO63 - C_OCEN_Output

   GpioCtrlRegs.GPBPUD.all = 0x00000000;    // Disable the Pull-Up resistors on selected outputs
   // Write appropriate values into GPO before setting direction of digital outputs
   LED_G_ON;
   LED_Y_ON;
   RST_FLT_PH_C;
   GpioCtrlRegs.GPBDIR.all = 0x80100200;

   // PORT C -  GPIO64 to GPIO95
   // GPIO64 - B_LEN_output
   // GPIO65 - LED_R output
   // GPIO66 - B_PSDIS_output
   // GPIO67 - tristate not used
   // GPIO68 - tristate not used
   // GPIO69 - tristate not used
   // GPIO70 - tristate not used
   // GPIO71 - tristate not used
   // GPIO72 - tristate not used
   // GPIO73 - tristate not used
   // GPIO74 - tristate not used
   // GPIO75 - tristate not used
   // GPIO76 - tristate not used
   // GPIO77 - tristate not used
   // GPIO78 - tristate not used
   // GPIO79 - tristate not used

   // GPIO80 - tristate not used
   // GPIO81 - tristate not used
   // GPIO82 - tristate not used
   // GPIO83 - tristate not used
   // GPIO84 - tristate not used
   // GPIO85 - tristate not used
   // GPIO86 - tristate not used
   // GPIO87 - tristate not used
   // GPIO88 - tristate not used
   // GPIO89 - tristate not used
   // GPIO90 - tristate not used
   // GPIO91 - tristate not used
   // GPIO92 - tristate not used
   // GPIO93 - tristate not used
   // GPIO94 - A_RTD input
   // GPIO95 - C_RTD input

   GpioCtrlRegs.GPCPUD.all = 0x00000000;   // Disable the Pull-Up resistor on selected outputs
   // Write appropriate values into GPO before setting direction of digital outputs
   DIS_PWM_PH_B;
   LED_R_ON;
   EN_PS_PH_B;
   GpioCtrlRegs.GPCDIR.all = 0x00000007;

   // PORT D  - GPIO96 to GPIO127
   // GPIO96  - tristate not used
   // GPIO97  - B_RTD input
   // GPIO98  - tristate not used
   // GPIO99  - tristate not used
   // GPIO100 - tristate not used
   // GPIO101 - tristate not used
   // GPIO102 - tristate not used
   // GPIO103 - tristate not used
   // GPIO104 - tristate not used
   // GPIO105 - tristate not used
   // GPIO106 - tristate not used
   // GPIO107 - tristate not used
   // GPIO108 - tristate not used
   // GPIO109 - tristate not used
   // GPIO110 - tristate not used
   // GPIO111 - tristate not used

   // GPIO112 - tristate not used
   // GPIO113 - tristate not used
   // GPIO114 - tristate not used
   // GPIO115 - tristate not used
   // GPIO116 - tristate not used
   // GPIO117 - tristate not used
   // GPIO118 - tristate not used
   // GPIO119 - tristate not used
   // GPIO120 - tristate not used
   // GPIO121 - tristate not used
   // GPIO122 - tristate not used
   // GPIO123 - tristate not used
   // GPIO124 - Shutdown+15 output
   // GPIO125 - Shutdown-15 output
   // GPIO126 - tristate not used
   // GPIO127 - tristate not used

   GpioCtrlRegs.GPDPUD.all = 0x00000000;   // Disable the Pull-Up resistor on selected outputs
   // Write appropriate values into GPO before setting direction of digital outputs
   ENABLE_15V;
   ENABLE_N15V;
   GpioCtrlRegs.GPDDIR.all = 0x30000000;

   // PORT E -  GPIO128 to GPIO159
   // GPIO128 - tristate not used
   // GPIO129 - tristate not used
   // GPIO130 - C_LEN_Output
   // GPIO131 - A_PSDIS_Output
   // GPIO132 - tristate not used
   // GPIO133 - tristate not used
   // GPIO134 - tristate not used
   // GPIO135 - tristate not used
   // GPIO136 - tristate not used
   // GPIO137 - tristate not used
   // GPIO138 - tristate not used
   // GPIO139 - tristate not used
   // GPIO140 - tristate not used
   // GPIO141 - tristate not used
   // GPIO142 - tristate not used
   // GPIO143 - tristate not used

   // GPIO144 - tristate not used
   // GPIO145 - tristate not used
   // GPIO146 - tristate not used
   // GPIO147 - tristate not used
   // GPIO148 - tristate not used
   // GPIO149 - tristate not used
   // GPIO150 - tristate not used
   // GPIO151 - tristate not used
   // GPIO152 - tristate not used
   // GPIO153 - tristate not used
   // GPIO154 - tristate not used
   // GPIO155 - tristate not used
   // GPIO156 - tristate not used
   // GPIO157 - EPWM7A - PWM_DAC3
   // GPIO158 - EPWM7B - PWM_DAC4
   // GPIO159 - EPWM8A - PWM_DAC1

   GpioCtrlRegs.GPEMUX1.all = 0x00000000;
   GpioCtrlRegs.GPEMUX2.all = 0x54000000;
   GpioCtrlRegs.GPEPUD.all = 0x00000000;   // Disable the Pull-Up resistor on selected outputs
   // Write appropriate values into GPO before setting direction of digital outputs
   DIS_PWM_PH_C;
   EN_PS_PH_A;
   GpioCtrlRegs.GPEDIR.all = 0x0000000C;

   // PORT F -  GPIO160 to GPIO168
   // GPIO160 - EPWM8B - PWM_DAC2
   // GPIO161 - tristate not used
   // GPIO162 - tristate not used
   // GPIO163 - tristate not used
   // GPIO164 - tristate not used
   // GPIO165 - tristate not used
   // GPIO166 - tristate not used
   // GPIO167 - tristate not used
   // GPIO168 - tristate not used

   GpioCtrlRegs.GPFMUX1.all = 0x00000001;
   GpioCtrlRegs.GPFPUD.all = 0x00000000;   // Disable the Pull-Up resistor on selected outputs
   GpioCtrlRegs.GPFDIR.all = 0x00000000;

   // Enable +/-15V for current sensors
   ENABLE_15V;
   ENABLE_N15V;

   // wait 10 us then clear gate drive reset
   DELAY_US(10);
   NOT_RST_FLT_PH_A;
   NOT_RST_FLT_PH_B;
   NOT_RST_FLT_PH_C;

   EDIS;
}

void InitXBar(void)
{
   // Configure input X-bar to connect A_RTD (GPIO94), B_RTD (GPIO97), and C_RTD (GPIO95)
   // to input 7, 8, and 9 that are connected to eCAP1,2, and 3, respectively

   InputXbarRegs.INPUT7SELECT = 94;
   InputXbarRegs.INPUT8SELECT = 97;
   InputXbarRegs.INPUT9SELECT = 95;

   // Connect Global Fault (GPIO15) to Input1. The input 1 is connected to /TZ1 (trip zone 1) that shuts down all PWM
   InputXbarRegs.INPUT1SELECT = 15;
}

void InitECap(void)
{


   // ECAP module 1 config Prescale by 1, rising edge capture
   ECap1Regs.ECCTL1.all = 0;
   ECap1Regs.ECCTL1.bit.CTRRST1 = 1;   // 1h (R/W) = Reset counter after Event 1 time-stamp has been
   ECap1Regs.ECCTL1.bit.CTRRST2 = 1;   // captured (used in difference mode operation)
   ECap1Regs.ECCTL1.bit.CTRRST3 = 1;
   ECap1Regs.ECCTL1.bit.CTRRST4 = 1;
   ECap1Regs.ECCTL1.bit.CAPLDEN = 1;   // 1h (R/W) = Enable CAP1-4 register loads at capture event time.

   ECap1Regs.ECCTL2.all = 0;  // eCap mode, sync disabled
   ECap1Regs.ECCTL2.bit.CONT_ONESHT = 1;    // operates in one shot mode, one read per background cycle
   ECap1Regs.ECCTL2.bit.STOP_WRAP = 3;      // 3h (R/W) = Stop after Capture Event 4 in one-shot mode
   ECap1Regs.ECCTL2.bit.TSCTRSTOP = 1;      // 1h (R/W) = TSCTR free-running
   ECap1Regs.ECCTL2.bit.CAP_APWM = 0;

   // ECAP module 2 config Prescale by 1, rising edge capture
   ECap2Regs.ECCTL1.all = 0;
   ECap2Regs.ECCTL1.bit.CTRRST1 = 1;   // 1h (R/W) = Reset counter after Event 1 time-stamp has been
   ECap2Regs.ECCTL1.bit.CTRRST2 = 1;   // captured (used in difference mode operation)
   ECap2Regs.ECCTL1.bit.CTRRST3 = 1;
   ECap2Regs.ECCTL1.bit.CTRRST4 = 1;
   ECap2Regs.ECCTL1.bit.CAPLDEN = 1;   // 1h (R/W) = Enable CAP1-4 register loads at capture event time.

   ECap2Regs.ECCTL2.all = 0;  // eCap mode, sync disabled
   ECap2Regs.ECCTL2.bit.CONT_ONESHT = 1;    // operates in one shot mode, one read per background cycle
   ECap2Regs.ECCTL2.bit.STOP_WRAP = 3;      // 3h (R/W) = Stop after Capture Event 4 in one-shot mode
   ECap2Regs.ECCTL2.bit.TSCTRSTOP = 1;      // 1h (R/W) = TSCTR free-running
   ECap2Regs.ECCTL2.bit.CAP_APWM = 0;

   // ECAP module 3 config Prescale by 1, rising edge capture
   ECap3Regs.ECCTL1.all = 0;
   ECap3Regs.ECCTL1.bit.CTRRST1 = 1;   // 1h (R/W) = Reset counter after Event 1 time-stamp has been
   ECap3Regs.ECCTL1.bit.CTRRST2 = 1;   // captured (used in difference mode operation)
   ECap3Regs.ECCTL1.bit.CTRRST3 = 1;
   ECap3Regs.ECCTL1.bit.CTRRST4 = 1;
   ECap3Regs.ECCTL1.bit.CAPLDEN = 1;   // 1h (R/W) = Enable CAP1-4 register loads at capture event time.

   ECap3Regs.ECCTL2.all = 0;  // eCap mode, sync disabled
   ECap3Regs.ECCTL2.bit.CONT_ONESHT = 1;    // operates in one shot mode, one read per background cycle
   ECap3Regs.ECCTL2.bit.STOP_WRAP = 3;      // 3h (R/W) = Stop after Capture Event 4 in one-shot mode
   ECap3Regs.ECCTL2.bit.TSCTRSTOP = 1;      // 1h (R/W) = TSCTR free-running
   ECap3Regs.ECCTL2.bit.CAP_APWM = 0;

   ECap1Regs.ECCTL2.bit.REARM = 1;  // Arm the eCap1
   ECap2Regs.ECCTL2.bit.REARM = 1;  // Arm the eCap2
   ECap3Regs.ECCTL2.bit.REARM = 1;  // Arm the eCap3
}

void InitEPwm(void)
{
   EALLOW;
   // The proper procedure for enabling ePWM clocks is as follows:
   // 1. Enable ePWM module clocks in the PCLKCR1 register.
   // 2. Set TBCLKSYNC to 0.
   // 3. Configure prescaler values and ePWM modes.
   // 4. Set TBCLKSYNC to 1.
   CpuSysRegs.PCLKCR0.bit.TBCLKSYNC = 0;

   // Master ePWM channel - Phase A
   EPwm1Regs.TBPRD = PeriodCounter;                  // Set timer period
   EPwm1Regs.TBPHS.bit.TBPHS = 0x0000;               // Phase is 0
   EPwm1Regs.TBCTR = 0x0000;                         // Clear counter
   EPwm1Regs.TBCTL.bit.PHSEN = TB_DISABLE;           // Disable phase loading
   EPwm1Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1;          // Clock ratio to SYSCLKOUT
   EPwm1Regs.TBCTL.bit.CLKDIV = TB_DIV1;
   EPwm1Regs.TBCTL.bit.PRDLD = TB_SHADOW;
   EPwm1Regs.TBCTL.bit.SYNCOSEL = TB_CTR_ZERO;       // Sync down-stream module
   EPwm1Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN;    // Count up and down
   EPwm1Regs.TBCTL.bit.FREE_SOFT = 3;                // Free run
   EPwm1Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;
   EPwm1Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;
   EPwm1Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO_PRD;
   EPwm1Regs.CMPCTL.bit.LOADBMODE = CC_LD_DISABLE;
   EPwm1Regs.CMPA.bit.CMPA = PeriodCounter/2;        // Set compare A value
   EPwm1Regs.CMPB.bit.CMPB = 0;                      // Set Compare B value - not used
   EPwm1Regs.AQCTLA.bit.CAU = AQ_CLEAR;              // Clear PWM1A on event A, up count
   EPwm1Regs.AQCTLA.bit.CAD = AQ_SET;                // Set PWM1A on event B, down count
   EPwm1Regs.AQCTLA.bit.ZRO = AQ_NO_ACTION;
   EPwm1Regs.AQCTLA.bit.PRD = AQ_NO_ACTION;
   EPwm1Regs.AQCTLA.bit.CBU = AQ_NO_ACTION;
   EPwm1Regs.AQCTLA.bit.CBD = AQ_NO_ACTION;
   EPwm1Regs.DBCTL.bit.OUT_MODE = DB_FULL_ENABLE;    // enable Dead-band module
   EPwm1Regs.DBCTL.bit.POLSEL = DB_ACTV_HIC;         // Active Hi complementary
   EPwm1Regs.DBFED.bit.DBFED = DeadTimeCounts;       // FED = 50 TBCLKs, dead time = 500ns;
   EPwm1Regs.DBRED.bit.DBRED = DeadTimeCounts;       // RED = 50 TBCLKs
   EPwm1Regs.TZSEL.bit.OSHT1 = TZ_ENABLE;            // Enable /TZ1 one shot mode
//   EPwm1Regs.TZSEL.bit.OSHT1 = TZ_DISABLE;            // disable /TZ1 one shot mode - debug
   EPwm1Regs.TZCTL.bit.TZA = TZ_FORCE_LO;            // Force low id trips
   EPwm1Regs.TZCTL.bit.TZB = TZ_FORCE_LO;
//   EPwm1Regs.TZFRC.bit.OST = 1;                      // Force One time trip zone event - turns off all switches fixme
   EPwm1Regs.TZEINT.bit.OST = TZ_DISABLE;            // Disable trip interrupt
   EPwm1Regs.ETSEL.bit.INTSEL = ET_CTR_PRDZERO;      // Select INT on Zero and Period events (2x controller)
   EPwm1Regs.ETSEL.bit.INTEN = 0;                    // Temporary disable INT, enabled in the main.c
   EPwm1Regs.ETPS.bit.INTPRD = ET_1ST;               // Generate INT on 1st event

   // Slave ePWM channels - Phase B
   EPwm2Regs.TBPRD = PeriodCounter;                  // Set timer period
   EPwm2Regs.TBPHS.bit.TBPHS = PeriodCounter;        // Phase is PeriodCounter so ePWM2 is 180 deg phase shifted
   EPwm2Regs.TBCTR = PeriodCounter;                  // Clear counter
   EPwm2Regs.TBCTL.bit.PHSEN = TB_ENABLE;            // Enable phase loading
   EPwm2Regs.TBCTL.bit.PHSDIR = TB_DOWN;             // Count down after phase is loaded to create 180 deg phase shift
   EPwm2Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1;          // Clock ratio to SYSCLKOUT
   EPwm2Regs.TBCTL.bit.CLKDIV = TB_DIV1;
   EPwm2Regs.TBCTL.bit.PRDLD = TB_SHADOW;
   EPwm2Regs.TBCTL.bit.SYNCOSEL = TB_CTR_ZERO;       // Sync down-stream module
   EPwm2Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN;    // Count up and down
   EPwm2Regs.TBCTL.bit.FREE_SOFT = 3;                // Free run
   EPwm2Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;
   EPwm2Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;
   EPwm2Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO_PRD;
   EPwm2Regs.CMPCTL.bit.LOADBMODE = CC_LD_DISABLE;
   EPwm2Regs.CMPA.bit.CMPA = PeriodCounter/2;        // Set compare A value
   EPwm2Regs.CMPB.bit.CMPB = 0;                      // Set Compare B value - not used
   EPwm2Regs.AQCTLA.bit.CAU = AQ_CLEAR;              // Clear PWM2A on event A, up count
   EPwm2Regs.AQCTLA.bit.CAD = AQ_SET;                // Set PWM2A on event B, down count
   EPwm2Regs.AQCTLA.bit.ZRO = AQ_NO_ACTION;
   EPwm2Regs.AQCTLA.bit.PRD = AQ_NO_ACTION;
   EPwm2Regs.AQCTLA.bit.CBU = AQ_NO_ACTION;
   EPwm2Regs.AQCTLA.bit.CBD = AQ_NO_ACTION;
   EPwm2Regs.DBCTL.bit.OUT_MODE = DB_FULL_ENABLE;    // enable Dead-band module
   EPwm2Regs.DBCTL.bit.POLSEL = DB_ACTV_HIC;         // Active Hi complementary
   EPwm2Regs.DBFED.bit.DBFED = DeadTimeCounts;       // FED = 50 TBCLKs, dead time = 500ns;
   EPwm2Regs.DBRED.bit.DBRED = DeadTimeCounts;       // RED = 50 TBCLKs
   EPwm2Regs.TZSEL.bit.OSHT1 = TZ_ENABLE;            // Enable /TZ1 one shot mode fixme
//   EPwm2Regs.TZSEL.bit.OSHT1 = TZ_DISABLE;            // Enable /TZ1 one shot mode
   EPwm2Regs.TZCTL.bit.TZA = TZ_FORCE_LO;            // Force low id trips
   EPwm2Regs.TZCTL.bit.TZB = TZ_FORCE_LO;
//   EPwm2Regs.TZFRC.bit.OST = 1;                      // Force One time trip zone event - turns off all switches fixme
   EPwm2Regs.TZEINT.bit.OST = TZ_DISABLE;            // Disable trip interrupt
   EPwm2Regs.ETSEL.bit.INTEN = 0;                    // Disable INT

   // Slave ePWM channels - Phase C
   EPwm3Regs.TBPRD = PeriodCounter;                  // Set timer period
   EPwm3Regs.TBPHS.bit.TBPHS = PeriodCounter;        // Phase is PeriodCounter so ePWM2 is 180 deg phase shifted
   EPwm3Regs.TBCTR = PeriodCounter;                  // Clear counter
   EPwm3Regs.TBCTL.bit.PHSEN = TB_ENABLE;            // Enable phase loading
   EPwm3Regs.TBCTL.bit.PHSDIR = TB_DOWN;             // Count down after phase is loaded to create 180 deg phase shift
   EPwm3Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1;          // Clock ratio to SYSCLKOUT
   EPwm3Regs.TBCTL.bit.CLKDIV = TB_DIV1;
   EPwm3Regs.TBCTL.bit.PRDLD = TB_SHADOW;
   EPwm3Regs.TBCTL.bit.SYNCOSEL = TB_CTR_ZERO;       // Sync down-stream module
   EPwm3Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN;    // Count up and down
   EPwm3Regs.TBCTL.bit.FREE_SOFT = 3;                // Free run
   EPwm3Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;
   EPwm3Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;
   EPwm3Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO_PRD;
   EPwm3Regs.CMPCTL.bit.LOADBMODE = CC_LD_DISABLE;
   EPwm3Regs.CMPA.bit.CMPA = PeriodCounter/2;        // Set compare A value
   EPwm3Regs.CMPB.bit.CMPB = 0;                      // Set Compare B value - not used
   EPwm3Regs.AQCTLA.bit.CAU = AQ_CLEAR;              // Clear PWM3A on event A, up count
   EPwm3Regs.AQCTLA.bit.CAD = AQ_SET;                // Set PWM3A on event B, down count
   EPwm3Regs.AQCTLA.bit.ZRO = AQ_NO_ACTION;
   EPwm3Regs.AQCTLA.bit.PRD = AQ_NO_ACTION;
   EPwm3Regs.AQCTLA.bit.CBU = AQ_NO_ACTION;
   EPwm3Regs.AQCTLA.bit.CBD = AQ_NO_ACTION;
   EPwm3Regs.DBCTL.bit.OUT_MODE = DB_FULL_ENABLE;    // enable Dead-band module
   EPwm3Regs.DBCTL.bit.POLSEL = DB_ACTV_HIC;         // Active Hi complementary
   EPwm3Regs.DBFED.bit.DBFED = DeadTimeCounts;       // FED = 50 TBCLKs, dead time = 500ns;
   EPwm3Regs.DBRED.bit.DBRED = DeadTimeCounts;       // RED = 50 TBCLKs
   EPwm3Regs.TZSEL.bit.OSHT1 = TZ_ENABLE;            // Enable /TZ1 one shot mode fixme
//   EPwm3Regs.TZSEL.bit.OSHT1 = TZ_DISABLE;           // Enable /TZ1 one shot mode
   EPwm3Regs.TZCTL.bit.TZA = TZ_FORCE_LO;            // Force low id trips
   EPwm3Regs.TZCTL.bit.TZB = TZ_FORCE_LO;
//   EPwm3Regs.TZFRC.bit.OST = 1;                      // Force One time trip zone event - turns off all switches fixme
   EPwm3Regs.TZEINT.bit.OST = TZ_DISABLE;            // Disable trip interrupt
   EPwm3Regs.ETSEL.bit.INTEN = 0;                    // Disable INT

   // ePWM4 generates 50% duty cycle for debug purpose (oscilloscope sync)
   EPwm4Regs.TBPRD = PeriodCounter;                  // Set timer period
   EPwm4Regs.TBPHS.bit.TBPHS = 0x0000;               // Phase is 0
   EPwm4Regs.TBCTR = 0x0000;                         // Clear counter
   EPwm4Regs.TBCTL.bit.PHSEN = TB_DISABLE;           // Disable phase loading
   EPwm4Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1;          // Clock ratio to SYSCLKOUT
   EPwm4Regs.TBCTL.bit.CLKDIV = TB_DIV1;
   EPwm4Regs.TBCTL.bit.PRDLD = TB_SHADOW;
   EPwm4Regs.TBCTL.bit.SYNCOSEL = TB_CTR_ZERO;       // Sync down-stream module
   EPwm4Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN;    // Count up and down
   EPwm4Regs.TBCTL.bit.FREE_SOFT = 3;                // Free run
   EPwm4Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;
   EPwm4Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;
   EPwm4Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO;
   EPwm4Regs.CMPCTL.bit.LOADBMODE = CC_LD_DISABLE;
   EPwm4Regs.AQCTLA.bit.CAU = AQ_NO_ACTION;
   EPwm4Regs.AQCTLA.bit.CAD = AQ_NO_ACTION;
   EPwm4Regs.AQCTLA.bit.ZRO = AQ_SET;                // Set PWM4A on counter equal zero (50% duty cycle)
   EPwm4Regs.AQCTLA.bit.PRD = AQ_CLEAR;              // Clear PWM4A on counter equal period (50% duty cycle)
   EPwm4Regs.AQCTLA.bit.CBU = AQ_NO_ACTION;
   EPwm4Regs.AQCTLA.bit.CBD = AQ_NO_ACTION;
   EPwm4Regs.DBCTL.bit.OUT_MODE = DB_DISABLE;        // Disable Dead-band module since this is not real PWM
   EPwm4Regs.DBCTL.bit.POLSEL = DB_ACTV_HIC;         // Active Hi complementary
   EPwm4Regs.DBFED.bit.DBFED = 0;                    // FED = 50 TBCLKs, dead time = 500ns;
   EPwm4Regs.DBRED.bit.DBRED = 0;                    // RED = 50 TBCLKs
   EPwm4Regs.TZSEL.bit.OSHT1 = TZ_DISABLE;           // Enable /TZ1 one shot mode
   EPwm4Regs.TZCTL.bit.TZA = TZ_NO_CHANGE;           // Force low id trips
   EPwm4Regs.TZCTL.bit.TZB = TZ_NO_CHANGE;
   EPwm4Regs.TZEINT.bit.OST = TZ_DISABLE;            // Disable trip interrupt
   EPwm4Regs.CMPA.bit.CMPA = 0;                      // Set compare A value - not used
   EPwm4Regs.CMPB.bit.CMPB = PeriodCounter;          // Set Compare B value - not used
   EPwm4Regs.ETSEL.bit.INTEN = 0;                    // Disable INT

   // ePWM5 generates ADC SOC at Cal.Oversampling rate
   EPwm5Regs.TBPRD = AdcCounter;                     // Set timer period
   EPwm5Regs.TBPHS.bit.TBPHS = 0x0000;               // Phase is 0
   EPwm5Regs.TBCTR = 0x0000;                         // Clear counter
   EPwm5Regs.TBCTL.bit.PHSEN = TB_DISABLE;           // Disable phase loading
   EPwm5Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1;          // Clock ratio to SYSCLKOUT
   EPwm5Regs.TBCTL.bit.CLKDIV = TB_DIV1;
   EPwm5Regs.TBCTL.bit.PRDLD = TB_SHADOW;
   EPwm5Regs.TBCTL.bit.SYNCOSEL = TB_SYNC_DISABLE;   // Sunc disables since ePWM5 is the last module
   EPwm5Regs.TBCTL.bit.CTRMODE = TB_COUNT_DOWN;      // Count down
   EPwm5Regs.TBCTL.bit.FREE_SOFT = 3;                // Free run
   EPwm5Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;
   EPwm5Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;
   EPwm5Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO;
   EPwm5Regs.CMPCTL.bit.LOADBMODE = CC_LD_DISABLE;
   EPwm5Regs.AQCTLA.bit.CAU = AQ_NO_ACTION;
   EPwm5Regs.AQCTLA.bit.CAD = AQ_NO_ACTION;
   EPwm5Regs.AQCTLA.bit.ZRO = AQ_SET;                // Set PWM4A on counter equal zero (50% duty cycle)
   EPwm5Regs.AQCTLA.bit.PRD = AQ_CLEAR;              // Clear PWM4A on counter equal period (50% duty cycle)
   EPwm5Regs.AQCTLA.bit.CBU = AQ_NO_ACTION;
   EPwm5Regs.AQCTLA.bit.CBD = AQ_NO_ACTION;
   EPwm5Regs.DBCTL.bit.OUT_MODE = DB_DISABLE;        // Disable Dead-band module since this is not real PWM
   EPwm5Regs.DBCTL.bit.POLSEL = DB_ACTV_HIC;         // Active Hi complementary
   EPwm5Regs.DBFED.bit.DBFED = 0;                    // FED = 50 TBCLKs, dead time = 500ns;
   EPwm5Regs.DBRED.bit.DBRED = 0;                    // RED = 50 TBCLKs
   EPwm5Regs.TZSEL.bit.OSHT1 = TZ_DISABLE;           // Disable /TZ1 one shot mode
   EPwm5Regs.TZCTL.bit.TZA = TZ_NO_CHANGE;           // Force low id trips
   EPwm5Regs.TZCTL.bit.TZB = TZ_NO_CHANGE;
   EPwm5Regs.TZEINT.bit.OST = TZ_DISABLE;            // Disable trip interrupt
   EPwm5Regs.CMPA.bit.CMPA = PeriodCounter/2;        // Set compare A value to half PeriodCounter
   EPwm5Regs.CMPB.bit.CMPB = 0;                      // Set Compare B value - not used
   EPwm5Regs.ETSEL.bit.INTEN = 0;                    // Disable INT
   EPwm5Regs.ETSEL.bit.SOCAEN = 0;                   // Disable SOCA - enabled in ePWM1 interrupt
   EPwm5Regs.ETSEL.bit.SOCASEL = ET_CTRD_CMPA;       // Start ADC when ePWM3 counter equal to Periodcounter/2
   EPwm5Regs.ETPS.bit.SOCAPRD = ET_1ST;              // Start ADC on the first period...

   CpuSysRegs.PCLKCR0.bit.TBCLKSYNC = 1;
   EDIS;
}

void InitDAC(void)
{
   EALLOW;
   DacaRegs.DACCTL.bit.LOADMODE = 0;      // load DAC as soon as it is given with a new value
   DacaRegs.DACCTL.bit.DACREFSEL = 1;     // use ADC reference (3V) for DAC range
   DacaRegs.DACVALS.all = 2048;           // mid scale
   DacaRegs.DACOUTEN.bit.DACOUTEN = 0;    // turn on the DAC

   DacbRegs.DACCTL.bit.LOADMODE = 0;      // load DAC as soon as we give it a new value
   DacbRegs.DACCTL.bit.DACREFSEL = 1;     // use ADC reference (3V) for DAC range
   DacbRegs.DACVALS.all = 2048;           // mid scale
   DacbRegs.DACOUTEN.bit.DACOUTEN = 1;    // turn on the DAC

   DaccRegs.DACCTL.bit.LOADMODE = 0;      // load DAC as soon as we give it a new value
   DaccRegs.DACCTL.bit.DACREFSEL = 1;     // use ADC reference (3V) for DAC range
   DaccRegs.DACVALS.all = 2048;           // mid scale
   DaccRegs.DACOUTEN.bit.DACOUTEN = 0;    // turn on the DAC
   EDIS;
}

//*********************************************************************************************************************************
// ADC channels configuration
//
// SOC0 => A0 - IindA     B2 -  IindC      C2 - VoutA
// SOC1 => A2 - IindB     B3 -  VoutC      C3 - Vdc
// SOC2 => A3 - VoutB     B15 - Case_Tempr C4 - B_Tempr
// SOC3 => A1 - Iext_curr
// SOC4 => A4 - A_Tempr
// SOC5 => A5 - C_Tempr
//
// The ADC Interrupt is placed after SOC5 conversion is done
//
//  |            Control Cycle                                  |              Control Cycle                                |
//  ePWM1                                                       ePWM1                                                       ePWM1
//  |                                                           |                                                           |
//  |              |              |              |              |              |              |              |              |
//  ePWM5          ePWM5          ePWM5          ePWM5          epWM5          ePWM5          ePWM5          ePWM5          epWM5
//  SOC0,1,2,3,4,5 SOC0,1,2,3,4,5 SOC0,1,2,3,4,5 SOC0,1,2,3,4,5 SOC0,1,2,3,4,5 SOC0,1,2,3,4,5 SOC0,1,2,3,4,5 SOC0,1,2,3,4,5 SOC0,1,2,3,4,5
//
//  AdcIntCounter
//    0            1              2              3              0              1              2              3              0
//************************************************************************************************************************************
void InitADC(void)
{
   EALLOW;

   AdcSetMode(ADC_ADCA, ADC_RESOLUTION_12BIT, ADC_SIGNALMODE_SINGLE);
   // ADC configuration from the Reference Manual: "11.2.5.2 Oversampled Conversion from ePWM Trigger"
   // SOC0, SOC1, SOC2, and  SOC3 are triggered by ePWM5.

   //Set pulse positions to late
   AdcaRegs.ADCCTL1.bit.INTPULSEPOS = 1;      // interrupt generated at the end of conversion
   AdcaRegs.ADCCTL2.bit.PRESCALE = 6;         // ADCCLK divider is set to divide by 4
   AdcaRegs.ADCCTL2.bit.RESOLUTION = 0;       // 12 bit
   AdcaRegs.ADCCTL2.bit.SIGNALMODE = 0;       // single ended

   AdcbRegs.ADCCTL1.bit.INTPULSEPOS = 0;      // interrupt not generated at the end of conversion
   AdcbRegs.ADCCTL2.bit.PRESCALE = 6;         // ADCCLK divider is set to divide by 4
   AdcbRegs.ADCCTL2.bit.RESOLUTION = 0;       // 12 bit
   AdcbRegs.ADCCTL2.bit.SIGNALMODE = 0;       // single ended

   AdccRegs.ADCCTL1.bit.INTPULSEPOS = 0;      // interrupt not generated at the end of conversion
   AdccRegs.ADCCTL2.bit.PRESCALE = 6;         // ADCCLK divider is set to divide by 4
   AdccRegs.ADCCTL2.bit.RESOLUTION = 0;       // 12 bit
   AdccRegs.ADCCTL2.bit.SIGNALMODE = 0;       // single ended

//   AdcdRegs.ADCCTL1.bit.INTPULSEPOS = 0;      // interrupt not generated at the end of conversion
//   AdcdRegs.ADCCTL2.bit.PRESCALE = 6;         // ADCCLK divider is set to divide by 4
//   AdcdRegs.ADCCTL2.bit.RESOLUTION = 0;       // 12 bit
//   AdcdRegs.ADCCTL2.bit.SIGNALMODE = 0;       // single ended

   AdcaRegs.ADCCTL1.bit.ADCPWDNZ = 1;         //power up the ADC
   AdcbRegs.ADCCTL1.bit.ADCPWDNZ = 1;
   AdccRegs.ADCCTL1.bit.ADCPWDNZ = 1;
//   AdcdRegs.ADCCTL1.bit.ADCPWDNZ = 1;

   AdcaRegs.ADCSOC0CTL.all = 0;
   AdcbRegs.ADCSOC0CTL.all = 0;
   AdccRegs.ADCSOC0CTL.all = 0;
//   AdcdRegs.ADCSOC0CTL.all = 0;

   // SOC0 ADC Channel A //
   AdcaRegs.ADCSOC0CTL.bit.ACQPS = 63;     // sample window min 320ns = 64 sysclk * (1/200MHz), 63 -> 64 cycles (n-1)
   AdcaRegs.ADCSOC0CTL.bit.CHSEL = 0;      // SOC0 converts pin A0 - IindA
   AdcaRegs.ADCSOC0CTL.bit.TRIGSEL = 0xD;  // trigger ePWM5 SOCA

   // SOC0 ADC Channel B //
   AdcbRegs.ADCSOC0CTL.bit.ACQPS = 63;     // sample window min 320ns = 64 sysclk * (1/200MHz), 63 -> 64 cycles (n-1)
   AdcbRegs.ADCSOC0CTL.bit.CHSEL = 2;      // SOC0 will convert pin B2 - IindC
   AdcbRegs.ADCSOC0CTL.bit.TRIGSEL = 0xD;  // trigger ePWM5 SOCA

   // SOC0 ADC Channel C //
   AdccRegs.ADCSOC0CTL.bit.ACQPS = 63;     // sample window min 320ns = 64 sysclk * (1/200MHz), 63 -> 64 cycles (n-1)
   AdccRegs.ADCSOC0CTL.bit.CHSEL = 2;      // SOC0 will convert pin C2 - VoutA
   AdccRegs.ADCSOC0CTL.bit.TRIGSEL = 0xD;  // trigger ePWM5 SOCA

//   // SOC0 ADC Channel D //
//   AdcdRegs.ADCSOC0CTL.bit.ACQPS = 63;     // sample window min 320ns = 64 sysclk * (1/200MHz), 63 -> 64 cycles (n-1)
//   AdcdRegs.ADCSOC0CTL.bit.CHSEL = 0;      // SOC0 will convert pin D0
//   AdcdRegs.ADCSOC0CTL.bit.TRIGSEL = 0xD;  // trigger ePWM5 SOCA

   // SOC1 ADC Channel A //
   AdcaRegs.ADCSOC1CTL.bit.ACQPS = 63;     // sample window min 320ns = 64 sysclk * (1/200MHz), 63 -> 64 cycles (n-1)
   AdcaRegs.ADCSOC1CTL.bit.CHSEL = 2;      // SOC1 converts pin A2 - IindB
   AdcaRegs.ADCSOC1CTL.bit.TRIGSEL = 0xD;  // trigger ePWM5 SOCA

   // SOC1 ADC Channel B //
   AdcbRegs.ADCSOC1CTL.bit.ACQPS = 63;     // sample window min 320ns = 64 sysclk * (1/200MHz), 63 -> 64 cycles (n-1)
   AdcbRegs.ADCSOC1CTL.bit.CHSEL = 3;      // SOC1 will convert pin B3 - VoutC
   AdcbRegs.ADCSOC1CTL.bit.TRIGSEL = 0xD;  // trigger ePWM5 SOCA

   // SOC1 ADC Channel C //
   AdccRegs.ADCSOC1CTL.bit.ACQPS = 63;     // sample window min 320ns = 64 sysclk * (1/200MHz), 63 -> 64 cycles (n-1)
   AdccRegs.ADCSOC1CTL.bit.CHSEL = 3;      // SOC1 will convert pin C3 - Vdc
   AdccRegs.ADCSOC1CTL.bit.TRIGSEL = 0xD;  // trigger ePWM5 SOCA

//   // SOC1 ADC Channel D //
//   AdcdRegs.ADCSOC1CTL.bit.ACQPS = 63;     // sample window min 320ns = 64 sysclk * (1/200MHz), 63 -> 64 cycles (n-1)
//   AdcdRegs.ADCSOC1CTL.bit.CHSEL = 2;      // SOC1 will convert pin D2
//   AdcdRegs.ADCSOC1CTL.bit.TRIGSEL = 0xD;  // trigger ePWM5 SOCA

   // SOC2 ADC Channel A //
   AdcaRegs.ADCSOC2CTL.bit.ACQPS = 63;     // sample window min 320ns = 64 sysclk * (1/200MHz), 63 -> 64 cycles (n-1)
   AdcaRegs.ADCSOC2CTL.bit.CHSEL = 3;      // SOC2 converts pin A3 - VoutB
   AdcaRegs.ADCSOC2CTL.bit.TRIGSEL = 0xD;  // trigger ePWM5 SOCA

   // SOC2 ADC Channel B //
   AdcbRegs.ADCSOC2CTL.bit.ACQPS = 63;     // sample window min 320ns = 64 sysclk * (1/200MHz), 63 -> 64 cycles (n-1)
   AdcbRegs.ADCSOC2CTL.bit.CHSEL = 15;     // SOC2 will convert pin B15 - Case Temperature
   AdcbRegs.ADCSOC2CTL.bit.TRIGSEL = 0xD;  // trigger ePWM5 SOCA

   // SOC2 ADC Channel C //
   AdccRegs.ADCSOC2CTL.bit.ACQPS = 63;     // sample window min 320ns = 64 sysclk * (1/200MHz), 63 -> 64 cycles (n-1)
   AdccRegs.ADCSOC2CTL.bit.CHSEL = 4;      // SOC2 will convert pin C4 - B Temperature
   AdccRegs.ADCSOC2CTL.bit.TRIGSEL = 0xD;  // trigger ePWM5 SOCA

//   // SOC2 ADC Channel D //
//   AdcdRegs.ADCSOC2CTL.bit.ACQPS = 63;     // sample window min 320ns = 64 sysclk * (1/200MHz), 63 -> 64 cycles (n-1)
//   AdcdRegs.ADCSOC2CTL.bit.CHSEL = 0;      // SOC0 will convert pin D0 - DC Link Voltage
//   AdcdRegs.ADCSOC2CTL.bit.TRIGSEL = 0;    // trigger software only since it is used in burst mode

   // SOC3 ADC Channel A //
   AdcaRegs.ADCSOC3CTL.bit.ACQPS = 63;     // sample window min 320ns = 64 sysclk * (1/200MHz), 63 -> 64 cycles (n-1)
   AdcaRegs.ADCSOC3CTL.bit.CHSEL = 1;      // SOC3 converts pin A1 - Iext_curr
   AdcaRegs.ADCSOC3CTL.bit.TRIGSEL = 0xD;  // trigger ePWM5 SOCA

#if 0
   // SOC3 ADC Channel B //
   AdcbRegs.ADCSOC3CTL.bit.ACQPS = 63;     // sample window min 320ns = 64 sysclk * (1/200MHz), 63 -> 64 cycles (n-1)
   AdcbRegs.ADCSOC3CTL.bit.CHSEL = 0;      // SOC3 will convert pin B0
//   AdcbRegs.ADCSOC3CTL.bit.TRIGSEL = 0;    // trigger software only

   // SOC3 ADC Channel C //
   AdccRegs.ADCSOC3CTL.bit.ACQPS = 63;     // sample window min 320ns = 64 sysclk * (1/200MHz), 63 -> 64 cycles (n-1)
   AdccRegs.ADCSOC3CTL.bit.CHSEL = 4;      // SOC3 will convert pin C4
//   AdccRegs.ADCSOC3CTL.bit.TRIGSEL = 0;    // trigger software only

   // SOC3 ADC Channel D //
   AdcdRegs.ADCSOC3CTL.bit.ACQPS = 63;     // sample window min 320ns = 64 sysclk * (1/200MHz), 63 -> 64 cycles (n-1)
   AdcdRegs.ADCSOC3CTL.bit.CHSEL = 2;      // SOC3 will convert pin D2
//   AdcdRegs.ADCSOC3CTL.bit.TRIGSEL = 0;    // trigger software only
#endif

   // SOC4 ADC Channel A //
   AdcaRegs.ADCSOC4CTL.bit.ACQPS = 63;     // sample window min 320ns = 64 sysclk * (1/200MHz), 63 -> 64 cycles (n-1)
   AdcaRegs.ADCSOC4CTL.bit.CHSEL = 4;      // SOC4 converts pin A4 - A_Temperature
   AdcaRegs.ADCSOC4CTL.bit.TRIGSEL = 0xD;  // trigger ePWM5 SOCA

#if 0
   // SOC4 ADC Channel B //
   AdcbRegs.ADCSOC4CTL.bit.ACQPS = 63;     // sample window min 320ns = 64 sysclk * (1/200MHz), 63 -> 64 cycles (n-1)
   AdcbRegs.ADCSOC4CTL.bit.CHSEL = 0;      // SOC4 will convert pin B0
//   AdcbRegs.ADCSOC3CTL.bit.TRIGSEL = 0;    // trigger software only

   // SOC3 ADC Channel C //
   AdccRegs.ADCSOC4CTL.bit.ACQPS = 63;     // sample window min 320ns = 64 sysclk * (1/200MHz), 63 -> 64 cycles (n-1)
   AdccRegs.ADCSOC4CTL.bit.CHSEL = 4;      // SOC4 will convert pin C4
//   AdccRegs.ADCSOC3CTL.bit.TRIGSEL = 0;    // trigger software only

   // SOC3 ADC Channel D //
   AdcdRegs.ADCSOC4CTL.bit.ACQPS = 63;     // sample window min 320ns = 64 sysclk * (1/200MHz), 63 -> 64 cycles (n-1)
   AdcdRegs.ADCSOC4CTL.bit.CHSEL = 2;      // SOC4 will convert pin D2
//   AdcdRegs.ADCSOC3CTL.bit.TRIGSEL = 0;    // trigger software only
#endif

   // SOC5 ADC Channel A //
   AdcaRegs.ADCSOC5CTL.bit.ACQPS = 63;     // sample window min 320ns = 64 sysclk * (1/200MHz), 63 -> 64 cycles (n-1)
   AdcaRegs.ADCSOC5CTL.bit.CHSEL = 5;      // SOC5 converts pin A5 - C_Temperature
   AdcaRegs.ADCSOC5CTL.bit.TRIGSEL = 0xD;  // trigger ePWM5 SOCA

#if 0
   // SOC5 ADC Channel B //
   AdcbRegs.ADCSOC5CTL.bit.ACQPS = 63;     // sample window min 320ns = 64 sysclk * (1/200MHz), 63 -> 64 cycles (n-1)
   AdcbRegs.ADCSOC5CTL.bit.CHSEL = 0;      // SOC4 will convert pin B0
//   AdcbRegs.ADCSOC3CTL.bit.TRIGSEL = 0;    // trigger software only

   // SOC5 ADC Channel C //
   AdccRegs.ADCSOC5CTL.bit.ACQPS = 63;     // sample window min 320ns = 64 sysclk * (1/200MHz), 63 -> 64 cycles (n-1)
   AdccRegs.ADCSOC5CTL.bit.CHSEL = 4;      // SOC4 will convert pin C4
//   AdccRegs.ADCSOC3CTL.bit.TRIGSEL = 0;    // trigger software only

   // SOC5 ADC Channel D //
   AdcdRegs.ADCSOC5CTL.bit.ACQPS = 63;     // sample window min 320ns = 64 sysclk * (1/200MHz), 63 -> 64 cycles (n-1)
   AdcdRegs.ADCSOC5CTL.bit.CHSEL = 2;      // SOC4 will convert pin D2
//   AdcdRegs.ADCSOC3CTL.bit.TRIGSEL = 0;    // trigger software only
#endif

   AdcdRegs.ADCSOCPRICTL.bit.SOCPRIORITY = 6; // SOC0, SOC1, SOC2, SOC3, SOC4, and SOC5 are high priority. Lower number has priority
                                              // therefore since all fours SOCs are high priority and all are triggered by ePWM5,
                                              // they convert in order SOC0, SOC1, SOC2, SOC3, SOC4, and SOC5. After SOC5 conversion is done
                                              // ADC1INT is placed

   // Trigger the ADC interrupt after SOC5 conversion is done on ADC A
   AdcaRegs.ADCINTFLGCLR.bit.ADCINT1 = 1;   // make sure INT1 flag is cleared
   AdcaRegs.ADCINTSEL1N2.all = 0;           // clear ADCINTSEL1N2 register
//   AdcaRegs.ADCINTSEL1N2.bit.INT1E = 0;     // disable interrupt at this time, enabled in ePWM1 interrupt
   AdcaRegs.ADCINTSEL1N2.bit.INT1SEL = 5;   // 5 - EOC5 end conversion triggers ADC1INT
   AdcaRegs.ADCINTSEL1N2.bit.INT1CONT = 1;  // Continuous mode - no need to clear interrupt to generate next one
   AdcaRegs.ADCINTSEL3N4.all = 0;
   // Other ADCs do not generate interrupts
   AdcbRegs.ADCINTSEL1N2.all = 0;
   AdcbRegs.ADCINTSEL3N4.all = 0;
   AdccRegs.ADCINTSEL1N2.all = 0;
   AdccRegs.ADCINTSEL3N4.all = 0;
//   AdcdRegs.ADCINTSEL1N2.all = 0;
//   AdcdRegs.ADCINTSEL3N4.all = 0;

   EDIS;
   //delay for 1ms to allow ADC time to power up
   DELAY_US(1000);
}

//*****************************************************************************************
// DMA has to read result registers 0 and 1 on ADCA (0xB00, 0xB01), then result register
// 0 and 1 on ADCB (0x0B20 and 0xB21), then result registers 0 and 1 on ADCC (0x0B40 and
// 0x0B41), and then result registers 0and 1 on ADCD (0xB60 and 0xB61). The way to achieve
// this is to use wrap feature that will add extra offset in the source address every two
// reads.
//*****************************************************************************************
#if 0
void InitDMA(void)
{
   EALLOW;
   // Perform a hard reset on DMA
   DmaRegs.DMACTRL.all = 3;
   // Allow DMA to run free on emulation suspend
   DmaRegs.DEBUGCTRL.bit.FREE = 1;
   // Set Ch1 priority
   DmaRegs.PRIORITYCTRL1.bit.CH1PRIORITY = 1; // Channel 1 has the highest priority
   // Set up MODE Register:
   DmaRegs.CH1.MODE.all = 0;
   DmaRegs.CH1.MODE.bit.PERINTSEL = 1;       // These are legacy bits and should be set to the channel number
   DmaRegs.CH1.MODE.bit.PERINTE = 1;         // Peripheral Event Trigger Enable
   DmaRegs.CH1.MODE.bit.CHINTMODE = 1;       // Generate interrupt to CPU at end of transfer
   DmaRegs.CH1.MODE.bit.CONTINUOUS = 1;      // Continuous enable
   DmaRegs.CH1.MODE.bit.ONESHOT = 1;         // One shot enabled
   DmaRegs.CH1.MODE.bit.CHINTE = 1;          // Channel Interrupt to CPU enable
   // From Table 5-1. 2837xD Technical Reference, 37xD DSPs have DMA source selected in DmaClaSrcSelRegs.DMACHSRCSEL1
   DmaClaSrcSelRegs.DMACHSRCSEL1.bit.CH1 = 1; // 1 = ADCA.1  ADCA INT1  selected as the source for DMA

   // Set up CONTROL register
   DmaRegs.CH1.CONTROL.all = 0;
   DmaRegs.CH1.CONTROL.bit.PERINTCLR = 1;    // Clear any spurious interrupt flags
   DmaRegs.CH1.CONTROL.bit.ERRCLR = 1;       // Clear any spurious sync error flags

   PieCtrlRegs.PIEIER7.bit.INTx1 = 1;        // Enable DMA CH1 interrupt in PIE

   // Set up BURST registers:
   DmaRegs.CH1.BURST_SIZE.bit.BURSTSIZE = 7; // Number of words (x+1) x-ferred in a burst (two channels per ADC x 4 = 8 samples)
   DmaRegs.CH1.SRC_BURST_STEP = 1;           // Increment source address between each word x-ferred
   DmaRegs.CH1.DST_BURST_STEP = 1;           // Increment destination address between each word x-ferred
   // Set up TRANSFER registers:
   // Number of bursts (x+1) per transfer, DMA interrupt will occur after completed transfer
   DmaRegs.CH1.TRANSFER_SIZE = (unsigned int) Cal.OversamplingRate - 1;  // 4 burst for 4 oversampling sets
   // TRANSFER_STEP is ignored when WRAP occurs
   DmaRegs.CH1.SRC_TRANSFER_STEP = 0x19;     // source to the next ADC output register set 0x7 + 0x19 = 0x20
   DmaRegs.CH1.DST_TRANSFER_STEP = 1;        // destination to the next address

   // Channel 1 setup - moves ADC data
   // Set up SOURCE address:
   DmaRegs.CH1.SRC_BEG_ADDR_SHADOW = (unsigned long)0x0B00; // Point to beginning of source buffer
   DmaRegs.CH1.SRC_ADDR_SHADOW = (unsigned long)0x0B00;
   // Set up DESTINATION address:
   DmaRegs.CH1.DST_BEG_ADDR_SHADOW = (unsigned long)&AdcBuffer[0]; // Point to beginning of destination buffer
   DmaRegs.CH1.DST_ADDR_SHADOW = (unsigned long)&AdcBuffer[0];

   // Set up WRAP registers:
   DmaRegs.CH1.SRC_WRAP_SIZE = 0xFFFF;  // Wrapping action never happens since it never reaches zero
   DmaRegs.CH1.SRC_WRAP_STEP = 0;       // Step for source wrap
   DmaRegs.CH1.DST_WRAP_SIZE = 0xFFFF;  // Wrapping action never happens since it never reaches zero
   DmaRegs.CH1.DST_WRAP_STEP = 0;       // Step for destination wrap

   DmaRegs.CH1.CONTROL.bit.RUN = 0;     // Do not start DMA channel 1 yet.
   EDIS;
}
#endif

//---------------------------------------------------------------------------
// This function initializes the PIE control registers to a known state.
//
void InitPieCtrl(void)
{
    // Disable Interrupts at the CPU level:
    DINT;

    // Disable the PIE
    PieCtrlRegs.PIECTRL.bit.ENPIE = 0;

   // Clear all PIEIER registers:
   PieCtrlRegs.PIEIER1.all = 0;
   PieCtrlRegs.PIEIER2.all = 0;
   PieCtrlRegs.PIEIER3.all = 0;
   PieCtrlRegs.PIEIER4.all = 0;
   PieCtrlRegs.PIEIER5.all = 0;
   PieCtrlRegs.PIEIER6.all = 0;
   PieCtrlRegs.PIEIER7.all = 0;
   PieCtrlRegs.PIEIER8.all = 0;
   PieCtrlRegs.PIEIER9.all = 0;
   PieCtrlRegs.PIEIER10.all = 0;
   PieCtrlRegs.PIEIER11.all = 0;
   PieCtrlRegs.PIEIER12.all = 0;

   // Clear all PIEIFR registers:
   PieCtrlRegs.PIEIFR1.all = 0;
   PieCtrlRegs.PIEIFR2.all = 0;
   PieCtrlRegs.PIEIFR3.all = 0;
   PieCtrlRegs.PIEIFR4.all = 0;
   PieCtrlRegs.PIEIFR5.all = 0;
   PieCtrlRegs.PIEIFR6.all = 0;
   PieCtrlRegs.PIEIFR7.all = 0;
   PieCtrlRegs.PIEIFR8.all = 0;
   PieCtrlRegs.PIEIFR9.all = 0;
   PieCtrlRegs.PIEIFR10.all = 0;
   PieCtrlRegs.PIEIFR11.all = 0;
   PieCtrlRegs.PIEIFR12.all = 0;
}

void InitMemory(void)
{
   // GS0 to  GS7 assigned to CPU1
   // GS8 to GS14 assigned to CPU2
   // GS15        assigned to CPU1 and DMA access
   MemCfgRegs.GSxMSEL.all = 0x00007F00;
   // D26 = 0 => GS15 DMA writes are allowed
   // D25 = 1 => GS15 CPU writes are blocked
   // D24 = 1 => GS15 CPU fetches are blocked
   MemCfgRegs.GSxACCPROT3.all = 0x03000000;
}
