/* --COPYRIGHT--,BSD
 * Copyright (c) 2012, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/
#ifndef _DCBUS_H_
#define _DCBUS_H_

//! \file   ~sw/modules/dcbus/src/32b/dcbus.h
//! \brief  Contains the public interface to the 
//!         DC bus estimation (DCBUS) estimation module routines
//!
//! (C) Copyright 2012, Texas Instruments, Inc.


// **************************************************************************
// the includes

#include "sv/IQmathLib_fpu32.h"
#include "sv/filter_fo.h"
#include "sv/types.h"

//!
//! \defgroup DCBUS

//!
//! \ingroup DCBUS
//@{


#ifdef __cplusplus
extern "C" {
#endif


// **************************************************************************
// the defines


// **************************************************************************
// the typedefs

//! \brief Defines the DCBUS object
//!
typedef struct _DCBUS_Obj_
{
  _iq               oneOverDcBusVoltage_pu;     //!< the inverse DC bus voltage value, pu
  _iq               oneOverDcBusVoltage_min_pu; //!< the minimum inverse DC bus voltage value, pu
  _iq               oneOverDcBusVoltage_max_pu; //!< the maximum inverse DC bus voltage value, pu

  FILTER_FO_Handle  lpfHandle;                  //!< the low pass filter handle
  FILTER_FO_Obj     lpf;                        //!< the low pass filter object
} DCBUS_Obj;


//! \brief Defines the DCBUS handle
//!
typedef struct DCBUS_Obj   *DCBUS_Handle;


// **************************************************************************
// the globals


// **************************************************************************
// the function prototypes


//! \brief     Gets the per unit DC bus voltage value
//! \param[in] handle  The DC bus (DCBUS) estimation module handle
//! \return    The DC bus value, pu
static inline _iq DCBUS_getDcBusVoltage_pu(DCBUS_Handle handle)
{
  DCBUS_Obj *obj = (DCBUS_Obj *)handle;

  return(_IQdiv(_IQ(1.0),obj->oneOverDcBusVoltage_pu));
} // end of DCBUS_getDcBusVoltage_pu() function


//! \brief     Gets the maximum per unit DC bus voltage value
//! \param[in] handle  The DC bus (DCBUS) estimation module handle
//! \return    The maximum per unit DC bus voltage value, pu
static inline _iq DCBUS_getDcBusVoltage_max_pu(DCBUS_Handle handle)
{
  DCBUS_Obj *obj = (DCBUS_Obj *)handle;

  return(_IQdiv(_IQ(1.0),obj->oneOverDcBusVoltage_min_pu));
} // end of DCBUS_getDcBusVoltage_max_pu() function


//! \brief     Sets the minimum per unit DC bus voltage value
//! \param[in] handle  The DC bus (DCBUS) estimation module handle
//! \return    The minimum per unit DC bus voltage value, pu
static inline _iq DCBUS_getDcBusVoltage_min_pu(DCBUS_Handle handle)
{
  DCBUS_Obj *obj = (DCBUS_Obj *)handle;

  return(_IQdiv(_IQ(1.0),obj->oneOverDcBusVoltage_max_pu));
} // end of DCBUS_getDcBusVoltage_min_pu() function


//! \brief     Gets the maximum inverse DC bus voltage value
//! \param[in] handle  The DC bus (DCBUS) estimation module handle
//! \return    The maximum inverse DC bus voltage value, pu
static inline _iq DCBUS_getOneOverDcBusVoltage_max_pu(DCBUS_Handle handle)
{
  DCBUS_Obj *obj = (DCBUS_Obj *)handle;

  return(obj->oneOverDcBusVoltage_max_pu);
} // end of DCBUS_getOneOverDcBusVoltage_max_pu() function


//! \brief     Gets the minimum inverse DC bus voltage value
//! \param[in] handle  The DC bus (DCBUS) estimation module handle
//! \return    The minimum inverse DC bus voltage value, pu
static inline _iq DCBUS_getOneOverDcBusVoltage_min_pu(DCBUS_Handle handle)
{
  DCBUS_Obj *obj = (DCBUS_Obj *)handle;

  return(obj->oneOverDcBusVoltage_min_pu);
} // end of DCBUS_getOneOverDcBusVoltage_min_pu() function


//! \brief     Gets the inverse DC bus voltage
//! \param[in] handle  The DC bus (DCBUS) estimation module handle
//! \return    The inverse DC bus voltage value, pu
static inline _iq DCBUS_getOneOverDcBusVoltage_pu(DCBUS_Handle handle)
{
  DCBUS_Obj *obj = (DCBUS_Obj *)handle;

  return(obj->oneOverDcBusVoltage_pu);
} // end of DCBUS_getOneOverDcBusVoltage_pu() function


//! \brief     Initializes the DC bus (DCBUS) module
//! \param[in] pMemory   A pointer to the memory for the DC bus object
//! \param[in] numBytes  The number of bytes allocated for the DC bus object, bytes
//! \return The DC bus (DCBUS) object handle
extern DCBUS_Handle DCBUS_init(void *pMemory,const size_t numBytes);


//! \brief     Sets the DC bus voltage value
//! \param[in] handle           The DC bus (DCBUS) estimation module handle
//! \param[in] dcBusVoltage_pu  The per unit DC bus voltage value, pu
static inline void DCBUS_setDcBusVoltage_pu(DCBUS_Handle handle,const _iq dcBusVoltage_pu)
{
  DCBUS_Obj *obj = (DCBUS_Obj *)handle;

  obj->oneOverDcBusVoltage_pu = _IQdiv(_IQ(1.0),dcBusVoltage_pu);

  return;
} // end of DCBUS_setDcBusVoltage_pu() function


//! \brief     Sets the maximum DC bus voltage value
//! \param[in] handle               The DC bus (DCBUS) estimation module handle
//! \param[in] dcBusVoltage_max_pu  The maximum per unit DC bus voltage value, pu
static inline void DCBUS_setDcBusVoltage_max_pu(DCBUS_Handle handle,const _iq dcBusVoltage_max_pu)
{
  DCBUS_Obj *obj = (DCBUS_Obj *)handle;

  obj->oneOverDcBusVoltage_min_pu = _IQdiv(_IQ(1.0),dcBusVoltage_max_pu);

  return;
} // end of DCBUS_setDcBusVoltage_max_pu() function


//! \brief     Sets the minimum DC bus voltage value
//! \param[in] handle               The DC bus (DCBUS) estimation module handle
//! \param[in] dcBusVoltage_min_pu  The minimum per unit DC bus voltage value, pu
static inline void DCBUS_setDcBusVoltage_min_pu(DCBUS_Handle handle,const _iq dcBusVoltage_min_pu)
{
  DCBUS_Obj *obj = (DCBUS_Obj *)handle;

  obj->oneOverDcBusVoltage_max_pu = _IQdiv(_IQ(1.0),dcBusVoltage_min_pu);

  return;
} // end of DCBUS_setDcBusVoltage_min_pu() function


//! \brief     Sets the low pass filter parameters
//!
//!            y[n] = b0*x[n] + b1*x[n-1] - a1*y[n-1]
//!
//! \param[in] handle  The DC bus (DCBUS) estimation module handle
//! \param[in] b0      The filter coefficient value for z^0
//! \param[in] b1      The filter coefficient value for z^(-1)
//! \param[in] a1      The filter coefficient value for z^(-1)
//! \param[in] x1      The input value at time sample n=-1
//! \param[in] y1      The output value at time sample n=-1
extern void DCBUS_setLpfParams(DCBUS_Handle handle,
                               const _iq b0,const _iq b1,const _iq a1,
                               const _iq x1,const _iq y1);


//! \brief     Sets the inverse DC bus voltage value
//! \param[in] handle                The DC bus (DCBUS) estimation module handle
//! \param[in] oneOverDcBusValue_pu  The inverse DC bus voltage value, pu
static inline void DCBUS_setOneOverDcBusVoltage_pu(DCBUS_Handle handle,const _iq oneOverDcBusVoltage_pu)
{
  DCBUS_Obj *obj = (DCBUS_Obj *)handle;

  obj->oneOverDcBusVoltage_pu = oneOverDcBusVoltage_pu;

  return;
} // end of DCBUS_setOneOverDcBusVoltage_pu() function


//! \brief     Sets the maximum inverse DC bus voltage value
//! \param[in] handle                    The DC bus (DCBUS) estimation module handle
//! \param[in] oneOverDcBusValue_max_pu  The maximum inverse DC bus voltage value, pu
static inline void DCBUS_setOneOverDcBusVoltage_max_pu(DCBUS_Handle handle,
                                                       const _iq oneOverDcBusVoltage_max_pu)
{
  DCBUS_Obj *obj = (DCBUS_Obj *)handle;

  obj->oneOverDcBusVoltage_max_pu = oneOverDcBusVoltage_max_pu;

  return;
} // end of DCBUS_setOneOverDcBusVoltage_max_pu() function


//! \brief     Sets the minimum inverse DC bus voltage value
//! \param[in] handle                    The DC bus (DCBUS) estimation module handle
//! \param[in] oneOverDcBusValue_min_pu  The minimum inverse DC bus voltage value, pu
static inline void DCBUS_setOneOverDcBusVoltage_min_pu(DCBUS_Handle handle,
                                                       const _iq oneOverDcBusVoltage_min_pu)
{
  DCBUS_Obj *obj = (DCBUS_Obj *)handle;

  obj->oneOverDcBusVoltage_min_pu = oneOverDcBusVoltage_min_pu;

  return;
} // end of DCBUS_setOneOverDcBusVoltage_min_pu() function


//! \brief     Sets the DC bus parameters
//! \param[in] handle               The DC bus (DCBUS) estimation module handle
//! \param[in] dcBusVoltage_pu      The per unit DC bus voltage, pu
//! \param[in] dcBusVoltage_min_pu  The minimum per unit DC bus voltage, pu
//! \param[in] dcBusVoltage_max_pu  The maximum per unit DC bus voltage, pu
extern void DCBUS_setParams(DCBUS_Handle handle,
                            const _iq dcBusVoltage_pu,
                            const _iq dcBusVoltage_min_pu,
                            const _iq dcBusVoltage_max_pu);


//! \brief     Runs the DC bus estimator
//! \param[in] handle           The DC bus (DCBUS) estimation module handle
//! \param[in] dcBusVoltage_pu  The per unit DC bus voltage, pu
static inline void DCBUS_run(DCBUS_Handle handle,const _iq dcBusVoltage_pu)
{
  DCBUS_Obj *obj = (DCBUS_Obj *)handle;
  _iq oneOverDcBusVoltage_m1_pu = DCBUS_getOneOverDcBusVoltage_pu(handle);
  _iq oneOverDcBusVoltage_pu,oneOverDcBusVoltage_lpf_pu;
  _iq min_pu = DCBUS_getOneOverDcBusVoltage_min_pu(handle);
  _iq max_pu = DCBUS_getOneOverDcBusVoltage_max_pu(handle);

  // estimate oneOverDcBusVoltage, oneOverDcBusVoltage[n]
  oneOverDcBusVoltage_pu = _IQmpy(_IQmpy(dcBusVoltage_pu,oneOverDcBusVoltage_m1_pu),
                                  oneOverDcBusVoltage_m1_pu);

  // filter the value
  oneOverDcBusVoltage_lpf_pu = FILTER_FO_run_form_0(obj->lpfHandle,oneOverDcBusVoltage_pu);

  // saturate the result
  oneOverDcBusVoltage_lpf_pu = _IQsat(oneOverDcBusVoltage_lpf_pu,max_pu,min_pu);

  // store the value
  DCBUS_setOneOverDcBusVoltage_pu(handle,oneOverDcBusVoltage_lpf_pu);

  // store the saturated value back into the filter
  FILTER_FO_set_y1(obj->lpfHandle,oneOverDcBusVoltage_lpf_pu);

  return;
} // end of DCBUS_run() function


#ifdef __cplusplus
}
#endif // extern "C"

//@}  // ingroup

#endif // end of _DCBUS_H_ definition

