//------------------------------------------------------------------------------
//	XCLKOUT_test - Set XCLKOUT clock source as XTALOSC and set a divider of 1.
//	Configure GPIO73 in PinMux to monitor XTALOSC, which operates at 20MHz on the LaunchXL board.
//	Since GPIO73 is not accessible on a LaunchXL-F28379D board,
//	mirror state of GPIO73 on GPIO15 so we can verify that XCLKOUT is working
//	by attaching an oscilloscope to pin GPIO15.
//------------------------------------------------------------------------------

#include "C2000types.h"
#include "F2837xD_sysctrl.h"
#include "F2837xD_gpio.h"
#include "pin_map.h"
#include "sysctl.h"
#include "gpio.h"
#include "xbar.h"

typedef enum ePinState_e
{
	GPIO_PIN_LOW, GPIO_PIN_HIGH, MAX_PIN_STATES
} PIN_STATE_t;

#define GPIO_PIN_LED1	31U  // GPIO pin number for LD10 on LaunchXL board.

//	Function prototypes:
void XtalClkInit( void );
void GpioInitPins( void );
void gpioInit( GPIO_Direction ePinDir, uint32_t nPinNo, PIN_STATE_t nPinState );
void SWTimeDlyMicroSec_Asm( uint32_t nNumMicrosec );
void LedBlinkUpdate( void );

//------------------------------------------------------------------------------
//	main.c
//------------------------------------------------------------------------------
int main( void )
{
	bool bTestDone = false;

	XtalClkInit();
	GpioInitPins();

	//	Configure XBar to mirror activity on GPIO73 to GPIO15.
	XBAR_setInputPin( XBAR_INPUT7, 73U );		//	Using XBarInputNum: XBAR_INPUT7, //!< eCAP1, X-BARsfrom ECAP1
	XBAR_setOutputMuxConfig( XBAR_OUTPUT4, XBAR_OUT_MUX00_ECAP1_OUT );
	XBAR_enableOutputMux( XBAR_OUTPUT4, XBAR_MUX00 );
	GPIO_setPinConfig( GPIO_15_OUTPUTXBAR4 );	//	We can access GPIO15 at OUTPUTXBAR4.

	while( !bTestDone )
	{	//	Demonstrate that the program IS running.
		LedBlinkUpdate();
	}

	return 0;
}

//----------------------------------------------------------------------------
//! Initialize XCLKOUT clock source to the XTALOSC external oscillator.
//
//! Set XCLKOUT source as XTALOSC and divider 1.
//! \param[in]  none
//! \return     none
//# REF-??
//----------------------------------------------------------------------------
void XtalClkInit( void )
{
	//	Configure XTAL oscillator to drive XCLKOUT.
	//	This should produce XCLKOUT at 5 MHz on a LaunchXL-F28379D board,
	//	and 10 MHz on most other F28379D based boards.
	SysCtl_selectClockOutSource( SYSCTL_CLOCKOUT_XTALOSC );
	SysCtl_setXClk( SYSCTL_XCLKOUT_DIV_1 );
}

//----------------------------------------------------------------------------
//! Initialize GPIO output pins.
//
//! Set GPIO output pins to their initial states.
//! \param[in]  none
//! \return     none
//----------------------------------------------------------------------------
void GpioInitPins( void )
{
	//  Configure DSP GPIO pin 73 to output the XCLKOUT clock signal.
	GPIO_setPinConfig( GPIO_73_XCLKOUT );
	//	The LaunchXL-F28379D demo board does NOT bring GPIO73 to an accessible pin.
	//	So try using XBar to mirror GPIO73 activity on GPIO15.
	gpioInit( GPIO_DIR_MODE_OUT, 15U, GPIO_PIN_LOW );
	//	Define GPIO pin 31 which controls LED10 on a LaunchXL board.
	gpioInit( GPIO_DIR_MODE_OUT, GPIO_PIN_LED1, GPIO_PIN_LOW );
}

void gpioInit( GPIO_Direction ePinDir, uint32_t nPinNo, PIN_STATE_t nPinState )
{
	GPIO_setDirectionMode( nPinNo, ePinDir );
	if( ePinDir == GPIO_DIR_MODE_OUT )
	{
		GPIO_writePin( nPinNo, (uint32_t) nPinState );
	}
}

void LedBlinkUpdate( void )
{
	static uint32_t nBlinkDelayCount = 0;
	static PIN_STATE_t nPinState = GPIO_PIN_LOW;

	nBlinkDelayCount++;
	if( nBlinkDelayCount >= 10000U )
	{
		nPinState = ( nPinState == GPIO_PIN_LOW ) ? GPIO_PIN_HIGH : GPIO_PIN_LOW;
		GPIO_writePin( GPIO_PIN_LED1, (uint32_t) nPinState );
		GPIO_writePin( 15U, (uint32_t) nPinState );

		nBlinkDelayCount = 0;
	}
}

