//###########################################################################
//	
//  FILE:   adcB_soc_epwm_cpu01.c
//	
//  TITLE:  ADC triggering via epwm for F2837xD.
//	
//! \addtogroup cpu01_example_list
//! <h1> ADC ePWM Triggering (adc_soc_epwm)</h1>
//!
//! This example sets up the ePWM to periodically trigger the ADC.
//!
//! After the program runs, the memory will contain:\n
//! - \b AdcbResults \b: A sequence of analog-to-digital conversion samples from
//! pin B5. The time between samples is determined based on the period
//! of the ePWM timer.
//	
//!	For Triumph. This ADC example project has been modified to read ADCB input channel 5.
//!	2024-04-08 - TVC.
//###########################################################################
//	
//	$Release Date:  $
//	$Copyright:
//	Copyright (C) 2013-2022 Texas Instruments Incorporated - http://	www.ti.com/
//	
//	Redistribution and use in source and binary forms, with or without 
//	modification, are permitted provided that the following conditions 
//	are met:
//	
//	  Redistributions of source code must retain the above copyright 
//	  notice, this list of conditions and the following disclaimer.
//	
//	  Redistributions in binary form must reproduce the above copyright
//	  notice, this list of conditions and the following disclaimer in the 
//	  documentation and/or other materials provided with the   
//	  distribution.
//	
//	  Neither the name of Texas Instruments Incorporated nor the names of
//	  its contributors may be used to endorse or promote products derived
//	  from this software without specific prior written permission.
//	
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
//	"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
//	LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
//	A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
//	OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
//	SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
//	LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
//	DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
//	THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
//	(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
//	OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//	$
//	###########################################################################

//	
//	Included Files
//	
#include "F28x_Project.h"

//	
//	Function Prototypes
//	
void ConfigureADC( void );
void ConfigureEPWM( void );
void SetupADCEpwm( Uint16 channel );
interrupt void adcb1_isr( void );

//	
//	Defines
//	
#define RESULTS_BUFFER_SIZE 256

//	
//	Globals
//	
Uint16 AdcbResults[RESULTS_BUFFER_SIZE];
Uint16 resultsIndex;
volatile Uint16 bufferFull;

void main( void )
{
//	
//	Step 1. Initialize System Control:
//	PLL, WatchDog, enable Peripheral Clocks
//	This example function is found in the F2837xD_SysCtrl.c file.
//	
	InitSysCtrl();

//	
//	Step 2. Initialize GPIO:
//	This example function is found in the F2837xD_Gpio.c file and
//	illustrates how to set the GPIO to it's default state.
//	
	InitGpio(); //	Skipped for this example

//	
//	Step 3. Clear all interrupts and initialize PIE vector table:
//	Disable CPU interrupts
//	
	DINT;

//	
//	Initialize the PIE control registers to their default state.
//	The default state is all PIE interrupts disabled and flags
//	are cleared.
//	This function is found in the F2837xD_PieCtrl.c file.
//	
	InitPieCtrl();

//	
//	Disable CPU interrupts and clear all CPU interrupt flags:
//	
	IER = 0x0000;
	IFR = 0x0000;

//	
//	Initialize the PIE vector table with pointers to the shell Interrupt
//	Service Routines (ISR).
//	This will populate the entire table, even if the interrupt
//	is not used in this example.  This is useful for debug purposes.
//	The shell ISR routines are found in F2837xD_DefaultIsr.c.
//	This function is found in F2837xD_PieVect.c.
//	
	InitPieVectTable();

//	
//	Map ISR functions
//	
	EALLOW;
	PieVectTable.ADCB1_INT = &adcb1_isr; //	function for ADCB interrupt 1
	EDIS;

//	
//	Configure the ADC and power it up
//	
	ConfigureADC();

//	
//	Configure the ePWM
//	
	ConfigureEPWM();

//	
//	Setup the ADC for ePWM triggered conversions on channel 0
//	
	SetupADCEpwm( 5 );

//	
//	Enable global Interrupts and higher priority real-time debug events:
//	
	IER |= M_INT1; //	Enable group 1 interrupts
	EINT;
	//	Enable Global interrupt INTM
	ERTM;
	//	Enable Global realtime interrupt DBGM

//	
//	Initialize results buffer
//	
	for( resultsIndex = 0; resultsIndex < RESULTS_BUFFER_SIZE; resultsIndex++ )
	{
		AdcbResults[resultsIndex] = 0;
	}
	resultsIndex = 0;
	bufferFull = 0;

//	
//	enable PIE interrupt
//	
	PieCtrlRegs.PIEIER1.bit.INTx1 = 1;

//	
//	sync ePWM
//	
	EALLOW;
	CpuSysRegs.PCLKCR0.bit.TBCLKSYNC = 1;

//	
//	take conversions indefinitely in loop
//	
	do
	{
		//	
		//	start ePWM
		//	
		EPwm1Regs.ETSEL.bit.SOCAEN = 1;  //	enable SOCA
		EPwm1Regs.TBCTL.bit.CTRMODE = 0; //	unfreeze, and enter up count mode

		//	
		//	wait while ePWM causes ADC conversions, which then cause interrupts,
		//	which fill the results buffer, eventually setting the bufferFull
		//	flag
		//	
		while( !bufferFull )
			;
		bufferFull = 0; //	clear the buffer full flag

		//	
		//	stop ePWM
		//	
		EPwm1Regs.ETSEL.bit.SOCAEN = 0;  //	disable SOCA
		EPwm1Regs.TBCTL.bit.CTRMODE = 3; //	freeze counter

		//	
		//	at this point, AdcbResults[] contains a sequence of conversions
		//	from the selected channel
		//	

		//	
		//	software breakpoint, hit run again to get updated conversions
		//	
		asm("   ESTOP0");
	}
	while( 1 );
}

//	
//	ConfigureADC - Write ADC configurations and power up the ADC for ADC B ONLY.
//	
void ConfigureADC( void )
{
	EALLOW;

	//	
	//	write configurations
	//	
	AdcbRegs.ADCCTL2.bit.PRESCALE = 6; //	set ADCCLK divider to /4
	AdcSetMode( ADC_ADCB, ADC_RESOLUTION_12BIT, ADC_SIGNALMODE_SINGLE );

	//	
	//	Set pulse positions to late
	//	
	AdcbRegs.ADCCTL1.bit.INTPULSEPOS = 1;

	//	
	//	power up the ADC
	//	
	AdcbRegs.ADCCTL1.bit.ADCPWDNZ = 1;

	//	
	//	delay for 1ms to allow ADC time to power up
	//	
	DELAY_US( 1000 );

	EDIS;
}

//	
//	ConfigureEPWM - Configure EPWM SOC and compare values
//	
void ConfigureEPWM( void )
{
	EALLOW;
	//	Assumes ePWM clock is already enabled
	EPwm1Regs.ETSEL.bit.SOCAEN = 0;			//	Disable SOC on A group
	EPwm1Regs.ETSEL.bit.SOCASEL = 4;		//	Select SOC on up-count
	EPwm1Regs.ETPS.bit.SOCAPRD = 1;			//	Generate pulse on 1st event
	EPwm1Regs.CMPA.bit.CMPA = 0x0800;		//	Set compare A value to 2048 counts
	EPwm1Regs.TBPRD = 0x1000;				//	Set period to 4096 counts
	EPwm1Regs.TBCTL.bit.CTRMODE = 3;		//	freeze counter
	EDIS;
}

//	
//	SetupADCEpwm - Setup ADC EPWM acquisition window
//	
void SetupADCEpwm( Uint16 channel )
{
	Uint16 acqps;

	//	
	//	Determine minimum acquisition window (in SYSCLKS) based on resolution
	//	
	if( ADC_RESOLUTION_12BIT == AdcbRegs.ADCCTL2.bit.RESOLUTION )
	{
		acqps = 14; //	75ns
	}
	else //	resolution is 16-bit
	{
		acqps = 63; //	320ns
	}

	//	
	//	Select the channels to convert and end of conversion flag
	//	
	EALLOW;
	AdcbRegs.ADCSOC0CTL.bit.CHSEL = channel;	//	SOC0 will convert pin B5
	AdcbRegs.ADCSOC0CTL.bit.ACQPS = acqps;		//	sample window is 100 SYSCLK cycles
	AdcbRegs.ADCSOC0CTL.bit.TRIGSEL = 5;		//	trigger on ePWM1 SOCA/C
	AdcbRegs.ADCINTSEL1N2.bit.INT1SEL = 0;		//	end of SOC0 will set INT1 flag
	AdcbRegs.ADCINTSEL1N2.bit.INT1E = 1;		//	enable INT1 flag
	AdcbRegs.ADCINTFLGCLR.bit.ADCINT1 = 1;		//	make sure INT1 flag is cleared
	EDIS;
}

//	
//	adcb1_isr - Read ADC results Buffer in ISR.
//	
interrupt void adcb1_isr( void )
{
	AdcbResults[resultsIndex++] = AdcbResultRegs.ADCRESULT0;
	if( RESULTS_BUFFER_SIZE <= resultsIndex )
	{
		resultsIndex = 0;
		bufferFull = 1;
	}

	AdcbRegs.ADCINTFLGCLR.bit.ADCINT1 = 1;		//	clear INT1 flag

	//	
	//	Check if overflow has occurred
	//	
	if( 1 == AdcbRegs.ADCINTOVF.bit.ADCINT1 )
	{
		AdcbRegs.ADCINTOVFCLR.bit.ADCINT1 = 1;	//	clear INT1 overflow flag
		AdcbRegs.ADCINTFLGCLR.bit.ADCINT1 = 1;	//	clear INT1 flag
	}

	PieCtrlRegs.PIEACK.all = PIEACK_GROUP1;
}

//	
//	End of file
//	
