/**
 * @file board.c
 * @brief
 *
 * @author  Firmware department
 * @copyright Ingenia Motion Control (c) 2016. All rights reserved.
 */

#include "board.h"
#include "board_mapping.h"
#include "board_config.h"
#include "tick_time.h"
#include "flash_mng.h"
#include "ip_communications.h"
#include "input_output.h"
#include "spi_interface.h"
#include "i2c_interface.h"

void BoardInit(void)
{
    InitSysCtrl();

    FlashInit(SYSTEM_FREQ_MHZ);

#if defined(CPU1)
    /* Configure MCB_SYNC1 pin as input */
    InOutConfig(PIN7, INPUT_PULLUP_MODE);
#endif
}

void BoardBootInit(void)
{
    uint16_t u16Intfs;
    uint32_t u32Password[4] =
            {
              (uint32_t)0x11223344UL,
              (uint32_t)0x55667788UL,
              (uint32_t)0x99AABBCCUL,
              (uint32_t)0xDDEEFF00UL
            };

    BoardUnlock(u32Password);

#if defined(CPU1)
    IpCommsInit(CPU_1);
#endif
#if defined(CPU2)
    IpCommsInit(CPU_2);
#endif

    IntMasterEnable();

#if NUMBER_OF_SPI > 0
    for (u16Intfs = 0; u16Intfs < (uint16_t)NUMBER_OF_SPI; u16Intfs++)
    {
        SPIInit(u16Intfs);
    }
#endif

#if NUMBER_OF_I2C > 0
    for (u16Intfs = 0; u16Intfs < (uint16_t)NUMBER_OF_NVM; u16Intfs++)
    {
        I2CInit(u16Intfs);
    }
#endif

    TimeInit(100);

    EINT;
    ERTM;
}

void BoardReset(void)
{
    EALLOW;
    /* enable watchdog and set clk divide by 64 */
    WdRegs.WDCR.all = 0x002F;
    /* disable bit change of WDDIS */
    WdRegs.SCSR.bit.WDOVERRIDE = 1;
    EDIS;

    /* wait for your reset */
    for (;;) ;
}

uint16_t BoardUnlock(uint32_t* p32Data)
{
    uint16_t u16Err = BOARD_OK;
    uint32_t u32Password0, u32Password1, u32Password2, u32Password3;
    uint16_t u16Idx;
    long int temp;
    volatile long int *CSMPWL = (volatile long int *)0x78028;

    /* TI recommended dummy read sequence */
    for (u16Idx = 0; u16Idx < 4; u16Idx++)
    {
        temp = *CSMPWL++;
    }

    EALLOW;

//    /* TI recommended dummy read sequence */
//    u32Password0 = DcsmZ1Regs.Z1_CSMKEY0;
//    u32Password1 = DcsmZ1Regs.Z1_CSMKEY1;
//    u32Password2 = DcsmZ1Regs.Z1_CSMKEY2;
//    u32Password3 = DcsmZ1Regs.Z1_CSMKEY3;

    /*
     * Forcing to use password variables to
     * avoid compilation warning
     */
    u32Password0 = p32Data[0];
    u32Password1 = p32Data[1];
    u32Password2 = p32Data[2];
    u32Password3 = p32Data[3];

    /* Unlock the device */
    DcsmZ1Regs.Z1_CSMKEY0 = u32Password0;
    DcsmZ1Regs.Z1_CSMKEY1 = u32Password1;
    DcsmZ1Regs.Z1_CSMKEY2 = u32Password2;
    DcsmZ1Regs.Z1_CSMKEY3 = u32Password3;
    EDIS;

    /* Check lock status */
    if (DcsmZ1Regs.Z1_CR.bit.UNSECURE == 0)
    {
        u16Err = BOARD_LOCKED;
    }
    return u16Err;
}

#if defined(CPU1)
void BoardSetCPUMapping(void)
{
    uint16_t u16Intfs;

#if NUMBER_OF_SPI > 0
    for (u16Intfs = 0; u16Intfs < (uint16_t) NUMBER_OF_SPI;
            u16Intfs++)
    {
        SPICPU2Mapping(u16Intfs);
    }
#endif

#if NUMBER_OF_I2C > 0
    for (u16Intfs = 0; u16Intfs < (uint16_t) NUMBER_OF_I2C;
            u16Intfs++)
    {
        I2CCPU2Mapping(u16Intfs);
    }
#endif

}
#endif

void BoardStartOtherCPU(void)
{
#if defined(CPU1)
    IpCommsRunOtherCPU(CPU_1);
#endif
#if defined(CPU2)
    IpCommsRunOtherCPU(CPU_2);
#endif
#if defined(CPU1)
    IpCommsWaitOtherCPU(CPU_1);
#endif
}
