/**
 * @file board.h
 * @brief This file offers a abstract layer that allows to initialize
 *        and execute basic board level functions
 *
 * @author  Firmware department
 * @copyright Ingenia Motion Control (c) 2016. All rights reserved.
 */

/**
 * \defgroup Board Board API
 *
 * @{
 */

#ifndef BOARD_H
#define BOARD_H

#include <stdint.h>

#ifdef __cplusplus
extern "C"
{
#endif

/** States of the board after self-test process */
typedef enum
{
    /** Board operates correctly */
    BOARD_OK = 0,
    /** Board - MCU unlock fails */
    BOARD_LOCKED,
    /** Some module is not working as expected */
    BOARD_KO
} EBoardStatus;

/** Initializes minimum elements for MCU operation */
void
BoardInit(void);

/** Initializes all elements needed for bootloader */
void
BoardBootInit(void);

/** Force a MCU reset */
void
BoardReset(void);

/**
 * Unlocks MCU for erase/program flash operations
 *
 * @param[in] pData
 *  BYTE array containing MCU unlock password
 * @note This array will be a WORD array in future versions
 *
 * @retval BOARD_OK if MCU is unlock correctly
 *         BOARD_LOCKED otherwise
 */
uint16_t
BoardUnlock(uint32_t* p32Data);

#if defined(CPU1)
/**
 * Gives the ownership of the bootloader peripherals
 * to CPU2
 */
void
BoardSetCPUMapping(void);
#endif

/**
 * Activates internal flags to enable operation of the
 * other CPU
 */
void
BoardStartOtherCPU(void);

#ifdef __cplusplus
}
#endif

#endif /* BOARD_H */

/** @} */
