/**
 * @file crc_tool.h
 * @brief This file offers a all CRC server service over serial
 *
 * @author  Firmware department
 * @copyright Ingenia Motion Control (c) 2016. All rights reserved.
 */

/**
 * \addtogroup SerialCRC Serial CRC
 *
 * @{
 *
 *   CRC object implementation for High speed serial protocol
 */

#ifndef CRC_TOOL_H
#define CRC_TOOL_H

#include <stdint.h>
#include "vcu2_crc.h"

#ifdef __cplusplus
extern "C"
{
#endif

/** Supported CRC types */
typedef enum
{
    CRC_CCITT_XMODEM = 0,
    CRC_8_C2
}ECRCType;

/** CRC instance */
typedef struct
{
    /** CRC type */
    ECRCType   eType;
    /** Crc object */
    CRC_Obj    CRC;
} TCrcIntf;

/**
 * Initializes a CRC instance.
 *
 * @note This module is a component of some of the main board
 *       modules so it doesn't follow the main flow of the board
 *       layer. Therefore, handler doesn't exist and initialization
 *       is executed directly on the argument instance.
 *
 * @param[out] u16Id
 *  Target instance to be initialized. @see TSerialSDOInstance
 */
void
CRCInit(TCrcIntf *ptInst);

/**
 * Deinitializes a CRC instance.
 *
 * @param[out] u16Id
 *  Target instance to be deinitialized. @see TSerialSDOInstance
 */
void
CRCDeinit(TCrcIntf *ptInst);

/**
 * Computes the CRC value of the input buffer
 *
 * @note In CRC_8_C2 pu16Bfr works as a word buffer, being
 *       the first byte input on the LSB position of the word 0,
 *       the second byte the MSB position of the word 0,
 *       the third byte the LSB position of the word 1, and so on.
 *       In CRC_CCITT_XMODEM the first byte input is the MSB of
 *       the word 0, the second byte the LSB position of the
 *       word 0, the third byte the MSB position of the word 1,
 *       and so on.
 *
 * @param[out] ptInst
 *  Target Instance. @see TCRCInstance
 * @param[in] pu16Bfr
 *  Data buffer
 * @param[in] u32Size
 *  Size of the data buffer
 *
 * @retval Crc result
 */
uint16_t
CRCProcess(TCrcIntf *ptInst, const uint16_t *pu16Bfr, uint32_t u32Size);

#ifdef __cplusplus
}
#endif

#endif /* SERIAL_CRC_H */

/** @} */
/** @} */
