#include "flash_mng.h"

#include "F28x_Project.h"
#include "F2837xD_Ipc_drivers.h"
#include "F021_F2837xD_C28x.h"

/* Constants ----- -----------------------------------------------------------*/
#define LATCH_BUF_SIZE      4

typedef union
{
    uint16_t u16Array[LATCH_BUF_SIZE];
    uint32_t u32Array[LATCH_BUF_SIZE >> 1];
    uint64_t u64Value;
} UFlashHalfBank;

static UFlashHalfBank uLastValue;

/* Static Functions ----------------------------------------------------------*/
/**
 * Functions defined in the TI library that needs to be populated in user code
 */
Fapi_StatusType Fapi_serviceWatchdogTimer(void);
Fapi_StatusType Fapi_setupEepromSectorEnable(void);
Fapi_StatusType Fapi_setupBankSectorEnable(void);

/* Functions -----------------------------------------------------------------*/
#pragma CODE_SECTION(Fapi_serviceWatchdogTimer, "ramfuncs");
Fapi_StatusType Fapi_serviceWatchdogTimer(void)
{
   return (Fapi_Status_Success);
}

#pragma CODE_SECTION(Fapi_setupEepromSectorEnable, "ramfuncs");
Fapi_StatusType Fapi_setupEepromSectorEnable(void)
{
    /*
     * Value must be 0xFFFF to enable erase and programming of the
     * EEPROM bank, 0 to disable
     */
    Fapi_GlobalInit.m_poFlashControlRegisters->Fbse.u32Register = 0xFFFF;

    /*
     * Enables sectors 32-63 for bank and sector erase
     */
    FAPI_WRITE_LOCKED_FSM_REGISTER(Fapi_GlobalInit.m_poFlashControlRegisters->
                                   FsmSector.u32Register, 0x0U);

    /*
     * Enables sectors 0-31 for bank and sector erase
     */
    FAPI_WRITE_LOCKED_FSM_REGISTER(Fapi_GlobalInit.m_poFlashControlRegisters->
                                   FsmSector1.u32Register, 0x0U);

    /*
     * Enables sectors 32-63 for bank and sector erase
     */
    FAPI_WRITE_LOCKED_FSM_REGISTER(Fapi_GlobalInit.m_poFlashControlRegisters->
                                   FsmSector2.u32Register, 0x0U);

    return (Fapi_Status_Success);
}

#pragma CODE_SECTION(Fapi_setupBankSectorEnable, "ramfuncs");
Fapi_StatusType Fapi_setupBankSectorEnable(void)
{
    /*
     * Enable sectors 0-15 for erase and programming
     */
    Fapi_GlobalInit.m_poFlashControlRegisters->Fbse.u32Register = 0xFFFF;
    FAPI_WRITE_LOCKED_FSM_REGISTER(Fapi_GlobalInit.m_poFlashControlRegisters->
                                   FsmSector.u32Register, 0x0U);

    return (Fapi_Status_Success);
}

#pragma CODE_SECTION(FlashInit, "ramfuncs");
int32_t
FlashInit(uint32_t u32SystemClock)
{
    int32_t i32Err = FLASH_OK;
    Fapi_StatusType status;

    SeizeFlashPump();

    EALLOW;

    Flash0EccRegs.ECC_ENABLE.bit.ENABLE = 0x0;

    status = Fapi_initializeAPI(F021_CPU0_BASE_ADDRESS, u32SystemClock);
    if (status != Fapi_Status_Success)
    {
        /* API cannot be initialized */
        i32Err = FLASH_API_NOT_INIT;
    }

    if (i32Err == FLASH_OK)
    {
        status = Fapi_setActiveFlashBank(Fapi_FlashBank0);

        if (status != Fapi_Status_Success)
        {
            /* Bank access failed */
            i32Err = FLASH_BANK_ACC_PROBLEM;
        }
    }

    Flash0EccRegs.ECC_ENABLE.bit.ENABLE = 0xA;
    EDIS;

    uLastValue.u64Value = (uint64_t)0xFFFFFFFFFFFFFFFFULL;

    ReleaseFlashPump();

    return i32Err;
}

#pragma CODE_SECTION(FlashInit, "ramfuncs");
int32_t
FlashDeInit()
{
    return FLASH_NOT_IMPLEMENTED;
}

#pragma CODE_SECTION(FlashErase, "ramfuncs");
int32_t
FlashErase(uint32_t u32Address)
{
    int32_t i32Err = FLASH_OK;
    Fapi_FlashStatusWordType statusWord;
    Fapi_StatusType status;
    uint32_t u32SectAdd;
    uint32_t u32Size;

    i32Err = FlashFindSector(u32Address, &u32SectAdd);
    if (i32Err == FLASH_OK)
    {
        i32Err == FlashFindSectorSize(u32SectAdd, &u32Size);
    }

    if (i32Err == FLASH_OK)
    {
        SeizeFlashPump();

        EALLOW;

        status = Fapi_issueAsyncCommandWithAddress(Fapi_EraseSector,
                                                  (uint32 *)u32SectAdd);

        while (Fapi_checkFsmForReady() == Fapi_Status_FsmBusy);

        if (Fapi_getFsmStatus() != Fapi_Status_Success)
        {
            i32Err = FLASH_ERASE_FAIL;
        }

        if (i32Err == FLASH_OK)
        {
            status = Fapi_doBlankCheck((uint32 *)u32SectAdd,
                                       u32Size / 2,
                                       &statusWord);
            if (status != Fapi_Status_Success)
            {
                /* Sector not empty */
                i32Err = FLASH_ERASE_CHECK_FAIL;
            }
        }
        EDIS;

        ReleaseFlashPump();
    }

    return i32Err;
}

#pragma CODE_SECTION(FlashWrite, "ramfuncs");
int32_t
FlashWrite(uint32_t u32Address, const uint16_t *pu16Buffer, uint16 u16Size,
           uint16_t *pu16PointIncrement)
{
    Fapi_StatusType status;
    Fapi_FlashStatusWordType statusWord;
    uint16_t u16WordSize;
    uint16_t u16RemainingBytes;
    uint32_t u32LocalAddress;
    uint32_t u32Alignment;
    uint32_t u32LocalBuffIndex;
    uint32_t u32TempAddress;
    int32_t i32Err = FLASH_OK;
    uint16_t u16i, u16j = 0;

    u16WordSize = 0;
    u16RemainingBytes = u16Size;
    *pu16PointIncrement = 0;
    u32LocalAddress = u32Address;
    u32Alignment = u32LocalAddress % LATCH_BUF_SIZE;
    u32LocalBuffIndex = u32Alignment;

    while (u16j < u16Size)
    {
        /* If the number of bytes is higher than 64 bits, split the writting procedure */
        if (u16RemainingBytes > (LATCH_BUF_SIZE << 1))
        {
            if (u32Alignment == 0)
            {
                u16WordSize = LATCH_BUF_SIZE;
            }
            else
            {
                u16WordSize = LATCH_BUF_SIZE - u32Alignment;
            }
            u16RemainingBytes -= (u16WordSize << 1);
        }
        else
        {
            if (((u16RemainingBytes >> 1) + u32Alignment) > LATCH_BUF_SIZE)
            {
                u16WordSize = LATCH_BUF_SIZE - u32Alignment;
                u16RemainingBytes -= (u16WordSize << 1);
            }
            else
            {
                if ((u16RemainingBytes % 2) == 1)
                {
                    u16WordSize = (u16RemainingBytes >> 1) + 1;
                }
                else
                {
                    u16WordSize = (u16RemainingBytes >> 1);
                }

                u16RemainingBytes = 0;
            }
        }

        /* Fill latch buffer */
        for (u16i = 0; u16i < u16WordSize; u16i++)
        {
            if (u16j < u16Size)
            {
                uLastValue.u16Array[u32LocalBuffIndex] = pu16Buffer[u16j];
            }
            else
            {
                uLastValue.u16Array[u32LocalBuffIndex] = 0xFF;
            }
            u16j++;

            uLastValue.u16Array[u32LocalBuffIndex] <<= 8;

            if (u16j < u16Size)
            {
                uLastValue.u16Array[u32LocalBuffIndex] += pu16Buffer[u16j];
            }
            else
            {
                uLastValue.u16Array[u32LocalBuffIndex] += 0xFF;
            }
            u16j++;

            /* Don't worry about buffer overflow as the data alignment is already
             * taken into account */
            u32LocalBuffIndex++;
        }
        /* If new address is aligned to 64 bits compute ECC */
        if (((u32LocalAddress + u16WordSize) % LATCH_BUF_SIZE) == 0)
        {
            SeizeFlashPump();

            /* Reset alignment, if a write is requested to the Flash, it means that
             * data is already aligned */
            u32LocalBuffIndex = 0;

            u32TempAddress = u32LocalAddress + u16WordSize - LATCH_BUF_SIZE;
            /* Master is responsible of Erasing */
            EALLOW;
            status = Fapi_issueProgrammingCommand((uint32 *)u32TempAddress,
                                                  uLastValue.u16Array,
                                                  LATCH_BUF_SIZE, 0, 0,
                                                  Fapi_AutoEccGeneration);

            if (status != Fapi_Status_Success)
            {
                i32Err = FLASH_WRITE_FAIL;
            }

            if (i32Err == FLASH_OK)
            {
                while (Fapi_checkFsmForReady() == Fapi_Status_FsmBusy)
                    ;

                if (Fapi_getFsmStatus() != Fapi_Status_Success)
                {
                    /* Write failed */
                    i32Err = FLASH_WRITE_FAIL;
                }
            }

            /* If address is not aligned with 32 bits, modify the
             * verifications buffers
             */
            status = Fapi_doVerify((uint32 *)u32TempAddress, 2,
                                   uLastValue.u32Array,
                                   &statusWord);
            EDIS;

            if (status != Fapi_Status_Success)
            {
                i32Err = FLASH_WRITE_CHECK_FAIL;
            }

            ReleaseFlashPump();
        }

        *pu16PointIncrement += u16WordSize;
        u32LocalAddress += u16WordSize;
    }

    return i32Err;
}

#pragma CODE_SECTION(FlashFindSector, "ramfuncs");
int32_t
FlashFindSector(uint32_t u32Addr, uint32_t* pu32StartAddress)
{
    int32_t i32Err = FLASH_OK;
    Fapi_FlashBankSectorsType fbs;
    Fapi_StatusType status;
    uint32_t u32SectorSize;
    uint32_t u32SectorAddr;
    uint16_t u16Id;

    SeizeFlashPump();

    status = Fapi_getBankSectors(Fapi_FlashBank0, &fbs);

    ReleaseFlashPump();

    if (Fapi_Status_Success != status)
    {
        i32Err = FLASH_GET_BANK_FAIL;
    }

    if (i32Err == FLASH_OK)
    {
        u32SectorAddr = fbs.u32BankStartAddress;
        for (u16Id = 0; u16Id < fbs.u32NumberOfSectors; u16Id++)
        {
            if (Fapi_FLEE == fbs.oFlashBankTech)
            {
                u32SectorSize = fbs.au8SectorSizes[0];
            }
            else
            {
                u32SectorSize = fbs.au8SectorSizes[u16Id];
            }

            u32SectorSize *= 1024;

            if ((u32SectorAddr <= u32Addr) &&
                (u32Addr < u32SectorAddr + u32SectorSize))
            {
                *pu32StartAddress = u32SectorAddr;
                break;
            }
            u32SectorAddr += u32SectorSize;
        }
    }

    return i32Err;
}

#pragma CODE_SECTION(FlashFindSectorSize, "ramfuncs");
int32_t
FlashFindSectorSize(uint32_t u32Addr, uint32_t* pu32Size)
{
    int32_t i32Err = FLASH_OK;
    Fapi_FlashBankSectorsType fbs;
    Fapi_StatusType status;
    uint32_t u32SectorSize;
    uint32_t u32SectorAddr;
    uint16_t u16Id;

    SeizeFlashPump();

    status = Fapi_getBankSectors(Fapi_FlashBank0, &fbs);

    ReleaseFlashPump();

    if (Fapi_Status_Success != status)
    {
        i32Err = FLASH_GET_BANK_FAIL;
    }

    if (i32Err == FLASH_OK)
    {
        u32SectorAddr = fbs.u32BankStartAddress;
        for (u16Id = 0; u16Id < fbs.u32NumberOfSectors; u16Id++)
        {
            if (Fapi_FLEE == fbs.oFlashBankTech)
            {
                u32SectorSize = fbs.au8SectorSizes[0];
            }
            else
            {
                u32SectorSize = fbs.au8SectorSizes[u16Id];
            }

            u32SectorSize *= 1024;

            if ((u32SectorAddr <= u32Addr) &&
                (u32Addr < u32SectorAddr + u32SectorSize))
            {
                *pu32Size = u32SectorSize;
                break;
            }
            u32SectorAddr += u32SectorSize;
        }
    }

    return i32Err;
}

#ifdef CPU1
void
FlashGetDeviceID(uint32_t* pu32Dev)
{
    *pu32Dev = DevCfgRegs.PARTIDH.all;
}
#endif
