/**
 * @file flash_mng.h
 * @brief General function for internal Flash management
 *
 * @note Main board instance style is not applied to current
 *       Flash API as current architecture only supports the
 *       unique and internal MCU flash.
 *
 * @author  Firmware department
 * @copyright Ingenia Motion Control (c) 2017. All rights reserved.
 */

/**
 * \addtogroup flashManager Flash API
 *
 * @{
 */


#ifndef FLASH_MNG_H_
#define FLASH_MNG_H_

#include <stdint.h>

#ifdef __cplusplus
extern "C"
{
#endif

/* minimim writting size into flash in words (uint16_t) */
#define MIN_BANK_WRITE_SIZE 4

typedef enum {
    FLASH_OK = 0, /**< No error */
    FLASH_ERROR = (int32_t)0x80010000 + 1, /**< Generic error */
    FLASH_API_NOT_INIT = (int32_t)0x80010000 + 2, /**< Flash API not init */
    FLASH_BANK_ACC_PROBLEM = (int32_t)0x80010000 + 3, /**< Problems accessing bank 0 */
    FLASH_ERASE_FAIL = (int32_t)0x80010000 + 4, /**< Erase process failed  */
    FLASH_ERASE_CHECK_FAIL = (int32_t)0x80010000 + 5, /**< Blank check failed */
    FLASH_WRITE_FAIL = (int32_t)0x80010000 + 6, /**< Write process failed */
    FLASH_WRITE_CHECK_FAIL = (int32_t)0x80010000 + 7, /**< Verification process failed */
    FLASH_GET_BANK_FAIL = (int32_t)0x80010000 + 8, /**< Get bank info failed */
    FLASH_NOT_IMPLEMENTED = (int32_t)0x80010000 + 9 /**< Not implemented function */
} FlashErrorCode;

/**
 * Initialized the Flash management module
 *
 * @param[out] u32SystemClock
 *  System clock in Mhz (i.e. 200 for 200 MHz)
 *
 * @retval See @FlashErrorCode
 */
int32_t
FlashInit(uint32_t u32SystemClock);

/**
 * Deinitializes Flash management module
 *
 * @note Not yet implemented
 * @retval See @FlashErrorCode
 */
int32_t
FlashDeInit(void);

/**
 * Erase a full sector of Flash memory.
 *
 * @details Erase the sector of Flash memory that contains the provided address.
 *
 * @param[in] u32Address
 *  Address of the flash sector to be erased.
 *
 * @note Sector address does not need to point to origin of the sector.
 *
 * @retval See @FlashErrorCode
 */
int32_t
FlashErase(uint32_t u32Addr);

/**
 * Write a byte array into Flash memory.
 *
 * @note F2837xD devices bank-width are 128, but 64 width writes
 *       are allowed. Align addres + data to write to 64 bits
 *
 * @param[in] u32Address
 *  Target start flash using word addressing.
 *
 * @param[in] pData
 *  Pointer to byte array of data to be programmed.
 *
 * @param[in] u16Size
 *  Number of bytes to be written into flash.
 *
 * @param[out] pu16PointIncrement
 *  returns the number of words increments needed to store
 *  the input buffer
 *
 * @note pData must be expressed in big-endian
 *
 * @retval See @FlashErrorCode
 */
int32_t
FlashWrite(uint32_t u32Addr, const uint16_t *pBuffer, uint16_t u16Size,
           uint16_t *pu16PointIncrement);

/**
 * Find the starting address of a sector that contains a given address
 *
 * @param[in] u32Addr
 *  Address to be searched.
 *
 * @param[out] ppu32StartAddress
 *  Pointer to sector address.
 *
 * @retval See @FlashErrorCode
 */
int32_t
FlashFindSector(uint32_t u32Addr, uint32_t* pu32StartAddress);

/**
 * Find the size of a sector that contains a given address
 *
 * @details Size is expressed in kwords
 * @param[in] u32Addr
 *  Address to be searched.
 *
 * @param[out] pu32Size
 *  Pointer to sector size.
 *
 * @retval See @FlashErrorCode
 */
int32_t
FlashFindSectorSize(uint32_t u32Addr, uint32_t* pu32Size);

/**
 * Return the Device part identification number
 *
 * @details Check Device Identification Registers on MCU datasheet for further
 *          information.
 *
 * @param[out] pu32Dev
 *  Pointer to device number id
 *
 */
void
FlashGetDeviceID(uint32_t* pu32Dev);

#ifdef __cplusplus
}
#endif

#endif /* FLASH_MNG_H_ */

/** @} */
