/**
 * @file i2c_interface.c
 * @brief
 *
 * @author  Firmware department
 * @copyright Ingenia Motion Control (c) 2018. All rights reserved.
 */

#include "i2c_interface.h"

#if NUMBER_OF_I2C > 0

#include "board_mapping.h"
#include "input_output.h"
#include <string.h>

/* Default I2C baud-rate is 400 kHz */
#define I2C_DFLT_BR (uint32_t)400000UL
/** Internal I2C module clock */
#define I2C_INT_CLOCK (uint32_t)10000000UL

struct TI2cPriv
{
    /** Handler identifier */
    uint16_t u16Id;
};

typedef struct
{
    /** I2c instance */
    TI2cIntf tInst;
    /** Configuration parameters */
    TI2cCfg tCfg;
    /** Indicates that the TX transmission is finished */
    volatile EI2cState e16State;
    /** Pointer to MCU registers structs */
    volatile struct I2C_REGS* pRegs;
    /** Rx interrupt identifier */
    uint32_t u32IntId;
    /** Transmission size in bytes */
    uint16_t u16TxSize;
} TI2cHandler;

static TI2cHandler tHandler;
static struct TI2cPriv tPriv;

#ifdef _FLASH
#pragma CODE_SECTION(I2C1Handler,"ramfuncs");
#endif

interrupt void I2C1Handler(void);

void I2CInit(uint16_t u16Id)
{
    switch (u16Id)
    {
        case I2C_INST_1:
#if defined(CPU1)
            GPIO_SetupPinOptions(
                    I2C1_SCL_PIN, GPIO_INPUT,
                    (uint32_t) GPIO_PUSHPULL | (uint32_t) GPIO_ASYNC);
            GPIO_SetupPinMux(I2C1_SCL_PIN, GPIO_MUX_CPU1, I2C1_SCL_PIN_MAP);
            GPIO_SetupPinOptions(
                    I2C1_SDA_PIN, GPIO_INPUT,
                    (uint32_t) GPIO_PUSHPULL | (uint32_t) GPIO_ASYNC);
            GPIO_SetupPinMux(I2C1_SDA_PIN, GPIO_MUX_CPU1, I2C1_SDA_PIN_MAP);
#endif

            IntRegister(INT_I2CINT1A, I2C1Handler);

            tHandler.tInst.ptPriv = &tPriv;
            tHandler.tInst.ptPriv->u16Id = u16Id;
            tHandler.pRegs = &I2C1_REGS;
            tHandler.u32IntId = I2C1_INT_SRC;
            tHandler.tInst.u32Br = I2C_DFLT_BR;
            break;
        default:
            /* Unsupported interface */
            break;
    }
}

void I2CDeinit(uint16_t u16Id)
{
    switch (u16Id)
    {
        case I2C_INST_1:
#if defined(CPU1)
            GPIO_SetupPinMux(I2C1_SCL_PIN, GPIO_MUX_CPU1, 0);
            GPIO_SetupPinMux(I2C1_SDA_PIN, GPIO_MUX_CPU1, 0);
#endif
            IntUnregister(I2C1_INT_SRC);

            tHandler.tInst.ptPriv = NULL;
            tHandler.pRegs = NULL;
            tHandler.u32IntId = (uint32_t)0UL;
            break;
        default:
            /* Unsupported interface */
            break;
    }
}

void I2CGetInstance(TI2cIntf** ptInst, uint16_t u16Id)
{
    if (u16Id < NUMBER_OF_I2C)
    {
        *ptInst = &(tHandler.tInst);
    }
}

void I2CStart(TI2cIntf* ptInst, const TI2cCfg* ptCfg)
{
    uint32_t u32Temp;

    if (ptInst != NULL)
    {
        if (ptCfg != NULL)
        {
            if (ptCfg->u16RxBfrSize > MAX_I2C_BFR_SIZE)
            {
                tHandler.tCfg.u16RxBfrSize = MAX_I2C_BFR_SIZE;
            }
            else
            {
                tHandler.tCfg.u16RxBfrSize = ptCfg->u16RxBfrSize;
            }

            tHandler.tCfg.pEvntCtx = ptCfg->pEvntCtx;
            tHandler.tCfg.ptStopCB = ptCfg->ptStopCB;
            tHandler.tCfg.ptArdyCB = ptCfg->ptArdyCB;

            tHandler.tCfg.u32Br = ptCfg->u32Br;
        }

        tHandler.e16State = NON_REQUEST;

        /* Keep I2C in reset state during configuration */
        tHandler.pRegs->I2CMDR.bit.IRS = 0;

        /** I2C clock input is system clock */
        /** Set internal I2C clock to 10 MHz */
        tHandler.pRegs->I2CPSC.bit.IPSC = (SYSTEM_FREQ_HZ / I2C_INT_CLOCK ) - 1;
        /** Set baudrate (Reference manual, section 20.1.5) */
        u32Temp = (I2C_INT_CLOCK / (2 * tHandler.tCfg.u32Br)) - (uint32_t)5UL;

        /* Amount of time the clock signal is Low (I2CCLKL) and High (I2CCLKH) */
        tHandler.pRegs->I2CCLKH = (uint16_t)u32Temp;
        tHandler.pRegs->I2CCLKL = (uint16_t)u32Temp;

        /* Clear flags */
        /* Enable FIFO enhancements */
        tHandler.pRegs->I2CFFTX.bit.I2CFFEN = 1;
        /* Clear FIFO flags */
        tHandler.pRegs->I2CFFTX.bit.TXFFINTCLR = 1;
        tHandler.pRegs->I2CFFRX.bit.RXFFINTCLR = 1;
        /* Release FIFO from resets */
        tHandler.pRegs->I2CFFRX.bit.RXFFRST = 1;
        tHandler.pRegs->I2CFFTX.bit.TXFFRST = 1;
        /* Set FIFO interrupt levels */
        tHandler.pRegs->I2CFFTX.bit.TXFFIL = 0;
        tHandler.pRegs->I2CFFRX.bit.RXFFIL = tHandler.tCfg.u16RxBfrSize;

        /** Enable desired interrupts */
        tHandler.pRegs->I2CIER.bit.SCD = 1;
        tHandler.pRegs->I2CIER.bit.ARDY = 1;

        IntEnable(tHandler.u32IntId);
        /* Release I2C module from reset state */
        tHandler.pRegs->I2CMDR.bit.IRS = 1;
    }
}

void I2CStop(const TI2cIntf* ptInst)
{
    if (ptInst != NULL)
    {
        /* Keep I2C in reset state during configuration */
        tHandler.pRegs->I2CMDR.bit.IRS = 0;

        tHandler.tCfg.ptStopCB = NULL;
        tHandler.tCfg.ptArdyCB = NULL;
        tHandler.tCfg.pEvntCtx = NULL;
    }
}

#ifdef CPU1
uint16_t I2CSelfTest(uint16_t u16Id)
{
    uint16_t u16TestResult = 0;

    return (u16TestResult);
}

void I2CCPU2Mapping(uint16_t u16Id)
{
    switch (u16Id)
    {
        case I2C_INST_1:
            /* Give I2C control to CPU2 */
            EALLOW;
            switch (I2C1_BASE)
            {
                case I2CA_BASE:
                    DevCfgRegs.CPUSEL7.bit.I2C_A = 1;
                    break;
                case I2CB_BASE:
                    DevCfgRegs.CPUSEL7.bit.I2C_B = 1;
                    break;
                default:
                    /* Nothing */
                    break;
            }
            EDIS;

            GPIO_SetupPinOptions(
                    I2C1_SCL_PIN, GPIO_INPUT,
                    (uint32_t) GPIO_PUSHPULL | (uint32_t) GPIO_ASYNC);
            GPIO_SetupPinMux(I2C1_SCL_PIN, GPIO_MUX_CPU2, I2C1_SCL_PIN_MAP);
            GPIO_SetupPinOptions(
                    I2C1_SDA_PIN, GPIO_INPUT,
                    (uint32_t) GPIO_PUSHPULL | (uint32_t) GPIO_ASYNC);
            GPIO_SetupPinMux(I2C1_SDA_PIN, GPIO_MUX_CPU2, I2C1_SDA_PIN_MAP);
            break;
        default:
            /* Nothing */
            break;
    }
}
#endif

void I2CConfig(TI2cIntf* ptInst, const TI2cCfg* ptCfg)
{
    uint32_t u32Temp;

    if (ptInst != NULL)
    {
        /* Keep I2C in reset state during configuration */
        tHandler.pRegs->I2CMDR.bit.IRS = 0;

        tHandler.tCfg.pEvntCtx = ptCfg->pEvntCtx;
        tHandler.tCfg.ptStopCB = ptCfg->ptStopCB;
        tHandler.tCfg.ptArdyCB = ptCfg->ptArdyCB;
        tHandler.e16State = NON_REQUEST;

        if (ptCfg->u16RxBfrSize > MAX_I2C_BFR_SIZE)
        {
            tHandler.tCfg.u16RxBfrSize = MAX_I2C_BFR_SIZE;
        }
        else
        {
            tHandler.tCfg.u16RxBfrSize = ptCfg->u16RxBfrSize;
        }

        /** I2C clock input is system clock */
        /** Set internal I2C clock to 10 MHz */
        tHandler.pRegs->I2CPSC.bit.IPSC = (SYSTEM_FREQ_HZ / I2C_INT_CLOCK ) - 1;
        /** Set baudrate (Reference manual, section 20.1.5) */
        u32Temp = (I2C_INT_CLOCK / (2 * tHandler.tCfg.u32Br)) - (uint32_t)5UL;

        /* Amount of time the clock signal is Low (I2CCLKL) and High (I2CCLKH) */
        tHandler.pRegs->I2CCLKH = (uint16_t)u32Temp;
        tHandler.pRegs->I2CCLKL = (uint16_t)u32Temp;

        /* Clear flags */
        /* Enable FIFO enhancements */
        tHandler.pRegs->I2CFFTX.bit.I2CFFEN = 1;
        /* Clear FIFO flags */
        tHandler.pRegs->I2CFFTX.bit.TXFFINTCLR = 1;
        tHandler.pRegs->I2CFFRX.bit.RXFFINTCLR = 1;
        /* Release FIFO from resets */
        tHandler.pRegs->I2CFFRX.bit.RXFFRST = 1;
        tHandler.pRegs->I2CFFTX.bit.TXFFRST = 1;
        /* Set FIFO interrupt levels */
        tHandler.pRegs->I2CFFTX.bit.TXFFIL = 0;
        tHandler.pRegs->I2CFFRX.bit.RXFFIL = tHandler.tCfg.u16RxBfrSize;

        /** Enable desired interrupts */
        tHandler.pRegs->I2CIER.bit.SCD = 1;
        tHandler.pRegs->I2CIER.bit.ARDY = 1;

        IntEnable(tHandler.u32IntId);
        /* Release I2C module from reset state */
        tHandler.pRegs->I2CMDR.bit.IRS = 1;
    }
}

void I2CSetCB(TI2cIntf* ptInst, void* pCtx, TI2cEvntCB ptStopCB,
              TI2cEvntCB ptArdyCB)
{
    if (ptInst != NULL)
    {
        tHandler.tCfg.pEvntCtx = pCtx;
        tHandler.tCfg.ptStopCB = ptStopCB;
        tHandler.tCfg.ptArdyCB = ptArdyCB;
    }
}

int32_t I2CWrite(const TI2cIntf* ptInst, uint16_t u16Add,
                 const uint16_t* pu16Src, uint16_t u16Size)
{
    uint16_t u16Idx;
    const uint16_t* pu16InBfr = pu16Src;
    int32_t i32Ret = REQUEST_IN_PROG;
    uint16_t u16RealSize;

    if (ptInst != NULL)
    {
        switch (tHandler.e16State)
        {
            case NON_REQUEST:
                if (tHandler.pRegs->I2CMDR.bit.STP == 0)
                {
                    /** Non stop mode is used for generate read request */
                    tHandler.e16State = REQUEST_W_IN_PROG;

                    tHandler.pRegs->I2CSAR.all = u16Add;

                    /**
                     * NACKMOD:FREE:STT: :STP:MST:TRX:XA:RM:DLB:ISR:STB:FDF:BC
                     *    0   :  1 : 1 :0: 0 : 1 : 1 : 0: 1: 0 : 1 : 0 : 0 :000
                     */
                    tHandler.pRegs->I2CMDR.all = 0x66A0;
                }
                else
                {
                    i32Ret = BUS_ERROR;
                }
                break;

            case REQUEST_OK:
                if (u16Size < MAX_I2C_BFR_SIZE)
                {
                    u16RealSize = u16Size;
                }
                else
                {
                    u16RealSize = MAX_I2C_BFR_SIZE;
                }

                tHandler.u16TxSize = 0;

                for (u16Idx = 0; u16Idx < u16RealSize; u16Idx++)
                {
                    if (tHandler.pRegs->I2CFFTX.bit.TXFFST < MAX_I2C_BFR_SIZE)
                    {
                        tHandler.pRegs->I2CDXR.all = pu16InBfr[u16Idx];
                        tHandler.u16TxSize++;
                    }
                }
                tHandler.e16State = WRITE_IN_PROG;
                break;

            case REQUEST_FAIL:
                /** Return error code */
                i32Ret = BUS_ERROR;
                tHandler.e16State = NON_REQUEST;
                break;

            case WRITE_FAIL:
                i32Ret = BUS_ERROR;
                tHandler.e16State = NON_REQUEST;
                break;

            case WRITE_OK:
                i32Ret = (int32_t)tHandler.u16TxSize;
                tHandler.e16State = NON_REQUEST;
                break;

            default:
                /** Wait for interrupt */
                break;
        }
    }

    return i32Ret;
}

int32_t I2CRead(const TI2cIntf* ptInst, uint16_t u16Add, uint16_t u16RegAdd,
                uint16_t* pu16Dst, uint16_t u16Size)
{
    uint16_t u16Idx;
    int32_t i32Ret = REQUEST_IN_PROG;
    uint16_t* pu16OutBfr = pu16Dst;
    uint16_t u16RealSize;

    if (ptInst != NULL)
    {
        switch (tHandler.e16State)
        {
            case NON_REQUEST:
                if ((tHandler.pRegs->I2CSTR.bit.BB == 0)
                    && (tHandler.pRegs->I2CMDR.bit.STP == 0))
                {
                    /** Non stop mode is used for generate read request */
                    tHandler.e16State = REQUEST_R_IN_PROG;
                    tHandler.pRegs->I2CSAR.all = u16Add;
                    tHandler.pRegs->I2CCNT = 2;

                    if (tHandler.pRegs->I2CFFTX.bit.TXFFST
                        < (MAX_I2C_BFR_SIZE - 1))
                    {
                        tHandler.pRegs->I2CDXR.all = (u16RegAdd >> (uint16_t)8)
                                & (uint16_t)0x00FF;
                        tHandler.pRegs->I2CDXR.all = u16RegAdd
                                & (uint16_t)0x00FF;
                    }

                    /**
                     * NACKMOD:FREE:STT: :STP:MST:TRX:XA:RM:DLB:ISR:STB:FDF:BC
                     *    0   :  0 : 1 :0: 0 : 1 : 1 : 0: 0: 0 : 1 : 0 : 0 :000
                     */
                    tHandler.pRegs->I2CMDR.all = 0x2620;
                }
                else
                {
                    i32Ret = BUS_ERROR;
                }
                break;

            case REQUEST_OK:
                /** Return size of written bytes through network */
                tHandler.e16State = READ_IN_PROG;

                if (u16Size < MAX_I2C_BFR_SIZE)
                {
                    u16RealSize = u16Size;
                }
                else
                {
                    u16RealSize = MAX_I2C_BFR_SIZE;
                }

                tHandler.pRegs->I2CSAR.all = u16Add;
                tHandler.pRegs->I2CCNT = u16RealSize;
                /**
                 * NACKMOD:FREE:STT: :STP:MST:TRX:XA:RM:DLB:ISR:STB:FDF:BC
                 *    0   :  0 : 1 :0: 0 : 1 : 0 : 0: 0: 0 : 1 : 0 : 0 :000
                 */
                tHandler.pRegs->I2CMDR.all = 0x2420;
                break;

            case REQUEST_FAIL:
                /** Return error code */
                i32Ret = BUS_ERROR;
                tHandler.e16State = NON_REQUEST;
                break;

            case READ_FAIL:
                i32Ret = BUS_ERROR;
                tHandler.e16State = NON_REQUEST;
                break;

            case READ_OK:
                for (u16Idx = 0; u16Idx < u16RealSize; u16Idx++)
                {
                    if (tHandler.pRegs->I2CFFRX.bit.RXFFST > 0)
                    {
                        pu16OutBfr[u16Idx] = tHandler.pRegs->I2CDRR.all;
                        i32Ret++;
                    }
                }
                tHandler.e16State = NON_REQUEST;
                break;
            default:
                /** Wait for interrupt */
                break;
        }
    }

    return (i32Ret);
}

uint16_t I2CAvailable(const TI2cIntf* ptInst)
{
    uint16_t u16Avbl = 0;

    if (ptInst != NULL)
    {
        u16Avbl = tHandler.pRegs->I2CFFRX.bit.RXFFST;
    }
    return u16Avbl;
}

interrupt void I2C1Handler(void)
{
    if (tHandler.pRegs->I2CISRC.all == I2C_SCD_ISRC)
    {
        if (tHandler.tCfg.ptStopCB != NULL)
        {
            tHandler.tCfg.ptStopCB(tHandler.tCfg.pEvntCtx, NULL);
        }
    }

    /* Interrupt source = Register Access Ready
     * This interrupt is used to determine when the EEPROM address setup
     * portion of the read data communication is complete. Since no stop bit is
     * commanded, this flag tells us when the message has been sent instead of
     * the SCD flag. If a NACK is received, clear the NACK bit and command a
     * stop. Otherwise, move on to the read data portion of the communication.
     */
    if (tHandler.pRegs->I2CISRC.all == I2C_ARDY_ISRC)
    {
        switch (tHandler.e16State)
        {
            case REQUEST_W_IN_PROG:
                if (tHandler.pRegs->I2CSTR.bit.NACK == 1)
                {
                    tHandler.pRegs->I2CSTR.all = I2C_CLR_NACK_BIT;
                    tHandler.e16State = REQUEST_FAIL;
                }
                else
                {
                    tHandler.e16State = REQUEST_OK;
                }
                break;
            case REQUEST_R_IN_PROG:
                if (tHandler.pRegs->I2CSTR.bit.NACK == 1)
                {
                    tHandler.pRegs->I2CMDR.bit.STP = 1;
                    tHandler.pRegs->I2CSTR.all = I2C_CLR_NACK_BIT;
                    tHandler.e16State = REQUEST_FAIL;
                }
                else
                {
                    tHandler.e16State = REQUEST_OK;
                }
                break;
            case WRITE_IN_PROG:
                if (tHandler.pRegs->I2CSTR.bit.NACK == 1)
                {
                    tHandler.pRegs->I2CSTR.all = I2C_CLR_NACK_BIT;
                    tHandler.e16State = WRITE_FAIL;
                }
                else
                {
                    tHandler.e16State = WRITE_OK;
                }
                tHandler.pRegs->I2CMDR.bit.STP = 1;
                break;
            case READ_IN_PROG:
                if (tHandler.pRegs->I2CSTR.bit.NACK == 1)
                {
                    tHandler.pRegs->I2CSTR.all = I2C_CLR_NACK_BIT;
                    tHandler.e16State = READ_FAIL;
                }
                else
                {
                    tHandler.e16State = READ_OK;
                }
                tHandler.pRegs->I2CMDR.bit.STP = 1;
                break;
            default:
                tHandler.pRegs->I2CMDR.bit.STP = 1;
                break;
        }

        if (tHandler.tCfg.ptArdyCB != NULL)
        {
            tHandler.tCfg.ptArdyCB(tHandler.tCfg.pEvntCtx,
                                   (uint16_t*)&tHandler.e16State);
        }
    }

    PieCtrlRegs.PIEACK.all = I2C1_INT_ACK;

    /* Restore nesting interrupt settings */
    DINT;
}
#endif
