/**
 * @file i2c_interface.h
 * @brief This file offers a abstract layer that allows to use any i2c module
 *
 * @author  Firmware department
 * @copyright Ingenia Motion Control (c) 2018. All rights reserved.
 */

/**
 * \addtogroup communications communications API
 *
 * @{
 */

#ifndef I2C_INTERFACE_H
#define I2C_INTERFACE_H

#include "board_config.h"

#if NUMBER_OF_I2C > 0

#include <stdint.h>
#include <stdbool.h>

#ifdef __cplusplus
extern "C"
{
#endif

/** Maximum internal buffer size for block TX/RX */
#define MAX_I2C_BFR_SIZE 16

/** Request error code */
#define BUS_ERROR       (int32_t)-1L
/** Request error code */
#define REQUEST_IN_PROG (int32_t)-2L

typedef enum
{
    /** Reading as master request not done */
    NON_REQUEST = 0,
    /** Writing as master request in progress */
    REQUEST_W_IN_PROG,
    /** Reading as master request in progress */
    REQUEST_R_IN_PROG,
    /** Reading as master request is finished */
    REQUEST_END,
    /** Reading as master request has been acknowledged */
    REQUEST_OK,
    /** Reading as master request has been non-acknowledged */
    REQUEST_FAIL,
    /** Writing operation in progress */
    WRITE_IN_PROG,
    /** Write operation fails */
    WRITE_FAIL,
    /** Write opretion successful */
    WRITE_OK,
    /** Reading operation in progress */
    READ_IN_PROG,
    /** Read operation fails */
    READ_FAIL,
    /** Read opretion successful */
    READ_OK
} EI2cState;

/** Private I2C parameters */
struct TI2cPriv;

/** Generic event callback */
typedef void (*TI2cEvntCB)(void* Ctx, uint16_t* u16Arg);

/** I2c interface struct */
typedef struct
{
    /** Baudrate of the interface in bps */
    uint32_t u32Br;
    /** Private parameters */
    struct TI2cPriv* ptPriv;
} TI2cIntf;

typedef struct
{
    /** Baudrate of the interface in kHz*/
    uint32_t u32Br;
    /** Size of RX buffer data */
    uint16_t u16RxBfrSize;
    /* Pointer to store Rx Event context */
    void* pEvntCtx;
    /** Callback for Stop condition detection */
    TI2cEvntCB ptStopCB;
    /** Callback for Nack condition detection */
    TI2cEvntCB ptArdyCB;
} TI2cCfg;

/**
 * Initializes an I2C instance.
 *
 * @note The I2C is not operative after this function. @see I2CStart
 *
 * @param[in] u16Id
 *  Specifies the target I2C to be initialized
 */
void
I2CInit(uint16_t u16Id);

/**
 * Deinitializes an I2C instance.
 *
 * @param[in] u16Id
 *  Specifies the target I2C to be deinitialized
 */
void
I2CDeinit(uint16_t u16Id);

/**
 * Returns the desired instance
 *
 * @param[out] ptInst
 *  Pointer to store the target instance
 * @param[in] u16Id
 *  Identifier of the target instance
 */
void
I2CGetInstance(TI2cIntf** ptInst, uint16_t u16Id);

/**
 * Launches the I2C to start standard operation
 *
 * @param[in] ptInst
 *  Specifies the target I2C to be launched.
 * @param[in] ptCfg
 *  Specifies the configuration of the I2C.
 */
void
I2CStart(TI2cIntf* ptInst, const TI2cCfg* ptCfg);

/**
 * Stops the I2C.
 *
 * @param[in] ptInst
 *  Specifies the target I2C to be stopped.
 */
void
I2CStop(const TI2cIntf* ptInst);

/**
 * Check that the I2C module is working.
 *
 * @param[in] u16Id
 *  Specifies the target I2C to be configured
 *
 * @Note Module must be already Initializated
 *
 * @retval 0 if success, error code otherwise
 */
uint16_t
I2CSelfTest(uint16_t u16Id);

/**
 * Map the peripheral to the mapped CPU.
 *
 * @param[in] u16Id
 *  Specifies the target I2C to be configured
 */
void
I2CCPU2Mapping(uint16_t u16Id);

/**
 * Configures an I2C instance.
 *
 * @param[in] ptInst
 *  Specifies the target I2C to be configured
 * @param[in] ptCfg
 *  Specifies the configuration of the I2C.
 */
void
I2CConfig(TI2cIntf* ptInst, const TI2cCfg* ptCfg);

/**
 * Set callback for an I2C instance.
 *
 * @note this is useful when different modules access to the same
 *       I2C instance and the callback has to be switched.
 * @param[in] ptInst
 *  Specifies the target I2C to be configured
 * @param[in] pCtx
 *  Specifies the context of the module that uses the I2C instance.
 * @param[in] ptCback
 *  Specifies the callback to be called.
 */
void
I2CSetCB(TI2cIntf* ptInst, void* pCtx, TI2cEvntCB ptStopCB,
         TI2cEvntCB ptNackCB);

/**
 * Sends data through an I2C instance.
 *
 * @note this function is implemented to access external
 *        I2C EEPROM slave devices
 *
 * @param[in] ptInst
 *  Specifies the target I2C
 * @param[in]  u16Add
 *  Slave address to be accessed
 * @param[in] pu16Src
 *  Contains a pointer to the buffer of data to be sent
 * @param[in] u16Size
 *  Indicates the number of BYTES to be sent
 *
 * @retval Number of BYTES that have been sent when request is finished or,
 *         REQUEST_ERROR if slave NACKs the request
 *         REQUEST_IN_PROG if request is being processing (non-blocking function)
 */
int32_t
I2CWrite(const TI2cIntf* ptInst, uint16_t u16Add, const uint16_t* pu16Src,
         uint16_t u16Size);

/**
 * Reads data through an I2C instance.
 *
 * @note this function is implemented to access external
 *        I2C EEPROM slave devices
 *
 * @param[in] ptInst
 *  Specifies the target I2C
 * @param[in] u16Add
 *  specifies the slave adddress where data comes from
 * @param[in] u16RegAdd
 *  specifies the address of the register to be accessed
 * @param[in] pu16Dst
 *  Contains a pointer to the buffer of data to be written
 * @param[in] u16Size
 *  Indicates the number of BYTES to be read
 *
 * @retval Number of BYTES that have been sent when request is finished or,
 *         REQUEST_ERROR if slave NACKs the request
 *         REQUEST_IN_PROG if request is being processing (non-blocking function)
 *
 * @see I2CAvailable
 */
int32_t
I2CRead(const TI2cIntf* ptInst, uint16_t u16Add, uint16_t u16RegAdd,
        uint16_t* pu16Dst, uint16_t u16Size);

/**
 * Returns the number of bytes available to be read.
 *
 * @param[in] ptInst
 *  Specifies the target I2C
 *
 * @retval Number of BYTES that can be read
 *
 * @see I2CRead
 */
uint16_t
I2CAvailable(const TI2cIntf* ptInst);

#ifdef __cplusplus
}
#endif

#endif

#endif /* I2C_INTERFACE_H */

/** @} */
