/**
 * @file inputoutput.h
 * @brief This file offers a abstract layer that allows to use any input output
 *        of a rockstarcore based drive
 *
 * @author  Firmware department
 * @copyright Ingenia Motion Control (c) 2016. All rights reserved.
 */

/**
 * \defgroup InOut Input Output API
 *
 * @{
 */

#ifndef INPUTOUTPUT_H
#define INPUTOUTPUT_H

#include <stdint.h>

#ifdef __cplusplus
extern "C"
{
#endif

/** List of the input/output configuration modes */
typedef enum
{
    /** Configure an input/output into an input */
    INPUT_MODE = 0,
    /** Configure an input/output into an output */
    OUTPUT_MODE = 1,
    /** Configure an input/output into an input with internal pull up */
    INPUT_PULLUP_MODE = 2,
    /** Configure an input/output into an output with internal pull up */
    OUTPUT_PULLUP_MODE = 3
} EInOutModes;

/** List of the available intput/output values (from MCU point of view) */
typedef enum
{
    /** Logic value 0 = (0 V) */
    LOW = 0,
    /** Logic value 1 = (Vdd V) */
    HIGH = 1
} EInOutValue;

/**
 * Configures an input/output pin to work as eMode.
 *
 * @param[in] u16Pin
 *  Specifies the target pin to be configured
 * @param[in]
 *  Specifies the mode of operation @ref eMode
 */
void
InOutConfig(uint16_t u16Pin, EInOutModes eMode);

/**
 * Reads the value from an input pin. The pin must be configured as input.
 *
 * @param[in] u16Pin
 *  Specifies the target pin to be configured
 *
 * @retval @ref EInOutValue
 *
 * @see InOutConfig
 */
EInOutValue
InRead(uint16_t u16Pin);

/**
 * Writes the value into an output pin. The pin must be configured as output.
 *
 * @param[in] u16Pin
 *  Specifies the target pin to be configured
 * @param[in] EInOutValue
 *  Specifies the value to be written @ref EInOutValue
 *
 * @see InOutConfig
 */
void
OutWrite(uint16_t u16Pin, EInOutValue eValue);

#ifdef __cplusplus
}
#endif

#endif /* INPUTOUTPUT_H */

/** @} */
