/**
 * @file ip_communications.c
 * @brief
 *
 * @author  Firmware department
 * @copyright Ingenia Motion Control (c) 2016. All rights reserved.
 */

#include "ip_communications.h"

#include "board_mapping.h"
#include "board_config.h"

#define NO_FLAG                     (uint32_t)0x00000000
#define IPC_FLAG0                   (uint32_t)0x00000001
#define IPC_FLAG1                   (uint32_t)0x00000002
#define IPC_FLAG2                   (uint32_t)0x00000004
#define IPC_FLAG3                   (uint32_t)0x00000008
#define IPC_FLAG4                   (uint32_t)0x00000010
#define IPC_FLAG5                   (uint32_t)0x00000020
#define IPC_FLAG6                   (uint32_t)0x00000040
#define IPC_FLAG7                   (uint32_t)0x00000080
#define IPC_FLAG8                   (uint32_t)0x00000100
#define IPC_FLAG9                   (uint32_t)0x00000200
#define IPC_FLAG10                  (uint32_t)0x00000400
#define IPC_FLAG11                  (uint32_t)0x00000800
#define IPC_FLAG12                  (uint32_t)0x00001000
#define IPC_FLAG13                  (uint32_t)0x00002000
#define IPC_FLAG14                  (uint32_t)0x00004000
#define IPC_FLAG15                  (uint32_t)0x00008000
#define IPC_FLAG16                  (uint32_t)0x00010000
#define IPC_FLAG17                  (uint32_t)0x00020000
#define IPC_FLAG18                  (uint32_t)0x00040000
#define IPC_FLAG19                  (uint32_t)0x00080000
#define IPC_FLAG20                  (uint32_t)0x00100000
#define IPC_FLAG21                  (uint32_t)0x00200000
#define IPC_FLAG22                  (uint32_t)0x00400000
#define IPC_FLAG23                  (uint32_t)0x00800000
#define IPC_FLAG24                  (uint32_t)0x01000000
#define IPC_FLAG25                  (uint32_t)0x02000000
#define IPC_FLAG26                  (uint32_t)0x04000000
#define IPC_FLAG27                  (uint32_t)0x08000000
#define IPC_FLAG28                  (uint32_t)0x10000000
#define IPC_FLAG29                  (uint32_t)0x20000000
#define IPC_FLAG30                  (uint32_t)0x40000000
#define IPC_FLAG31                  (uint32_t)0x80000000

/**
 * The following values report on the CPU02 boot ROM status at all times while
 * the CPU02 is booting, and will reside in IPCBOOTSTS[11:0].
 */


/** CPU02 has not filled in a valid value yet */
#define C2_BOOTROM_BOOTSTS_C2TOC1_IGNORE        (uint32_t)0x00000000

/**
 * CPU02 has started to boot, but not completed
 * the boot process yet
 */
#define C2_BOOTROM_BOOTSTS_SYSTEM_START_BOOT    (uint32_t)0x00000001

/**
 * CPU02 has completed the boot and is ready for
 * CPU01 TO CPU02 IPC commands
 */
#define C2_BOOTROM_BOOTSTS_SYSTEM_READY         (uint32_t)0x00000002

/**
 * CPU02 ACKs the command in CPU01 TO CPU01
 * BOOTMODE register
 */
#define C2_BOOTROM_BOOTSTS_C2TOC1_BOOT_CMD_ACK  (uint32_t)0x00000003

/**
 * CPU02 un-supported command in CPU01 TO CPU01
 * BOOTMODE register
 */
#define C2_BOOTROM_BOOTSTS_C2TOC1_BOOT_CMD_NAK_STATUS_NOT_SUPPORTED  (uint32_t)0x00000004

/**
 * CPU2 NAKs the current boot command in
 * CPU01 TO CPU01 BOOTMODE register
 */
#define C2_BOOTROM_BOOTSTS_C2TOC1_BOOT_CMD_NAK_STATUS_BUSY_WITH_BOOT (uint32_t)0x00000005

#define BROM_IPC_EXECUTE_BOOTMODE_CMD     0x00000013

#define QUEUES_SIZE 0x80

struct TIpCommsPriv
{
    /** Handler identifier */
    uint16_t u16Id;
};

typedef struct
{
    /** Inter-processor communications instance */
    TIpCommsIntf tInst;
} TIpCommsHandler;


static TIpCommsHandler tHandler[NUMBER_OF_PROCESSORS];
static struct TIpCommsPriv tPriv[NUMBER_OF_PROCESSORS];

void IpCommsInit(uint16_t u16Id)
{
    switch (u16Id)
    {
        case CPU_1:
            tHandler[u16Id].tInst.ptPriv = &tPriv[u16Id];
            tHandler[u16Id].tInst.ptPriv->u16Id = u16Id;
            break;
        case CPU_2:
            tHandler[u16Id].tInst.ptPriv = &tPriv[u16Id];
            tHandler[u16Id].tInst.ptPriv->u16Id = u16Id;
            break;
        default:
            break;
    }
}

uint16_t IpCommsRunOtherCPU(uint16_t u16Id)
{
    uint16_t u16Ret = 0;

    switch (u16Id)
    {
        case CPU_1:
            /* If CPU2 has already booted, return a fail to let the application
             * know that something is out of the ordinary. */
            if ((IpcRegs.IPCBOOTSTS & (uint32_t)0xF) == C2_BOOTROM_BOOTSTS_C2TOC1_BOOT_CMD_ACK)
            {
                u16Ret = 1;
            }

            if (u16Ret == 0)
            {
                /* Wait until CPU02 control system boot ROM is ready to receive
                 * CPU01 to CPU02 INT1 interrupts. */
                while ((IpcRegs.IPCBOOTSTS & C2_BOOTROM_BOOTSTS_SYSTEM_READY ) != C2_BOOTROM_BOOTSTS_SYSTEM_READY )
                {
                }

                while ((IpcRegs.IPCFLG.bit.IPC0 == (uint16_t)1) ||
                       (IpcRegs.IPCFLG.bit.IPC31 == (uint16_t)1))
                {
                }

                /** Boot up CPU2 from FLASH */
                IpcRegs.IPCBOOTMODE = (uint32_t)0xB;

                /* CPU01 To CPU02 IPC Command Register */
                IpcRegs.IPCSENDCOM = BROM_IPC_EXECUTE_BOOTMODE_CMD;

                /* CPU01 to CPU02 IPC flag register */
                IpcRegs.IPCSET.all = (uint32_t)0x8F0F0F01UL;

            }
            break;
        case CPU_2:
            IpcRegs.IPCSET.all = (uint32_t)0x55555555UL;
            break;
        default:
            break;
     }

    return u16Ret;
}

void IpCommsWaitOtherCPU(uint16_t u16Id)
{
    switch (u16Id)
    {
        case CPU_1:
            while (IpcRegs.IPCSTS.all != (uint32_t)0x55555555UL)
            {

            }
            break;
        default:
            break;
    }
}

bool IpCommsCheckFlags(uint16_t u16Id)
{
    bool isBoot = true;
    switch (u16Id)
    {
        case CPU_2:
            if (IpcRegs.IPCSTS.all != (uint32_t)0x0F0F0F00UL)
            {
                IpcRegs.IPCACK.all = (uint32_t)0x0F0F0F00UL;
                isBoot = false;
            }
            break;
        default:
            break;
    }

    return isBoot;
}

