/**
 * @file ip_communications.h
 * @brief This file offers an abstract layer that allows to communicate with
 * other processors through internal buses.
 *
 * @author  Firmware department
 * @copyright Ingenia Motion Control (c) 2016. All rights reserved.
 */

/**
 * \defgroup interprocessor Inter-processor API
 *
 * @{
 */

#ifndef IP_COMMUNICATIONS_H
#define IP_COMMUNICATIONS_H

#include <stdint.h>
#include <stdbool.h>

#ifdef __cplusplus
extern "C"
{
#endif

/** Private IpComm parameters */
struct TIpCommsPriv;
/**
 * Interprocessor communications intance
 */
typedef struct
{
    /** Private parameters */
    struct TIpCommsPriv* ptPriv;
} TIpCommsIntf;

/**
 * Initializes an IPC instance.
 *
 * @note The IPC is not operative after this function. @see IpCommsStart
 *
 * @param[in] u16Id
 *  Specifies the target IPC to be initialized
 */
void
IpCommsInit(uint16_t u16Id);

/**
 * Runs other CPU.
 *
 * @param[in] u16Id
 *  Specifies the CPU that is calling this function
 *
 *  @retval 0 if other CPU has been run correctly
 *          1 if other CPU was already initialized
 */
uint16_t
IpCommsRunOtherCPU(uint16_t u16Id);

#ifdef _FLASH
#pragma CODE_SECTION(IpCommsWaitOtherCPU,"ramfuncs");
#endif
/**
 * Locks current CPU in an infinite loop until other
 * CPU executes IpCommsRunOtherCPU
 *
 * @param[in] u16Id
 *  Specifies the CPU that is calling this function
 */
void
IpCommsWaitOtherCPU(uint16_t u16Id);

/**
 * This function is used by CPU2 to check the
 * selected boot mode by CPU1:
 *  - Bootloader mode
 *  - Application mode
 *
 * @param[in] u16Id
 *  Specifies the CPU that is calling this function
 *
 * @retval false if boot mode is selected
 *         true if app mode is selected
 */
bool
IpCommsCheckFlags(uint16_t u16Id);

#ifdef __cplusplus
}
#endif

#endif /* IP_COMMUNICATIONS_H */

/** @} */
