/**
 * @file spi_interface.c
 * @brief
 *
 * @author  Firmware department
 * @copyright Ingenia Motion Control (c) 2018. All rights reserved.
 */

#include "spi_interface.h"

#if NUMBER_OF_SPI > 0

#include "board_mapping.h"
#include "input_output.h"
#include <string.h>

#define TEST_CHAR   0x7465
/* Default SPI baud-rate is 10 MHz */
#define SPI_DFLT_BR 10000

/* Register access macros */
#define HND_ACCESS tHandler[ptInst->ptPriv->u16Id]

struct TSpiPriv
{
    /** Handler identifier */
    uint16_t u16Id;
    /** Loopback pin identifier */
    uint16_t u16Lpbk;
};

typedef struct
{
    /** Spi instance */
    TSpiIntf tInst;
    /** Configuration parameters */
    TSpiCfg tCfg;
    /** Indicates that the TX transmission is finished */
    volatile bool isTxFinished;
    /** Pointer to MCU registers structs */
    volatile struct SPI_REGS* pRegs;
    /** Rx interrupt identifier */
    uint32_t u32RxIntId;
    /** Tx interrupt identifier */
    uint32_t u32TxIntId;
} TSpiHandler;

static TSpiHandler tHandler[NUMBER_OF_SPI];
static struct TSpiPriv tPriv[NUMBER_OF_SPI];

#ifdef _FLASH
#pragma CODE_SECTION(SPIRx1Handler,"ramfuncs");
#pragma CODE_SECTION(SPITx1Handler,"ramfuncs");
#endif

interrupt void SPIRx1Handler(void);
interrupt void SPITx1Handler(void);

void SPIInit(uint16_t u16Id)
{
    switch (u16Id)
    {
        case SPI_INST_1:
#if defined(CPU1)
            GPIO_SetupPinOptions(
                    SPI1_CS_PIN, GPIO_OUTPUT,
                    (uint32_t)GPIO_PUSHPULL | (uint32_t)GPIO_ASYNC);
            GPIO_SetupPinMux(SPI1_CS_PIN, GPIO_MUX_CPU1, SPI1_CS_PIN_MAP);
            GPIO_SetupPinOptions(
                    SPI1_CLK_PIN, GPIO_OUTPUT,
                    (uint32_t)GPIO_PUSHPULL | (uint32_t)GPIO_ASYNC);
            GPIO_SetupPinMux(SPI1_CLK_PIN, GPIO_MUX_CPU1, SPI1_CLK_PIN_MAP);
            GPIO_SetupPinOptions(
                    SPI1_MOSI_PIN, GPIO_OUTPUT,
                    (uint32_t)GPIO_PUSHPULL | (uint32_t)GPIO_ASYNC);
            GPIO_SetupPinMux(SPI1_MOSI_PIN, GPIO_MUX_CPU1, SPI1_MOSI_PIN_MAP);
            GPIO_SetupPinOptions(
                    SPI1_MISO_PIN, GPIO_OUTPUT,
                    (uint32_t)GPIO_PUSHPULL | (uint32_t)GPIO_ASYNC);
            GPIO_SetupPinMux(SPI1_MISO_PIN, GPIO_MUX_CPU1, SPI1_MISO_PIN_MAP);

            GPIO_SetupPinOptions(SPI1_LOOPBACK_PIN, GPIO_OUTPUT,
                    (uint32_t)GPIO_PUSHPULL);
            GPIO_SetupPinMux(SPI1_LOOPBACK_PIN, GPIO_MUX_CPU1,
                    0);

            GPIO_SetupPinOptions(SPI1_IRQ_PIN, GPIO_OUTPUT, (uint32_t)GPIO_PUSHPULL);
            GPIO_SetupPinMux(SPI1_IRQ_PIN, GPIO_MUX_CPU1, 0);

            if (SPI1_CPU == GPIO_MUX_CPU1)
            {
                IntRegister(SPI1_INT_SRC_RX, SPIRx1Handler);
                IntRegister(SPI1_INT_SRC_TX, SPITx1Handler);
            }
#elif defined(CPU2)
            if (SPI1_CPU == GPIO_MUX_CPU1)
            {
                IntRegister(SPI1_INT_SRC_RX, SPIRx1Handler);
                IntRegister(SPI1_INT_SRC_TX, SPITx1Handler);
            }
#endif
            tHandler[SPI_INST_1].tInst.ptPriv = &tPriv[SPI_INST_1];
            tHandler[SPI_INST_1].tInst.ptPriv->u16Id = u16Id;
            tHandler[SPI_INST_1].tInst.ptPriv->u16Lpbk = SPI1_LOOPBACK_PIN;
            tHandler[SPI_INST_1].pRegs = &SPI1_REGS;
            tHandler[SPI_INST_1].u32RxIntId = SPI1_INT_SRC_RX;
            tHandler[SPI_INST_1].u32TxIntId = SPI1_INT_SRC_TX;
            tHandler[SPI_INST_1].tInst.isMaster = false;
            tHandler[SPI_INST_1].tInst.u32Br = SPI_DFLT_BR;
            tHandler[SPI_INST_1].tInst.u16Cs = SPI1_CS_PIN;
            tHandler[SPI_INST_1].tInst.u16Irq = SPI1_IRQ_PIN;
            break;
        default:
            /* Unsupported interface */
            break;
    }
}

void SPIDeinit(uint16_t u16Id)
{
    switch (u16Id)
    {
        case SPI_INST_1:
#if defined(CPU1)
            GPIO_SetupPinMux(SPI1_CLK_PIN, GPIO_MUX_CPU1, 0);
            GPIO_SetupPinMux(SPI1_MOSI_PIN, GPIO_MUX_CPU1, 0);
            GPIO_SetupPinMux(SPI1_MISO_PIN, GPIO_MUX_CPU1, 0);
            GPIO_SetupPinMux(SPI1_CS_PIN, GPIO_MUX_CPU1, 0);

            IntUnregister(SPI1_INT_SRC_RX);
            IntUnregister(SPI1_INT_SRC_TX);
#elif defined(CPU2)
            IntUnregister(SPI1_INT_SRC_RX);
            IntUnregister(SPI1_INT_SRC_TX);
#endif
            tHandler[SPI_INST_1].tInst.ptPriv = NULL;
            tHandler[SPI_INST_1].pRegs = NULL;
            tHandler[SPI_INST_1].u32RxIntId = (uint32_t)0UL;
            tHandler[SPI_INST_1].u32TxIntId = (uint32_t)0UL;
            break;
        default:
            /* Unsupported interface */
            break;
    }
}

void SPIGetInstance(TSpiIntf** ptInst, uint16_t u16Id)
{
    if (u16Id < NUMBER_OF_SPI)
    {
        *ptInst = &(tHandler[u16Id].tInst);
    }
}

void SPIStart(TSpiIntf* ptInst, const TSpiCfg* ptCfg)
{
    uint16_t u16SpiBr;

    if (ptInst != NULL)
    {
        switch (ptInst->ptPriv->u16Id)
        {
            case SPI_INST_1:
                if (ptCfg->u16RxBfrSize > MAX_SPI_BFR_SIZE)
                {
                    HND_ACCESS.tCfg.u16RxBfrSize = MAX_SPI_BFR_SIZE;
                }
                else
                {
                    HND_ACCESS.tCfg.u16RxBfrSize = ptCfg->u16RxBfrSize;
                }

                if (ptCfg->u16RxBfrSize > MAX_SPI_BFR_SIZE)
                {
                    HND_ACCESS.tCfg.u16TxBfrSize = MAX_SPI_BFR_SIZE;
                }
                else
                {
                    HND_ACCESS.tCfg.u16TxBfrSize = ptCfg->u16TxBfrSize;
                }

                HND_ACCESS.tCfg.pRxCtx = ptCfg->pRxCtx;
                HND_ACCESS.tCfg.ptRxEvntCB = ptCfg->ptRxEvntCB;
                HND_ACCESS.tCfg.pTxCtx = ptCfg->pTxCtx;
                HND_ACCESS.tCfg.ptTxEvntCB = ptCfg->ptTxEvntCB;
                HND_ACCESS.isTxFinished = true;

                /* Keep SPI in reset state during configuration */
                HND_ACCESS.pRegs->SPICCR.bit.SPISWRESET = 0;
                /* Config SPI as master */
                if (HND_ACCESS.tInst.isMaster == false)
                {
                    HND_ACCESS.pRegs->SPICTL.bit.MASTER_SLAVE = 0;
                }
                else
                {
                    HND_ACCESS.pRegs->SPICTL.bit.MASTER_SLAVE = 1;
                }
                /* Clock at high level if no data is transmited */
                HND_ACCESS.pRegs->SPICCR.bit.CLKPOLARITY = 1;

                /* Clock phase enabled */
                HND_ACCESS.pRegs->SPICTL.bit.CLK_PHASE = 1;

                /* Limit Baudrate for using peripheral clock */
                if (ptCfg->u32Br > 10000)
                {
                    /** SPI clock is system clock */
                    HND_ACCESS.pRegs->SPICCR.bit.HS_MODE = 1;
                    /** (SYSCLKOUT / BR SetPoint) -1 */
                    u16SpiBr = (200000 / ptCfg->u32Br) - 1;
                    HND_ACCESS.pRegs->SPIBRR.bit.SPI_BIT_RATE = u16SpiBr;
                    /** Update instance withreal applied baudrate */
                    /** (SYSCLKOUT / SPIBR + 1) */
                    HND_ACCESS.tInst.u32Br = 200000 / (u16SpiBr + 1);
                }
                else
                {
                    /* SPI clock is peripheral clock */
                    HND_ACCESS.pRegs->SPICCR.bit.HS_MODE = 0;
                    /** (Peripheral clock / BR SetPoint) -1 */
                    u16SpiBr = (50000 / ptCfg->u32Br) - 1;
                    HND_ACCESS.pRegs->SPIBRR.bit.SPI_BIT_RATE = u16SpiBr;
                    /** Update instance withreal applied baudrate */
                    /** (Peripheral clock / SPIBR + 1) */
                    HND_ACCESS.tInst.u32Br = 50000 / (u16SpiBr + 1);
                }

                /* Set default character length to 16 bits */
                HND_ACCESS.pRegs->SPICCR.bit.SPICHAR = 0xf;
                /* Clear flags */
                HND_ACCESS.pRegs->SPISTS.bit.OVERRUN_FLAG = 1;
                /* Enable FIFO enhancements */
                HND_ACCESS.pRegs->SPIFFTX.bit.SPIFFENA = 1;
                /* Clear FIFO flags */
                HND_ACCESS.pRegs->SPIFFTX.bit.TXFFINTCLR = 1;
                HND_ACCESS.pRegs->SPIFFRX.bit.RXFFINTCLR = 1;
                HND_ACCESS.pRegs->SPIFFRX.bit.RXFFOVFCLR = 1;
                /* Release FIFO from resets */
                HND_ACCESS.pRegs->SPIFFTX.bit.TXFIFO = 1;
                HND_ACCESS.pRegs->SPIFFRX.bit.RXFIFORESET = 1;
                HND_ACCESS.pRegs->SPIFFTX.bit.SPIRST = 1;
                /* Set FIFO interrupt levels */
                HND_ACCESS.pRegs->SPIFFTX.bit.TXFFIL = 0;
                HND_ACCESS.pRegs->SPIFFRX.bit.RXFFIL = HND_ACCESS.tCfg.u16RxBfrSize;
                /* Enable transmission */
                HND_ACCESS.pRegs->SPICTL.bit.TALK = 1;

                /* Enable reception interrupt */
                HND_ACCESS.pRegs->SPIFFRX.bit.RXFFIENA = 1;
                IntEnable(HND_ACCESS.u32RxIntId);
                /* enable transmission interrupt */
                HND_ACCESS.pRegs->SPIFFTX.bit.TXFFIENA = 1;
                IntEnable(HND_ACCESS.u32TxIntId);

                /* Release SPI module from reset state */
                HND_ACCESS.pRegs->SPICCR.bit.SPISWRESET = 1;
                break;
            default:
                /* Unsupported interface */
                break;
        }

    }
}

void SPIStop(const TSpiIntf* ptInst)
{
    if (ptInst != NULL)
    {
        /* Keep SPI in reset state during configuration */
        HND_ACCESS.pRegs->SPICCR.bit.SPISWRESET = 0;

        HND_ACCESS.tCfg.ptRxEvntCB = NULL;
        HND_ACCESS.tCfg.ptTxEvntCB = NULL;
    }
}

#ifdef CPU1
void SPICPU2Mapping(uint16_t u16Id)
{
    switch (u16Id)
    {
        case SPI_INST_1:
            /* Give UART control to selected CPU */
            EALLOW;
            switch (SPI1_BASE)
            {
                case SPIA_BASE:
                    DevCfgRegs.CPUSEL6.bit.SPI_A = 1;
                    break;
                case SPIB_BASE:
                    DevCfgRegs.CPUSEL6.bit.SPI_B = 1;
                    break;
                case SPIC_BASE:
                    DevCfgRegs.CPUSEL6.bit.SPI_C = 1;
                    break;
                default:
                    break;
            }
            EDIS;

            GPIO_SetupPinOptions(
                    SPI1_CS_PIN, GPIO_OUTPUT,
                (uint32_t)GPIO_PUSHPULL | (uint32_t)GPIO_ASYNC | (uint32_t)GPIO_PULLUP);
            GPIO_SetupPinMux(SPI1_CS_PIN, GPIO_MUX_CPU2, SPI1_CS_PIN_MAP);
            GPIO_SetupPinOptions(
                    SPI1_CLK_PIN, GPIO_OUTPUT,
                    (uint32_t)GPIO_PUSHPULL | (uint32_t)GPIO_ASYNC);
            GPIO_SetupPinMux(SPI1_CLK_PIN, GPIO_MUX_CPU2, SPI1_CLK_PIN_MAP);
            GPIO_SetupPinOptions(
                    SPI1_MOSI_PIN, GPIO_OUTPUT,
                    (uint32_t)GPIO_PUSHPULL | (uint32_t)GPIO_ASYNC);
            GPIO_SetupPinMux(SPI1_MOSI_PIN, GPIO_MUX_CPU2,
            SPI1_MOSI_PIN_MAP);
            GPIO_SetupPinOptions(
                    SPI1_MISO_PIN, GPIO_OUTPUT,
                    (uint32_t)GPIO_PUSHPULL | (uint32_t)GPIO_ASYNC);
            GPIO_SetupPinMux(SPI1_MISO_PIN, GPIO_MUX_CPU2,
            SPI1_MISO_PIN_MAP);

            GPIO_SetupPinOptions(SPI1_LOOPBACK_PIN, GPIO_OUTPUT,
            GPIO_PUSHPULL);
            GPIO_SetupPinMux(SPI1_LOOPBACK_PIN, GPIO_MUX_CPU2, 0);

            GPIO_SetupPinOptions(SPI1_IRQ_PIN, GPIO_OUTPUT, (uint32_t)GPIO_PUSHPULL);
            GPIO_SetupPinMux(SPI1_IRQ_PIN, GPIO_MUX_CPU2, 0);
            break;
        default:
            /* Nothing */
            break;
    }
}
#endif

void
SPIConfig(TSpiIntf* ptInst, const TSpiCfg* ptCfg)
{
    uint16_t u16SpiBr;

    if (ptInst != NULL)
    {
        /* Keep SPI in reset state during configuration */
        HND_ACCESS.pRegs->SPICCR.bit.SPISWRESET = 0;

        HND_ACCESS.tCfg.pRxCtx = ptCfg->pRxCtx;
        HND_ACCESS.tCfg.ptRxEvntCB = ptCfg->ptRxEvntCB;
        HND_ACCESS.tCfg.pTxCtx = ptCfg->pTxCtx;
        HND_ACCESS.tCfg.ptTxEvntCB = ptCfg->ptTxEvntCB;
        HND_ACCESS.isTxFinished = true;

        if (ptCfg->u16RxBfrSize > MAX_SPI_BFR_SIZE)
        {
            HND_ACCESS.tCfg.u16RxBfrSize = MAX_SPI_BFR_SIZE;
        }
        else
        {
            HND_ACCESS.tCfg.u16RxBfrSize = ptCfg->u16RxBfrSize;
        }

        if (ptCfg->u16RxBfrSize > MAX_SPI_BFR_SIZE)
        {
            HND_ACCESS.tCfg.u16TxBfrSize = MAX_SPI_BFR_SIZE;
        }
        else
        {
            HND_ACCESS.tCfg.u16TxBfrSize = ptCfg->u16TxBfrSize;
        }

        /* Config SPI as master */
        if (HND_ACCESS.tInst.isMaster == false)
        {
            HND_ACCESS.pRegs->SPICTL.bit.MASTER_SLAVE = 0;
        }
        else
        {
            HND_ACCESS.pRegs->SPICTL.bit.MASTER_SLAVE = 1;
        }

        /* Limit Baudrate for using peripheral clock */
        if (ptCfg->u32Br > 10000)
        {
            /** SPI clock is system clock */
            HND_ACCESS.pRegs->SPICCR.bit.HS_MODE = 1;
            /** (SYSCLKOUT / BR SetPoint) -1 */
            u16SpiBr = (200000 / ptCfg->u32Br) - 1;
            HND_ACCESS.pRegs->SPIBRR.bit.SPI_BIT_RATE = u16SpiBr;
            /** Update instance withreal applied baudrate */
            /** (SYSCLKOUT / SPIBR + 1) */
            HND_ACCESS.tInst.u32Br = 200000 / (u16SpiBr + 1);
        }
        else
        {
            /* SPI clock is peripheral clock */
            HND_ACCESS.pRegs->SPICCR.bit.HS_MODE = 0;
            /** (Peripheral clock / BR SetPoint) -1 */
            u16SpiBr = (50000 / ptCfg->u32Br) - 1;
            HND_ACCESS.pRegs->SPIBRR.bit.SPI_BIT_RATE = u16SpiBr;
            /** Update instance withreal applied baudrate */
            /** (Peripheral clock / SPIBR + 1) */
            HND_ACCESS.tInst.u32Br = 50000 / (u16SpiBr + 1);
        }

        /* Clear flags */
        HND_ACCESS.pRegs->SPISTS.bit.OVERRUN_FLAG = 1;
        /* Clear FIFO flags */
        HND_ACCESS.pRegs->SPIFFTX.bit.TXFFINTCLR = 1;
        HND_ACCESS.pRegs->SPIFFRX.bit.RXFFINTCLR = 1;
        HND_ACCESS.pRegs->SPIFFRX.bit.RXFFOVFCLR = 1;
        /* Release FIFO from resets */
        HND_ACCESS.pRegs->SPIFFTX.bit.TXFIFO = 1;
        HND_ACCESS.pRegs->SPIFFRX.bit.RXFIFORESET = 1;
        HND_ACCESS.pRegs->SPIFFTX.bit.SPIRST = 1;
        /* Set FIFO interrupt levels */
        HND_ACCESS.pRegs->SPIFFTX.bit.TXFFIL = 0;
        HND_ACCESS.pRegs->SPIFFRX.bit.RXFFIL = HND_ACCESS.tCfg.u16RxBfrSize;
        /* Enable transmission */
        HND_ACCESS.pRegs->SPICTL.bit.TALK = 1;
        /* Enable reception interrupt */
        HND_ACCESS.pRegs->SPIFFRX.bit.RXFFIENA = 1;
        IntEnable(HND_ACCESS.u32RxIntId);
        /* enable transmission interrupt */
        HND_ACCESS.pRegs->SPIFFTX.bit.TXFFIENA = 1;
        IntEnable(HND_ACCESS.u32TxIntId);
        /* Release SPI module from reset state */
        HND_ACCESS.pRegs->SPICCR.bit.SPISWRESET = 1;
    }
}

uint32_t SPIWrite(const TSpiIntf* ptInst, const uint16_t* pu16Src,
                  uint16_t u16Size)
{
    uint16_t u16Idx;
    const uint16_t* pu16InBfr = pu16Src;
    uint32_t u32RealWords = 0;

    if (ptInst != NULL)
    {
        for (u16Idx = 0; u16Idx < u16Size; u16Idx++)
        {
            if (HND_ACCESS.pRegs->SPIFFTX.bit.TXFFST < MAX_SPI_BFR_SIZE)
            {
                HND_ACCESS.pRegs->SPITXBUF = pu16InBfr[u16Idx];
                u32RealWords++;
            }
        }
        /* Enable interrupt */
        HND_ACCESS.pRegs->SPIFFTX.bit.TXFFINTCLR = 1;

        if (HND_ACCESS.tInst.isMaster == false)
        {
            GPIO_WritePin(HND_ACCESS.tInst.u16Irq, (uint16_t)HIGH);
        }
    }
    return (u32RealWords);
}

uint32_t SPIRead(const TSpiIntf* ptInst, uint16_t* pu16Dst, uint16_t u16Size)
{
    uint16_t u16Idx;
    uint32_t u32RealSize = 0;
    uint16_t* pu16OutBfr = pu16Dst;

    if (ptInst != NULL)
    {
        for (u16Idx = 0; u16Idx < u16Size; u16Idx++)
        {
            if (HND_ACCESS.pRegs->SPIFFRX.bit.RXFFST > 0)
            {
                pu16OutBfr[u16Idx] = HND_ACCESS.pRegs->SPIRXBUF;
                u32RealSize++;
            }
        }
    }

    return (u32RealSize);
}

uint16_t SPIAvailable(const TSpiIntf* ptInst)
{
    uint16_t u16Avbl = 0;

    if (ptInst != NULL)
    {
        u16Avbl = HND_ACCESS.pRegs->SPIFFRX.bit.RXFFST;
    }
    return u16Avbl;
}

uint16_t SPITxPending(const TSpiIntf* ptInst)
{
    uint16_t u16Avbl = 0;

        if (ptInst != NULL)
        {
            u16Avbl = HND_ACCESS.pRegs->SPIFFTX.bit.TXFFST;
        }
        return u16Avbl;
}

interrupt
void SPIRx1Handler(void)
{
    if (tHandler[SPI_INST_1].pRegs->SPIFFRX.bit.RXFFINT == 1)
    {
        if (tHandler[SPI_INST_1].tInst.isMaster == false)
        {
            GPIO_WritePin(tHandler[SPI_INST_1].tInst.u16Irq, (uint16_t)LOW);
        }

        if (tHandler[SPI_INST_1].tCfg.ptRxEvntCB != NULL)
        {
            tHandler[SPI_INST_1].tCfg.ptRxEvntCB(
                    tHandler[SPI_INST_1].tCfg.pRxCtx, NULL);
        }
    }

    tHandler[SPI_INST_1].pRegs->SPIFFRX.bit.RXFFINTCLR = 1;
    PieCtrlRegs.PIEACK.all = SPI1_INT_ACK;
}

interrupt
void SPITx1Handler(void)
{
    if (tHandler[SPI_INST_1].pRegs->SPIFFTX.bit.TXFFINT == 1)
    {
        if (tHandler[SPI_INST_1].tCfg.ptTxEvntCB != NULL)
        {
            tHandler[SPI_INST_1].tCfg.ptTxEvntCB(
                    tHandler[SPI_INST_1].tCfg.pTxCtx, NULL);
        }
    }

    PieCtrlRegs.PIEACK.all = SPI1_INT_ACK;
}

#endif
