/**
 * @file spi_interface.h
 * @brief This file offers a abstract layer that allows to use any spi module
 *
 * @author  Firmware department
 * @copyright Ingenia Motion Control (c) 2018. All rights reserved.
 */

/**
 * \addtogroup communications communications API
 *
 * @{
 */

#ifndef SPI_INTERFACE_H
#define SPI_INTERFACE_H

#include <stdint.h>
#include <stdbool.h>
#include "board_config.h"

#ifdef __cplusplus
extern "C"
{
#endif

/** Maximum internal buffer size for block TX/RX */
#define MAX_SPI_BFR_SIZE 16

/** Private SPI parameters */
struct TSpiPriv;

/** Generic event callback */
typedef void (*TSpiEvntCB)(void* Ctx, const uint16_t* u16Arg);

/** Spi interface struct */
typedef struct
{
    /** Indicates the SPI mode: master or slave */
    bool isMaster;
    /** Buadrate of the interface */
    uint32_t u32Br;
    /** Chip select pint identifier */
    uint16_t u16Cs;
    /** IRQ pin identifier */
    uint16_t u16Irq;
    /** Private parameters */
    struct TSpiPriv* ptPriv;
} TSpiIntf;

typedef struct
{
    /** Buadrate of the interface in kHz*/
    uint32_t u32Br;
    /** Size of RX buffer data */
    uint16_t u16RxBfrSize;
    /** Size of TX buffer data */
    uint16_t u16TxBfrSize;
    /* Pointer to store Rx Event context */
    void* pRxCtx;
    /** Callback for RX interrupts */
    TSpiEvntCB ptRxEvntCB;
    /* Pointer to store Tx Event context */
    void* pTxCtx;
    /** Callback for TX interrupts */
    TSpiEvntCB ptTxEvntCB;
} TSpiCfg;

/**
 * Initializes an SPI instance.
 *
 * @note The SPI is not operative after this function. @see SPIStart
 *
 * @param[in] u16Id
 *  Specifies the target SPI to be initialized
 */
void
SPIInit(uint16_t u16Id);

/**
 * Deinitializes an SPI instance.
 *
 * @param[in] u16Id
 *  Specifies the target SPI to be deinitialized
 */
void
SPIDeinit(uint16_t u16Id);

/**
 * Returns the desired instance
 *
 * @param[out] ptInst
 *  Pointer to store the target instance
 * @param[in] u16Id
 *  Identifier of the target instance
 */
void
SPIGetInstance(TSpiIntf** ptInst, uint16_t u16Id);

/**
 * Launches the SPI to start standard operation
 *
 * @param[in] ptInst
 *  Specifies the target SPI to be launched.
 * @param[in] ptCfg
 *  Specifies the configuration of the SPI.
 */
void
SPIStart(TSpiIntf* ptInst, const TSpiCfg* ptCfg);

/**
 * Stops the SPI.
 *
 * @param[in] ptInst
 *  Specifies the target SPI to be stopped.
 */
void
SPIStop(const TSpiIntf* ptInst);

/**
 * Map the peripheral to the CPU2.
 *
 * @param[in] u16Id
 *  Specifies the target spi to be configured
 */
void
SPICPU2Mapping(uint16_t u16Id);

/**
 * Configures an SPI instance.
 *
 * @param[in] ptInst
 *  Specifies the target SPI to be configured
 * @param[in] ptCfg
 *  Specifies the configuration of the SPI.
 */
void
SPIConfig(TSpiIntf* ptInst, const TSpiCfg* ptCfg);

/**
 * Sends data through an SPI instance.
 *
 * @param[in] ptInst
 *  Specifies the target SPI
 * @param[in] pu16Src
 *  Contains a pointer to the buffer of data to be sent
 * @param[in] u16Size
 *  Indicates the number of words to be sent
 *
 * @retval Number of words that have been sent
 */
uint32_t
SPIWrite(const TSpiIntf* ptInst, const uint16_t* pu16Src, uint16_t u16Size);

/**
 * Reads data through an SPI instance.
 *
 * @param[in] ptInst
 *  Specifies the target SPI
 * @param[in] pu16Dst
 *  Contains a pointer to the buffer of data to be written
 * @param[in] u16Size
 *  Indicates the number of words to be read
 *
 * @retval Number of words that have been read
 *
 * @see SPIAvailable
 */
uint32_t
SPIRead(const TSpiIntf* ptInst, uint16_t* pu16Dst, uint16_t u16Size);

/**
 * Returns the number of words available to be read.
 *
 * @param[in] ptInst
 *  Specifies the target SPI
 *
 * @retval Number of words that can be read
 *
 * @see SPIRead
 */
uint16_t
SPIAvailable(const TSpiIntf* ptInst);

/**
 * Returns the number of words pending in Tx buffer.
 *
 * @param[in] ptInst
 *  Specifies the target SPI
 *
 * @retval Number of words that havent been sent
 */
uint16_t
SPITxPending(const TSpiIntf* ptInst);

#ifdef __cplusplus
}
#endif

#endif /* SPI_INTERFACE_H */

/** @} */
