/**
 * @bootloader.c
 * @brief Main bootloader functions
 *
 * @author  Roger Juanpere
 * @copyright Ingenia Motion Control (c) 2017. All rights reserved.
 */

#include "bootloader.h"
#include <stdint.h>
#include <stdbool.h>
#include "board_config.h"
#include "flash_mng.h"
#include "version.h"
#include "board.h"
#include "ip_communications.h"
#include "input_output.h"


/* Constants ----- -----------------------------------------------------------*/
#define BOOT_WRITE_BLOCK_SIZE  40
/* Functions -----------------------------------------------------------------*/
void
BootGetVersion(uint16_t* pu16Major,
               uint16_t* pu16Minor,
               uint16_t* pu16Compile)
{
    *pu16Major = MAJOR_VERSION;
    *pu16Minor = MINOR_VERSION;
    *pu16Compile = COMPILE_VERSION;
}

void
BootGetAppAddress(uint32_t* pu32Address)
{
    *pu32Address = BOOT_APP_ADDRESS;
}

#if defined(CPU1)
int32_t
BootSetSignature(void)
{
    uint16_t u16Dummy;
    uint16_t u8Signature[8] =
            { 0xFF, 0xFF,
              0xFF, 0xFF,
              0xFF, 0xFF,
              0xFF, 0xFF,
            };
    return FlashWrite(BOOT_SHARED_ADDRESS, u8Signature, 8, &u16Dummy);
}

int32_t
BootEraseSignature(void)
{
    int32_t i32Err = BOOT_OK;

    /* Erase valid flag */
    i32Err = FlashErase(BOOT_SHARED_ADDRESS);

    return i32Err;
}
#endif

int32_t
BootErase(uint32_t u32Address)
{
    int32_t i32Err = BOOT_OK;
    /* Check boot loader zone */
    if (u32Address > BOOT_END_ADDRESS)
    {
        i32Err = FlashErase(u32Address);
    }
    else
    {
        i32Err = BOOT_EACCES;
    }
    return i32Err;
}

int32_t
BootWrite(uint32_t u32Address, uint16_t* pData, uint16_t u16Size)
{
    uint16_t u16Dummy;
    int32_t i32Err = BOOT_OK;
    uint16_t u16Temp[80];
    uint16_t u16SizeBytes = u16Size << 1;

    /* Check boot loader zone */
    /** input data are words, so change the code */
    if ((u32Address > BOOT_END_ADDRESS) &&
        (u16Size == BOOT_WRITE_BLOCK_SIZE))
    {
        i32Err = FlashWrite(u32Address, u16Temp, u16SizeBytes, &u16Dummy);
    }
    else
    {
        i32Err = BOOT_EACCES;
    }
    return i32Err;
}
