/**
 * @file @bootloader.c
 * @brief Main bootloader functions
 *
 * @author  Roger Juanpere
 * @copyright Ingenia Motion Control (c) 2017. All rights reserved.
 */

#ifndef BOOTLOADER_H_
#define BOOTLOADER_H_

#include <stdint.h>
#include <stdbool.h>
#include "boot.h"

/**
 * Returns the version of the bootloader.
 *
 * @param[out] pu8Major
 *  Pointer to Major version
 *
 * @param[out] pu8Minor
 *  Pointer to Minor version
 *
 * @param[out] pu8Compile
 *  Pointer to Compilation number
 */
void
BootGetVersion(uint16_t* pu16Major,
               uint16_t* pu16Minor,
               uint16_t* pu16Compile);

/**
 * Return the Flash memory address of the Final Application
 *
 * @param[out] pu32Address
 *  Pointer to Address
 */
void
BootGetAppAddress(uint32_t* pu32Address);


/**
 * Check GPIO MCB_SYNC1 and shared flash memory signature which will indicate that Final
 * Application is valid.
 *
 * @retval BOOT_OK if memory signature it is valid and GPIO not set, else BOOT_EAPP
 */
int32_t
BootCheckSignature(void);

#if defined(CPU1)
/**
 * Write a signature in shared Flash memory which will indicate that Final
 * Application is valid.
 *
 * @retval See BootErrorCode
 */
int32_t
BootSetSignature(void);

/**
 * Erase the signature in shared Flash memory which will indicate that Final
 * Application is invalid.
 *
 * @retval See BootErrorCode
 */
int32_t
BootEraseSignature(void);
#endif

/**
 * Erase a full sector of Flash memory if it does not affect bootloader.
 *
 * @details Erase the sector of Flash memory that contains the provided address
 *          if it does not overlap bootloader region.
 *
 * @param[in] u32Address
 *  Address of the flash sector to be erased.
 *
 * @note Sector address does not need to point to origin of the sector.
 *
 * @retval See BootErrorCode
 */
int32_t
BootErase(uint32_t u32Address);

/**
 * Write a byte array into Flash memory if it does not affect bootloader.
 *
 * @details Write a byte array into Flash memory on provided address if it does
 * not overlap bootloader region.
 *
 * @param[in] u32Address
 *  Target start flash using word addressing.
 *
 * @param[in] pData
 *  Pointer to encrypted byte array of data to be programmed.
 *
 * @param[in] u16Size
 *  Number of bytes in the encrypted byte array.
 *
 * @note pData must be expressed in big-endian
 *
 * @retval See BootErrorCode
 */
int32_t
BootWrite(uint32_t u32Address, uint16_t* pData, uint16_t u16Size);

/**
 * Unlock the MCU access.
 *
 * @param[in] pData
 *  Pointer to encrypted byte array of data to be programmed.
 *
 * @param[in] u16Size
 *  Number of bytes in the encrypted byte array.
 *
 * @retval See BootErrorCode
 */
int32_t
BootUnlock(uint16_t* pData, uint16_t u16Size);

#endif /* BOOTLOADER_H_ */
