/* @file serial_boot.h
 *
 * @brief This file offers a the NMT boot serice of the HSSP protocol
 *
 * @author  Firmware department
 * @copyright Ingenia Motion Control (c) 2016. All rights reserved.
 */

/**
 * \addtogroup Serial High Speed Serial Protocol
 *
 * @{
 *
 *   Protocol designed to use serial interfaces such as RS232/485 in an efficient way
 */

/**
 * \addtogroup SerialBoot Serial boot
 *
 *  @{
 *
 *  Application programming interface of the high speed serial protocol module
 *
 */

#ifndef SERIAL_H
#define SERIAL_H

#include <stdint.h>
#include <stdbool.h>

/** Bootloader protocol functions */
/** Check bootloader version command */
#define BOOT_CMD_VERSION            (uint16_t)0x67F
/** Open bootloader command */
#define BOOT_CMD_OPEN               (uint16_t)0x67B
/** Erase command */
#define BOOT_CMD_ERASE              (uint16_t)0x67D
/** Write command */
#define BOOT_CMD_WRITE              (uint16_t)0x67C
/** Close bootloader process command */
#define BOOT_CMD_CLOSE              (uint16_t)0x67E

/** Open EEPROM command */
#define BOOT_CMD_OPEN_EEPROM        (uint16_t)0x67A
/** EEPROM command */
#define BOOT_CMD_EEPROM             (uint16_t)0x679
/** Close EEPROM command */
#define BOOT_CMD_CLOSE_EEPROM       (uint16_t)0x678

/** Ingenia protocol static function requests/replies */
/** Read request */
#define MCB_REQ_READ                (uint16_t)0x1U
/** Write request */
#define MCB_REQ_WRITE               (uint16_t)0x2U
/** Idle request  */
#define MCB_REQ_IDLE                (uint16_t)0x7U

/** Acknowledge */
#define MCB_REP_ACK                 (uint16_t)0x3U
/** Error detected during read */
#define MCB_REP_READ_ERROR          (uint16_t)0x5U
/** Error detected during write */
#define MCB_REP_WRITE_ERROR         (uint16_t)0x6U

/** Boot Write command, data index */
#define BOOT_WRITE_DATA_IDX         (uint16_t)0x2U


/** Request frame struct */
typedef struct
{
    /** Network command */
    uint16_t u16Cmd;
    /** Target address. Not used */
    uint16_t u16Address;
    /** Size of received data */
    uint16_t u16Size;
    /** Array word with data */
    uint16_t u16Data[5];
} TRequest;

/** Serial instance */
typedef struct
{
    /** Instance ID */
    uint16_t u16Id;
    /** Indicates if a request is beeing processed */
    bool isRequest;
    /** Indicates if command ack has been send */
    volatile bool isReplySent;
    /** Indicates if end process command received */
    volatile bool isEndReceived;
    /** Message from master */
    TRequest tRequest;
    /** Message from slave */
    TRequest tReply;
    /** Current SPI baud-rate */
    uint16_t u16SPIBr;
    /** Is boot opened */
    bool isBootOpen;
    /** Is eeprom opened */
    bool isEepromOpen;
} TSerialInst;

/**
 * Initialize a serial instance. Only u16AvailableServices have to be specified before
 * initialization. This function can only be called once per available Serial
 * interface.
 *
 * @param[in] ptInst
 *  Serial intance to be initialized
 *
 * @retval ID of the initialized instance if all is ok.
 *         ALL_SERIAL_INTERFACES_USED if there isn't more available interfaces
 *         INCORRECT_SERIAL_INTERFACE for an incorrect initialization
 */
void
SerialBootInit(TSerialInst *ptInst);

/**
 * Deinitialize a serial instance. This function can only be called once per initilized
 * Serial interface.
 *
 * @param[in] ptInst
 *  Serial intance to be initialized
 */
void
SerialBootDeInit(TSerialInst *ptInst);

/**
 * Starts a serial instance.
 *
 * @param[in] ptInst
 *  Serial intance to be initialized
 *
 * @retval 0 if all is ok.
 */
void
SerialBootStart(TSerialInst *ptInst);

/**
 * Stops a serial instance.
 *
 * @param[in] ptInst
 *  Serial intance to be initialized
 *
 * @retval 0 if all is ok
 */
void
SerialBootStop(TSerialInst *ptInst);

/**
 * Main serial protocol routine. It must be called periodically.
 *
 * @param[in] ptInst
 *  Target serial intance
 */
bool
SerialBootProcess(TSerialInst *ptInst, uint32_t* pu32Address);

#endif /* SERIAL_SDO_H */

/** @} */
