//----------------------------------------------------------------------------------
//
// FILE:	CLAACCESS-DevInit_F2803x.c
//
// DESCRIPTION:	Device initialization for F2803x series
// 
//----------------------------------------------------------------------------------
//  Copyright Texas Instruments  2010
//----------------------------------------------------------------------------------

//============================================================================
// includes
//============================================================================

//#include "F2807x_device.h"     // Headerfile Include File
//#include "F2807x_settings.h"   // Settings Include File
//#include "F2807x_Cla_defines.h"
//typedef int _Bool;
#include <CLAACCESS-IRQ.h>
#include <CLAACCESS-Shared_C.h>
#include "F28x_Project.h"
#include "F2807x_Adc.h"
#include "F2807x_SysCtrl_defines.h"
#include "F2807x_EPwm_defines.h"

//============================================================================
// definitions
//============================================================================

#pragma diag_suppress 70,770

// Functions that will be run from RAM need to be assigned to
// a different section.  This section will then be mapped to a load and
// run address using the linker cmd file.
#pragma CODE_SECTION(InitFlash_Bank0, "ramfuncs");

// TODO: uncomment when WATCHDOG is requested
//#define USE_WATCHDOG
#define ENABLE_ADC_IRQ 1

#define ACQPS_ 100

//============================================================================
//  External function prototypes
//============================================================================


//extern void InitGpio();


//============================================================================
//  Function prototypes
//============================================================================

//void ISR_ILLEGAL(void);
//void CLA_Init(void);

//void CurrentCommandControl(void);
//void InitControl(void);
void PWM_1ch_CNF(int16 n, int16 period, int16 mode, int16 phase);

//============================================================================
//  Static Function prototypes
//============================================================================

//static void InitPeripherials(void);

//============================================================================
//  Variables
//============================================================================

// Used for running BackGround in flash, and ISR in RAM
//extern Uint16 *RamfuncsLoadStart, *RamfuncsLoadEnd, *RamfuncsRunStart;

// used by CLA Initialaization function
#ifdef _FLASH
extern Uint16 *Cla1funcsLoadStart, *Cla1funcsLoadEnd, *Cla1funcsRunStart,
		Cla1funcsLoadSize, *Cla1Prog_Start;
#endif

extern const struct PIE_VECT_TABLE PieVectTableInit;

extern __cregister volatile unsigned int IFR;
extern __cregister volatile unsigned int IER;

//============================================================================
// function definitions
//============================================================================

//============================================================================


#ifdef USE_WATCHDOG
/**
 * @brief This function initializes Watchdog function
 * @param  None
 * @retval None
 */
static void InitWatchdog(void)
{
	EALLOW;

//--- Disable the Watchdog Timer
	SysCtrlRegs.WDCR = 0x00A8;
// bit 15-8      0's:    reserved
// bit 7         1:      WDFLAG, write 1 to clear
// bit 6         0:      WDDIS, 1=disable WD
// bit 5-3       101:    WDCHK, WD check bits, always write as 101b
// bit 2-0       000:    WDPS, WD prescale bits, 000: WDCLK=OSCCLK/512/1

//--- System and Control Register
	SysCtrlRegs.SCSR = 0x0000;
// bit 15-3      0's:    reserved
// bit 2         0:      WDINTS, WD interrupt status bit (read-only)
// bit 1         0:      WDENINT, 0=WD causes reset, 1=WD causes WDINT
// bit 0         0:      WDOVERRIDE, write 1 to disable disabling of the WD (clear-only)

	EDIS;

}
#endif



/**
 * @brief
 * @param  None
 * @retval None
 */
static void CLA_configClaMemory(void) {
	EALLOW;

#ifdef _FLASH
	// Copy over code from FLASH to RAM
	memcpy((uint32_t *) &Cla1funcsRunStart, (uint32_t *) &Cla1funcsLoadStart,
			(uint32_t) &Cla1funcsLoadSize);
#endif //_FLASH

	// Initialize and wait for CLA1ToCPUMsgRAM
	MemCfgRegs.MSGxINIT.bit.INIT_CLA1TOCPU = 1;
	while (MemCfgRegs.MSGxINITDONE.bit.INITDONE_CLA1TOCPU != 1) {
	};

	// Initialize and wait for CPUToCLA1MsgRAM
	MemCfgRegs.MSGxINIT.bit.INIT_CPUTOCLA1 = 1;
	while (MemCfgRegs.MSGxINITDONE.bit.INITDONE_CPUTOCLA1 != 1) {
	};

	// Select LS5RAM to be the programming space for the CLA
	// First configure the CLA to be the master for LS5 and then
	// set the space to be a program block
	MemCfgRegs.LSxMSEL.bit.MSEL_LS5 = 1;
	MemCfgRegs.LSxCLAPGM.bit.CLAPGM_LS5 = 1;

	// Next configure LS0RAM and LS1RAM as data spaces for the CLA
	// First configure the CLA to be the master for LS0(1) and then
	// set the spaces to be code blocks
	MemCfgRegs.LSxMSEL.bit.MSEL_LS0 = 1;
	MemCfgRegs.LSxCLAPGM.bit.CLAPGM_LS0 = 0;

	MemCfgRegs.LSxMSEL.bit.MSEL_LS1 = 1;
	MemCfgRegs.LSxCLAPGM.bit.CLAPGM_LS1 = 0;

	EDIS;
}

// the following code sets up the CLA and assigns memory for CLA use
// It assigns only the program memory by default,
// the User can edit the file, for data RAM allocation to the CLA
// It also forces Task 8 which is used to carry out any initialization function
// on variables residing in the CLAtoCPUMsgRAM

/**
 * @brief Initialization of the CLA system
 * @param  None
 * @retval None
 */
static void CLA_C_Init(void) {
	// This code assumes the CLA clock is already enabled in
	// the call to DevInit();
	//
	// EALLOW: is needed to write to EALLOW protected registers
	// EDIS: is needed to disable write to EALLOW protected registers
	//
	EALLOW;

	Cla1Regs.MVECT1 = (uint16_t) (&Cla1Task1);
	Cla1Regs.MVECT2 = (uint16_t) (&Cla1Task1);
	Cla1Regs.MVECT3 = (uint16_t) (&Cla1Task1);
	Cla1Regs.MVECT4 = (uint16_t) (&Cla1Task4);
	Cla1Regs.MVECT5 = (uint16_t) (&Cla1Task1);
	Cla1Regs.MVECT1 = (uint16_t) (&Cla1Task1);
	Cla1Regs.MVECT7 = (uint16_t) (&Cla1Task1);
	Cla1Regs.MVECT8 = (uint16_t) (&Cla1Task1);

	// Enable the IACK instruction to start a task on CLA in software
	// for all  8 CLA tasks. Also, globally enable all 8 tasks (or a
	// subset of tasks) by writing to their respective bits in the
	// MIER register
	Cla1Regs.MCTL.bit.IACKE = 1;
	Cla1Regs.MIER.all = (0xFF);

	// Set the epwm.6 as the trigger for task 1
	// CLA_TRIG_EPWM1INT
	// CLA_TRIG_EPWM6INT
	// CLA_TRIG_ADCAINT1
	DmaClaSrcSelRegs.CLA1TASKSRCSEL1.bit.TASK4 = CLA_TRIG_EPWM4INT;

	EDIS;
}

/**
 * @brief ADC CALIBRATION
 * @param  None
 * @retval None
 */
static void ADCCalibration(void) {
	//---------------------------------------------------
	// The Device_cal function, which copies the ADC & oscillator calibration values
	// from TI reserved OTP into the appropriate trim registers, occurs automatically
	// in the Boot ROM. If the boot ROM code is bypassed during the debug process, the
	// following function MUST be called for the ADC and oscillators to function according
	// to specification.

	// migration TODO DONE
	// SysCtrlRegs.PCLKCR0.bit.ADCENCLK = 1; // Enable ADC peripheral clock
	CpuSysRegs.PCLKCR13.bit.ADC_A = 1;
	(*Device_cal )();                     // Auto-calibrate from TI OTP
	CpuSysRegs.PCLKCR13.bit.ADC_A = 0;

	// TODO ?
	 CpuSysRegs.PCLKCR13.bit.ADC_B = 1;
	 (*Device_cal )();                     // Auto-calibrate from TI OTP
	 CpuSysRegs.PCLKCR13.bit.ADC_B = 0;
}

/**
 * @brief  Configure the ADC and power it up
 * @param  None
 * @retval None
 */
void ADC_initAdcA(void) {
	uint16_t i;


	EALLOW;
	// write configurations
	AdcaRegs.ADCCTL2.bit.PRESCALE = 1;
	AdcSetMode(ADC_ADCA, ADC_RESOLUTION_12BIT, ADC_SIGNALMODE_SINGLE);

	// Set pulse positions to late
	AdcaRegs.ADCCTL1.bit.INTPULSEPOS = 1;

	// power up the ADC
	AdcaRegs.ADCCTL1.bit.ADCPWDNZ = 1;

	// delay for > 1ms to allow ADC time to power up
	for (i = 0; i < 1000; i++) {
		asm("   RPT#255 || NOP");
	}

	//Enable ADC burst mode
	AdcaRegs.ADCBURSTCTL.bit.BURSTEN = 1;
	// BURSTTRIG5  - ePWM1, ADCSOCA
	// BURSTTRIG9  - ePWM3, ADCSOCA
	// BURSTTRIG11 - ePWM4, ADCSOCA
	AdcaRegs.ADCBURSTCTL.bit.BURSTTRIGSEL = 9;
	//conversion bursts are 1+1=2 conversions long
	AdcaRegs.ADCBURSTCTL.bit.BURSTSIZE = 1;
	//SOC0 to SOC11 are high priority
	AdcaRegs.ADCSOCPRICTL.bit.SOCPRIORITY = 12;

	/*
    TRIGSEL = 5; //trigger on ePWM1 SOCA/C
    TRIGSEL = 7; //trigger on ePWM2 SOCA/C
    */

	// Select the channels to convert and end of conversion flag ADCA
	// ADCA-2 VSEC_OVP 0x2
	// ADCA-3 REFIN
	// ADCA-4 OUT_VOLTAGE 0x04
	AdcaRegs.ADCSOC0CTL.bit.CHSEL = 2;  	// SOC0 will convert pin A2
	AdcaRegs.ADCSOC0CTL.bit.ACQPS = ACQPS_;	// sample window is n SYSCLK cycles
	AdcaRegs.ADCSOC0CTL.bit.TRIGSEL = 0xB;  	// trigger on ePWM4 SOCA
	AdcaRegs.ADCSOC1CTL.bit.CHSEL = 4;  	// SOC1 will convert pin A4
	AdcaRegs.ADCSOC1CTL.bit.ACQPS = ACQPS_;	// sample window is n SYSCLK cycles
	AdcaRegs.ADCSOC1CTL.bit.TRIGSEL = 0xC;  	// trigger on ePWM4 SOCB

	AdcaRegs.ADCINTSEL1N2.bit.INT1SEL = 0;  // end of SOC0 will set INT1 flag
	AdcaRegs.ADCINTSEL1N2.bit.INT1E = 1;    // enable INT1 flag
	AdcaRegs.ADCINTFLGCLR.bit.ADCINT1 = 1;  // make sure INT1 flag is cleared
	EDIS;
}

/**
 * @brief  Configure the ADC and power it up
 * @param  None
 * @retval None
 */
void ADC_initAdcB(void) {
	uint16_t i;

	EALLOW;

	// write configurations
	AdcbRegs.ADCCTL2.bit.PRESCALE = 1;
	AdcSetMode(ADC_ADCB, ADC_RESOLUTION_12BIT, ADC_SIGNALMODE_SINGLE);

	// Set pulse positions to late
	AdcbRegs.ADCCTL1.bit.INTPULSEPOS = 1;

	// power up the ADC
	AdcbRegs.ADCCTL1.bit.ADCPWDNZ = 1;

	// delay for > 1ms to allow ADC time to power up
	for (i = 0; i < 1000; i++) {
		asm("   RPT#255 || NOP");
	}

	// http://www.ti.com/lit/ug/spruhm9b/spruhm9b.pdf
	// 9.1.7.1
	// Enable ADC burst mode
//	AdcbRegs.ADCBURSTCTL.bit.BURSTEN = 1;
	// BURSTTRIG5  - ePWM1, ADCSOCA
	// BURSTTRIG9  - ePWM3, ADCSOCA
	// BURSTTRIG11 - ePWM4, ADCSOCA
//	AdcbRegs.ADCBURSTCTL.bit.BURSTTRIGSEL = 11;
	// conversion bursts are 1+1=2 conversions long
//	AdcbRegs.ADCBURSTCTL.bit.BURSTSIZE = 1;
	// SOC0 to SOC11 are high priority
	AdcbRegs.ADCSOCPRICTL.bit.SOCPRIORITY = 12;

	// Select the channels to convert and end of conversion flag ADCA
	// ADCB-1 AUX_VOLTAGE
	// ADCB-2 IN_CURRENT
	// ADCB-4 IN_VOLTAGE
	// ADCB-5 VCLAMP
	AdcbRegs.ADCSOC0CTL.bit.CHSEL = 2;  	// SOC0 will convert pin B2
	AdcbRegs.ADCSOC0CTL.bit.ACQPS = ACQPS_;	// sample window is n SYSCLK cycles
	AdcbRegs.ADCSOC0CTL.bit.TRIGSEL = 11;  	// TODO: trigger on ePWM4 SOCA/C
	AdcbRegs.ADCSOC1CTL.bit.CHSEL = 4;  	// SOC1 will convert pin B4
	AdcbRegs.ADCSOC1CTL.bit.ACQPS = ACQPS_;	// sample window is n SYSCLK cycles
	AdcbRegs.ADCSOC1CTL.bit.TRIGSEL = 11;  	// TODO: trigger on ePWM4 SOCA/C

	AdcbRegs.ADCINTSEL1N2.bit.INT2SEL = 0;  // end of SOC0 will set INT2 flag
	AdcbRegs.ADCINTSEL1N2.bit.INT2E = 1;    // enable INT2 flag
	AdcbRegs.ADCINTFLGCLR.bit.ADCINT2 = 1;  // make sure INT1 flag is cleared

	EDIS;
}

/**
 * @brief Application specific peripherial initialization
 * @param  None
 * @retval None
 */
static void InitPeripherials(void) {
	EALLOW;

	// Configure EPWM to run at SYSCLK
	ClkCfgRegs.PERCLKDIVSEL.bit.EPWMCLKDIV = 1;
	// TODO ?
	CpuSysRegs.PCLKCR0.bit.TBCLKSYNC = 1;

	//
	// PWM L & H for main power supply
	//
	// init epwm6 pins
	GpioCtrlRegs.GPAPUD.bit.GPIO10 = 1;    // Disable pull-up on GPIO10 (EPWM6A)
	GpioCtrlRegs.GPAPUD.bit.GPIO11 = 1;    // Disable pull-up on GPIO11 (EPWM6B)
	GpioCtrlRegs.GPAMUX1.bit.GPIO10 = 1;   // Configure GPIO10 as EPWM6A
	GpioCtrlRegs.GPAMUX1.bit.GPIO11 = 1;   // Configure GPIO11 as EPWM6B
	EPwm6Regs.CMPA.bit.CMPA = PWM_PERIOD / 2;
	EPwm6Regs.CMPB.bit.CMPB = PWM_PERIOD / 2;

	//
	// Configure PWM4 for current command ISR triggering
	//
	PWM_1ch_CNF(4, IRQ_TASK_PERIOD, 1, 0);
	EPwm4Regs.CMPA.bit.CMPA = IRQ_TASK_PERIOD;
	EPwm4Regs.ETSEL.bit.INTSEL = ET_CTR_ZERO;  	// INT on TBCTR=0x00
	EPwm4Regs.ETSEL.bit.INTEN = 1;              // Enable INT
	EPwm4Regs.ETPS.bit.INTPRD = ET_1ST;         // Generate INT on every event
	EPwm4Regs.ETSEL.bit.SOCAEN	= 1;	        // Enable SOC on A group
	EPwm4Regs.ETPS.bit.SOCAPRD = 1;		        // Generate pulse on 1st event
	EPwm4Regs.ETSEL.bit.SOCASEL	= 0x1;	        // Select SOCA on TBCTR=0

	//
	// configure epwm3 for cla tracing
	//
	GpioCtrlRegs.GPAMUX1.bit.GPIO4 = 1;   		// Configure GPIO4 as EPWM3A
	PWM_1ch_CNF(3, IRQ_TASK_PERIOD, 1, 0);
	EPwm3Regs.CMPA.bit.CMPA = IRQ_TASK_PERIOD - 200;

	//EPwm1Regs.CMPA.bit.CMPA = 250;
	CpuSysRegs.SECMSEL.all = 15;

	// configure ADC
	ADC_initAdcA();
	ADC_initAdcB();
	EDIS;
}

/**
 * @brief INTERRUPTS & ISR INITIALIZATION
 * @param  None
 * @retval None
 */
static void InitInterrupts(void) {
	EALLOW;

	asm("   RPT #3 || NOP");

	// EPWM4 INT
	PieVectTable.EPWM4_INT = &CLAACCESS_EPWM4_ISR;    	// Map Interrupt
	PieCtrlRegs.PIEIER3.bit.INTx4 = 1;          // PIE level enable, EPWM4: Grp3 / Int4
    EPwm4Regs.ETCLR.bit.INT = 1;
	IER |= (M_INT1 | M_INT2 | M_INT3 | M_INT4);

	// ADCA/B INTs
#if ENABLE_ADC_IRQ

	PieVectTable.ADCA1_INT = &CLAACCESS_ADCA1_ISR;    	// Map Interrupt
	PieCtrlRegs.PIEIER1.bit.INTx1 = 1;          // PIE level enable, ADCA: Grp1 / Int1
	AdcaRegs.ADCINTFLGCLR.bit.ADCINT1 = 1;  	// make sure INT1 flag is cleared

	PieVectTable.ADCB1_INT = &CLAACCESS_ADCB1_ISR;    	// Map Interrupt
	PieCtrlRegs.PIEIER1.bit.INTx2 = 1;          // PIE level enable, ADCB: Grp1 / Int2
	AdcbRegs.ADCINTFLGCLR.bit.ADCINT2 = 1;  	// make sure INT1 flag is cleared

	IER |= (M_INT1 | M_INT2);
#endif

	// Enable the PIE Vector Table
	PieCtrlRegs.PIECTRL.bit.ENPIE = 1;

	// Enable Global interrupt INTM
	EINT;

	// Enable Global realtime interrupt DBGM
	ERTM;

	EDIS;
}

/**
 * @brief Configure Device for target Application Here
 * @param  None
 * @retval None
 */
void DeviceInit(void) {
	// Disable the watchdog initially
	DisableDog();

	//
	// Global Disable all Interrupts
	//
	EALLOW;
	DINT;
	// Disable CPU interrupts
	IER = 0x0000;
	// Clear all CPU interrupt flags
	IFR = 0x0000;
	EDIS;

	// SYSTEM CLOCK speed based on internal oscillator
	//PLLSYSCLK = 10MHz(INT_OSC) * 24 (IMULT) * 1 (FMULT) /  2 (PLLCLK_BY_2)
	InitSysPll(INT_OSC1, IMULT_24, FMULT_0, PLLCLK_BY_2);

	// migration TODO DONE
	// InitAuxPll(); needed?

	// Initialise interrupt controller and Vector Table
	// to defaults for now. Application ISR mapping done later.
	InitPieCtrl();
	InitPieVectTable();

	ADCCalibration();

	InitPeripheralClocks();

#ifdef USE_WATCHDOG
	InitWatchdog();
#endif

	EDIS;
	// Disable register access

	// Only used if running from FLASH
	// Note that the variable FLASH is defined by the compiler with -d FLASH

#ifdef _FLASH
	// Copy time critical code and Flash setup code to RAM
	// The  RamfuncsLoadStart, RamfuncsLoadEnd, and RamfuncsRunStart
	// symbols are created by the linker. Refer to the linker files.
	MemCopy((Uint16*) &RamfuncsLoadStart, (Uint16*) &RamfuncsLoadEnd,
			(Uint16*) &RamfuncsRunStart);

	// Call Flash Initialization to setup flash waitstates
	// This function must reside in RAM
	InitFlash_Bank0();
#endif

	EALLOW;
	CpuTimer1Regs.PRD.all = mSec100;
	CpuTimer2Regs.PRD.all = mSec1;
	EDIS;

	InitGpio();

	// initialize test pins
	// LED
	GPIO_SetupPinMux(84, GPIO_MUX_CPU1, 0);
	GPIO_SetupPinOptions(84, GPIO_OUTPUT, GPIO_PUSHPULL);
	// Test Pin #1
	GPIO_SetupPinMux(89, GPIO_MUX_CPU1, 0);
	GPIO_SetupPinOptions(89, GPIO_OUTPUT, GPIO_PUSHPULL);
	// CLA trace pin
	//GPIO_SetupPinMux(4, GPIO_MUX_CPU1, 0);
	//GPIO_SetupPinOptions(4, GPIO_OUTPUT, GPIO_PUSHPULL);
	// Test Pin #2
	GPIO_SetupPinMux(3, GPIO_MUX_CPU1, 0);
	GPIO_SetupPinOptions(3, GPIO_OUTPUT, GPIO_PUSHPULL);

	// init CLA subsystem
	CLA_configClaMemory();
	CLA_C_Init();

	// init cla task
	// Cla1ForceTask8();

	// app specific settings
	InitPeripherials();

	InitInterrupts();

	// init application specific system
	//InitControl();

	Cla1ForceTask1();
}

// Used to indirectly access all EPWM modules
volatile struct EPWM_REGS *ePWM[] =
{
        &EPwm1Regs,         //intentional: (ePWM[0] not used)
        &EPwm1Regs,
        &EPwm2Regs,
        &EPwm3Regs,
        &EPwm4Regs,
        &EPwm5Regs,
        &EPwm6Regs,
        &EPwm7Regs
};

/**
 * @brief int16 PWMn, int16 period, int16 mode, int16 phase)
 * @param  None
 * @retval None
 */
void PWM_1ch_CNF(int16 n, int16 period, int16 mode, int16 phase)
{
	// Time Base SubModule Registers
	(*ePWM[n]).TBCTL.bit.PRDLD = TB_IMMEDIATE;	// set immediate load
	(*ePWM[n]).TBPRD = period-1;					// PWM frequency = 1 / period
	// migration TODO DONE
	(*ePWM[n]).TBPHS.all = 0;
	(*ePWM[n]).TBCTR = 0;
	(*ePWM[n]).TBCTL.bit.CTRMODE = TB_COUNT_UP;
	(*ePWM[n]).TBCTL.bit.HSPCLKDIV = TB_DIV1;
	(*ePWM[n]).TBCTL.bit.CLKDIV = TB_DIV1;

	if(mode == 1) // config as a Master
	{
		(*ePWM[n]).TBCTL.bit.PHSEN = TB_DISABLE;
		(*ePWM[n]).TBCTL.bit.SYNCOSEL = TB_CTR_ZERO; // sync "down-stream"
	}
	if(mode == 0) // config as a Slave (Note: Phase+2 value used to compensate for logic delay)
	{
		(*ePWM[n]).TBCTL.bit.PHSEN = TB_ENABLE;
		(*ePWM[n]).TBCTL.bit.SYNCOSEL = TB_SYNC_IN;

		// migration TODO DONE
		if ((0 <= phase)&&(phase <= 2))
			(*ePWM[n]).TBPHS.all = (2-phase);
		else if (phase > 2)
			(*ePWM[n]).TBPHS.all = (period-phase+2);
	}

	// Counter Compare Submodule Registers
	// migration TODO DONE
	(*ePWM[n]).CMPA.all = 0;						// set duty 0% initially
	(*ePWM[n]).CMPB.all = 0;						// set duty 0% initially
	(*ePWM[n]).CMPCTL.bit.SHDWAMODE = CC_SHADOW;
	(*ePWM[n]).CMPCTL.bit.LOADAMODE = CC_CTR_PRD;

	// Action Qualifier SubModule Registers
	(*ePWM[n]).AQCTLA.bit.ZRO = AQ_SET;
	(*ePWM[n]).AQCTLA.bit.CAU = AQ_CLEAR;

	(*ePWM[n]).AQCTLB.bit.ZRO = AQ_NO_ACTION;
	(*ePWM[n]).AQCTLB.bit.CAU = AQ_NO_ACTION;
	(*ePWM[n]).AQCTLB.bit.PRD = AQ_NO_ACTION;
}


//===========================================================================
// End of file.
//===========================================================================
