//###########################################################################
// FILE:   epwm_updown_aq_cpu01.c
// TITLE:  Action Qualifier Module - Using up/down count.
//
//! \addtogroup cpu01_example_list
//! <h1> EPWM Action Qualifier (epwm_updown_aq)</h1>
//!
//! This example configures ePWM1, ePWM2, ePWM3 to produce an
//! waveform with independent modulation on EPWMxA and
//! EPWMxB.
//!
//! The compare values CMPA and CMPB are modified within the ePWM's ISR.
//!
//! The TB counter is in up/down count mode for this example.
//!
//! View the EPWM1A/B(PA0_GPIO0 & PA1_GPIO1), EPWM2A/B(PA2_GPIO2 & PA3_GPIO3)
//! and EPWM3A/B(PA4_GPIO4 & PA5_GPIO5) waveforms via an oscilloscope.
//
//###########################################################################
// $TI Release: F2837xD Support Library v150 $
// $Release Date: Thu Mar  5 14:18:39 CST 2015 $
// $Copyright: Copyright (C) 2013-2015 Texas Instruments Incorporated -
//             http://www.ti.com/ ALL RIGHTS RESERVED $
//###########################################################################

#include "F28x_Project.h"     // Device Headerfile and Examples Include File

typedef struct
{
    volatile struct EPWM_REGS *EPwmRegHandle;
    Uint16 EPwm_CMPA_Direction;
    Uint16 EPwm_CMPB_Direction;
    Uint16 EPwmTimerIntCount;
    Uint16 EPwmMaxCMPA;
    Uint16 EPwmMinCMPA;
    Uint16 EPwmMaxCMPB;
    Uint16 EPwmMinCMPB;
}EPWM_INFO;

// Prototype statements for functions found within this file.
void InitEPwm1Example(void);
void InitEPwm2Example(void);
void InitEPwm3Example(void);
__interrupt void epwm1_isr(void);
__interrupt void epwm2_isr(void);
__interrupt void epwm3_isr(void);
void update_compare(EPWM_INFO*);

// Global variables used in this example
EPWM_INFO epwm1_info;
EPWM_INFO epwm2_info;
EPWM_INFO epwm3_info;

// Configure the period for each timer
#define EPWM1_TIMER_TBPRD  242  // Period register
#define EPWM1_MAX_CMPA     1950
#define EPWM1_MIN_CMPA       50
#define EPWM1_MAX_CMPB     1950
#define EPWM1_MIN_CMPB       50

#define EPWM2_TIMER_TBPRD  242  // Period register
#define EPWM2_MAX_CMPA     1950
#define EPWM2_MIN_CMPA       50
#define EPWM2_MAX_CMPB     1950
#define EPWM2_MIN_CMPB       50

#define EPWM3_TIMER_TBPRD  2000  // Period register
#define EPWM3_MAX_CMPA      950
#define EPWM3_MIN_CMPA       50
#define EPWM3_MAX_CMPB     1950
#define EPWM3_MIN_CMPB     1050

// To keep track of which way the compare value is moving
#define EPWM_CMP_UP   1
#define EPWM_CMP_DOWN 0


//definitions for selecting DACH reference
#define REFERENCE_VDDA     0
#define REFERENCE_VDAC     1

//definitions for COMPH input selection
#define NEGIN_DAC          0
#define NEGIN_PIN          1

//definitions for CTRIPH/CTRIPOUTH output selection
#define CTRIP_ASYNCH       0
#define CTRIP_SYNCH        1
#define CTRIP_FILTER       2
#define CTRIP_LATCH        3

//definitions for selecting output pin
#define GPIO_CTRIP_PIN_NUM      14 //OUTPUTXBAR3 is mux'd with GPIO14
#define GPIO_CTRIP_PER_NUM       6 //OUTPUTXBAR3 is peripheral option 6 for GPIO14
#define GPIO_CTRIPOUT_PIN_NUM   15 //EPWM8B is mux'd with GPIO15
#define GPIO_CTRIPOUT_PER_NUM    1 //EPWM8B is peripheral option 1 for GPIO15

unsigned short change_method = 0 , method = 0;

void InitCMPSS(void);

void main(void)
{
// Step 1. Initialize System Control:
// PLL, WatchDog, enable Peripheral Clocks
// This example function is found in the F2837xD_SysCtrl.c file.
    InitSysCtrl();

// Step 2. Initialize GPIO:
// This example function is found in the F2837xD_Gpio.c file and
// illustrates how to set the GPIO to it's default state.
//    InitGpio();

//allocate PWM1, PWM2 and PWM3 to CPU1
//    CpuSysRegs.CPUSEL0.bit.EPWM1 = 0;
//    CpuSysRegs.CPUSEL0.bit.EPWM2 = 0;
//    CpuSysRegs.CPUSEL0.bit.EPWM3 = 0;

// enable PWM1, PWM2 and PWM3
    CpuSysRegs.PCLKCR2.bit.EPWM1=1;
    CpuSysRegs.PCLKCR2.bit.EPWM2=1;
    CpuSysRegs.PCLKCR2.bit.EPWM3=1;

// For this case just init GPIO pins for ePWM1, ePWM2, ePWM3
// These functions are in the F28M36x_EPwm.c file
    InitEPwm1Gpio();
	InitEPwm2Gpio();
	InitEPwm3Gpio();

	InitCMPSS();

// Step 3. Clear all interrupts and initialize PIE vector table:
// Disable CPU interrupts
    DINT;

// Initialize the PIE control registers to their default state.
// The default state is all PIE interrupts disabled and flags
// are cleared.
// This function is found in the F2837xD_PieCtrl.c file.
    InitPieCtrl();

// Disable CPU interrupts and clear all CPU interrupt flags:
    IER = 0x0000;
    IFR = 0x0000;

// Initialize the PIE vector table with pointers to the shell Interrupt
// Service Routines (ISR).
// This will populate the entire table, even if the interrupt
// is not used in this example.  This is useful for debug purposes.
// The shell ISR routines are found in F2837xD_DefaultIsr.c.
// This function is found in F2837xD_PieVect.c.
    InitPieVectTable();

// Interrupts that are used in this example are re-mapped to
// ISR functions found within this file.
	EALLOW; // This is needed to write to EALLOW protected registers
	PieVectTable.EPWM1_INT = &epwm1_isr;
	PieVectTable.EPWM2_INT = &epwm2_isr;
	PieVectTable.EPWM3_INT = &epwm3_isr;
	EDIS;   // This is needed to disable write to EALLOW protected registers

// For this example, only initialize the ePWM

	EALLOW;
	CpuSysRegs.PCLKCR0.bit.TBCLKSYNC =0;

	EDIS;

	// phase 1
	InitEPwm1Example();
	InitEPwm2Example();
	InitEPwm3Example();
	InitEPwm4Example();

	// phase 2
    //InitEPwm5Example();
    //InitEPwm6Example();
    //InitEPwm7Example();
    //InitEPwm8Example();
	EALLOW;

    //Configure TRIP4 to be CTRIP1H
    EPwmXbarRegs.TRIP4MUX0TO15CFG.bit.MUX0 = 0;
    //Enable TRIP4 Mux for Output
    EPwmXbarRegs.TRIP4MUXENABLE.bit.MUX0   = 1;
    // inputXbar1
    EPwmXbarRegs.TRIP5MUX0TO15CFG.bit.MUX1 = 1;
    //Enable TRIP4 Mux for Output
    EPwmXbarRegs.TRIP5MUXENABLE.bit.MUX1   = 1;



    //Use VDAC as the reference for DAC
    DacaRegs.DACCTL.bit.DACREFSEL  = REFERENCE_VDAC;
    //Enable DAC output
    DacaRegs.DACOUTEN.bit.DACOUTEN = 1;
    DacaRegs.DACVALS.all =  800;


    EDIS;


    // Configure GPIO15 to output CTRIP1H/EPWM TRIP4
        GPIO_SetupPinMux(   GPIO_CTRIP_PIN_NUM, GPIO_MUX_CPU1,    GPIO_CTRIP_PER_NUM);
    // Configure GPIO14 to output CTRIPOUT1H
        GPIO_SetupPinMux(GPIO_CTRIPOUT_PIN_NUM, GPIO_MUX_CPU1, GPIO_CTRIPOUT_PER_NUM);

    // configure input cross bar
    EALLOW;
    InputXbarRegs.INPUT1SELECT = 6; // GPIO06 --> INPUT1
    EDIS;



	EALLOW;
	CpuSysRegs.PCLKCR0.bit.TBCLKSYNC =1;

	EDIS;

// Step 4. User specific code, enable interrupts:

// Enable CPU INT3 which is connected to EPWM1-3 INT:
	IER |= M_INT3;

// Enable EPWM INTn in the PIE: Group 3 interrupt 1-3
	PieCtrlRegs.PIEIER3.bit.INTx1 = 1;
	PieCtrlRegs.PIEIER3.bit.INTx2 = 1;
	PieCtrlRegs.PIEIER3.bit.INTx3 = 1;


// Enable global Interrupts and higher priority real-time debug events:
    EINT;  // Enable Global interrupt INTM
    ERTM;  // Enable Global realtime interrupt DBGM

// Step 5. IDLE loop. Just sit and loop forever (optional):
    for(;;)
    {
        asm ("          NOP");

        if(change_method ==1)
        {
            switch(method)
            {
                case 0:
                    EALLOW;
                    EPwm2Regs.DCBHTRIPSEL.all = 0;               // clear OR source
                    EPwm2Regs.DCTRIPSEL.bit.DCBHCOMPSEL = 15;
                    EPwm2Regs.DCBHTRIPSEL.bit.TRIPINPUT4 =1;        // CMSS input trigger T1/T2

                    EDIS;
                    break;
                case 1:
                    EALLOW;
                    // Set GPIO06 to INPUT1
                    // INPUT1 to be TZ1/TRIPINPT1 for EPWM
                    EPwm2Regs.DCTRIPSEL.bit.DCBHCOMPSEL = 0;
                    EDIS;
                    break;

                case 3:
                    EALLOW;
                    EPwm2Regs.DCBHTRIPSEL.all = 0;               // clear OR source
                    EPwm2Regs.DCTRIPSEL.bit.DCBHCOMPSEL = 15;
                    EPwm2Regs.DCBHTRIPSEL.bit.TRIPINPUT5 =1;        //  using inputxbar1 to be source for T1/T2 through EPWMXBAR
                    EDIS;
                    break;
            }
            change_method = 0;
        }

    }
}

__interrupt void epwm1_isr(void)
{
    // Update the CMPA and CMPB values
    //update_compare(&epwm1_info);

    // Clear INT flag for this timer
    EPwm1Regs.ETCLR.bit.INT = 1;

    // Acknowledge this interrupt to receive more interrupts from group 3
    PieCtrlRegs.PIEACK.all = PIEACK_GROUP3;
}

__interrupt void epwm2_isr(void)
{

    // Update the CMPA and CMPB values
    //update_compare(&epwm2_info);

    // Clear INT flag for this timer
    EPwm2Regs.ETCLR.bit.INT = 1;

    // Acknowledge this interrupt to receive more interrupts from group 3
    PieCtrlRegs.PIEACK.all = PIEACK_GROUP3;
}

__interrupt void epwm3_isr(void)
{

    // Update the CMPA and CMPB values
    //update_compare(&epwm3_info);

    // Clear INT flag for this timer
    EPwm3Regs.ETCLR.bit.INT = 1;

    // Acknowledge this interrupt to receive more interrupts from group 3
    PieCtrlRegs.PIEACK.all = PIEACK_GROUP3;
}

void InitEPwm1Example()
{

    // Setup TBCLK
    EPwm1Regs.TBPRD = EPWM1_TIMER_TBPRD;          // Set timer period 801 TBCLKs
    EPwm1Regs.TBPHS.bit.TBPHS = 0x0000;           // Phase is 0
    EPwm1Regs.TBCTR = 0x0000;                     // Clear counter

    // Set Compare values
    EPwm1Regs.CMPA.bit.CMPA = EPWM1_MIN_CMPA;    // Set compare A value
    EPwm1Regs.CMPB.bit.CMPB = EPWM1_MAX_CMPB;    // Set Compare B value

    // Setup counter mode
    EPwm1Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN; // Count up and down
    EPwm1Regs.TBCTL.bit.PHSEN = TB_DISABLE;       // Disable phase loading
    EPwm1Regs.TBCTL2.bit.SYNCOSELX = 0;
    EPwm1Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1;      // Clock ratio to SYSCLKOUT
    EPwm1Regs.TBCTL.bit.CLKDIV = TB_DIV1;

    // Setup shadowing
    EPwm1Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;
    EPwm1Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;
    EPwm1Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO; // Load on Zero
    EPwm1Regs.CMPCTL.bit.LOADBMODE = CC_CTR_ZERO;

    // Set actions
#if 0
    EPwm1Regs.AQCTLA.bit.CAU = AQ_SET;            // Set PWM1A on event A, up
                                                   // count
    EPwm1Regs.AQCTLA.bit.CAD = AQ_CLEAR;          // Clear PWM1A on event A,
                                                  // down count

    EPwm1Regs.AQCTLB.bit.CBU = AQ_SET;            // Set PWM1B on event B, up
                                                  // count
    EPwm1Regs.AQCTLB.bit.CBD = AQ_CLEAR;          // Clear PWM1B on event B,
#else                                                  // down count

    EPwm1Regs.AQCTLA.bit.ZRO = AQ_SET;
    EPwm1Regs.AQCTLA.bit.PRD = AQ_CLEAR;

    EPwm1Regs.AQCTLB.bit.ZRO = AQ_SET;
    EPwm1Regs.AQCTLB.bit.PRD = AQ_CLEAR;

#endif

    // dead time

    EPwm1Regs.DBCTL.bit.OUT_MODE = DB_FULL_ENABLE;
    EPwm1Regs.DBCTL.bit.IN_MODE  = DBA_RED_DBB_FED;
    EPwm1Regs.DBCTL.bit.POLSEL   = DB_ACTV_HIC;

    EPwm1Regs.DBFED = 60;
    EPwm1Regs.DBRED = 60;


    // Interrupt where we will change the Compare Values
    EPwm1Regs.ETSEL.bit.INTSEL = ET_CTR_ZERO;     // Select INT on Zero event
    EPwm1Regs.ETSEL.bit.INTEN = 1;                // Enable INT
    EPwm1Regs.ETPS.bit.INTPRD = ET_3RD;           // Generate INT on 3rd event

    // Information this example uses to keep track
    // of the direction the CMPA/CMPB values are
    // moving, the min and max allowed values and
    // a pointer to the correct ePWM registers
    epwm1_info.EPwm_CMPA_Direction = EPWM_CMP_UP;  // Start by increasing CMPA &
    epwm1_info.EPwm_CMPB_Direction = EPWM_CMP_DOWN; // decreasing CMPB
    epwm1_info.EPwmTimerIntCount = 0;              // Zero the interrupt counter
    epwm1_info.EPwmRegHandle = &EPwm1Regs;         // Set the pointer to the
                                                   // ePWM module
    epwm1_info.EPwmMaxCMPA = EPWM1_MAX_CMPA;       // Setup min/max CMPA/CMPB
                                                   // values
    epwm1_info.EPwmMinCMPA = EPWM1_MIN_CMPA;
    epwm1_info.EPwmMaxCMPB = EPWM1_MAX_CMPB;
    epwm1_info.EPwmMinCMPB = EPWM1_MIN_CMPB;

}

void InitEPwm2Example()
{

    // Setup TBCLK
    EPwm2Regs.TBPRD = 242;         // Set timer period 801 TBCLKs
    EPwm2Regs.TBPHS.bit.TBPHS = 0x0000;          // Phase is 0
    EPwm2Regs.TBCTR = 0x0000;                    // Clear counter

    // Set Compare values
    EPwm2Regs.CMPA.bit.CMPA = 242;    // Set compare A value
    EPwm2Regs.CMPB.bit.CMPB = 0;    // Set Compare B value

    // Setup counter mode
    EPwm2Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN; // Count up and down
    EPwm2Regs.TBCTL.bit.PHSEN = 0;       // Disable phase loading
    EPwm2Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1;      // Clock ratio to SYSCLKOUT
    EPwm2Regs.TBCTL.bit.CLKDIV = TB_DIV1;

    // Setup shadowing
    EPwm2Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;
    EPwm2Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;
    EPwm2Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO; // Load on Zero
    EPwm2Regs.CMPCTL.bit.LOADBMODE = CC_CTR_ZERO;


#if 0
    // Set actions
    EPwm2Regs.AQCTLA.bit.CAU = AQ_SET;           // Set PWM2A on event A, up
                                                 // count
    EPwm2Regs.AQCTLA.bit.CBD = AQ_CLEAR;         // Clear PWM2A on event B, down
                                                 // count

    EPwm2Regs.AQCTLB.bit.ZRO = AQ_CLEAR;         // Clear PWM2B on zero
    EPwm2Regs.AQCTLB.bit.PRD = AQ_SET  ;         // Set PWM2B on period
#else

    EPwm2Regs.AQCTLA.bit.ZRO = AQ_SET;
    EPwm2Regs.AQCTLA2.bit.T1D = AQ_SET;   // peak current mode
    EPwm2Regs.AQCTLA2.bit.T1U = AQ_CLEAR;
    EPwm2Regs.AQCTLA.bit.CAU = AQ_CLEAR;  // for duty limit
    EPwm2Regs.AQCTLA.bit.PRD = AQ_CLEAR;

    EPwm2Regs.AQCTLB.bit.ZRO = AQ_SET;
    EPwm2Regs.AQCTLB.bit.CBD = AQ_SET;    // for duty limit
    EPwm2Regs.AQCTLB.bit.PRD = AQ_CLEAR;
    EPwm2Regs.AQCTLB2.bit.T1U = AQ_CLEAR;
    EPwm2Regs.AQCTLB2.bit.T1D = AQ_SET;

#endif

    // dead time

    EPwm2Regs.DBCTL.bit.OUT_MODE = DB_FULL_ENABLE;
    EPwm2Regs.DBCTL.bit.IN_MODE  = DBA_RED_DBB_FED;
    EPwm2Regs.DBCTL.bit.POLSEL   = DB_ACTV_HIC;
    EPwm2Regs.DBCTL.bit.OUTSWAP  = 0x03;            // A: B-path  B: A-path


    EPwm2Regs.DBFED = 60;
    EPwm2Regs.DBRED = 60;




    EALLOW;

    EPwm2Regs.AQTSRCSEL.bit.T1SEL          = 3;// DCBEVT2 to be T1

    //Configure EPWM2B to output low on  TRIP
    EPwm2Regs.TZCTL2.bit.TZAD               = TZ_FORCE_LO;
    EPwm2Regs.TZCTL2.bit.TZBU               = TZ_FORCE_LO;
    //Configure DCB to be TRIP4
    EPwm2Regs.TZDCSEL.bit.DCBEVT2          = TZ_DCBH_HI;
    EPwm2Regs.DCTRIPSEL.bit.DCBHCOMPSEL    = 0xF; //DCBHTRIPSEL
    EPwm2Regs.DCBHTRIPSEL.bit.TRIPINPUT4   = 1;
    //Configure DCB as CBC
    //EPwm2Regs.TZSEL.bit.DCBEVT2           = 1;
    //Configure DCB path to be unfiltered & async
    EPwm2Regs.DCBCTL.bit.EVT2SRCSEL        = DC_EVT2;
    EPwm2Regs.DCBCTL.bit.EVT2FRCSYNCSEL    = DC_EVT_ASYNC;

    EPwm2Regs.TZCLR.bit.CBC                = 1;

    EPwm2Regs.DCFCTL.bit.PULSESEL = 0x10;
    EPwm2Regs.DCFCTL.bit.SRCSEL   = 0x11;
    EPwm2Regs.DCFWINDOW  = 0x60;
    EPwm2Regs.DCFCTL.bit.BLANKE   = 1;
    EDIS;






    // Interrupt where we will change the Compare Values
    EPwm2Regs.ETSEL.bit.INTSEL = ET_CTR_ZERO;    // Select INT on Zero event
    EPwm2Regs.ETSEL.bit.INTEN = 1;               // Enable INT
    EPwm2Regs.ETPS.bit.INTPRD = ET_3RD;          // Generate INT on 3rd event

    // Information this example uses to keep track
    // of the direction the CMPA/CMPB values are
    // moving, the min and max allowed values and
    // a pointer to the correct ePWM registers
    epwm2_info.EPwm_CMPA_Direction = EPWM_CMP_UP;  // Start by increasing CMPA &
    epwm2_info.EPwm_CMPB_Direction = EPWM_CMP_UP;  // increasing CMPB
    epwm2_info.EPwmTimerIntCount = 0;              // Zero the interrupt counter
    epwm2_info.EPwmRegHandle = &EPwm2Regs;         // Set the pointer to the
                                                   // ePWM module
    epwm2_info.EPwmMaxCMPA = EPWM2_MAX_CMPA;       // Setup min/max CMPA/CMPB
                                                   // values
    epwm2_info.EPwmMinCMPA = EPWM2_MIN_CMPA;
    epwm2_info.EPwmMaxCMPB = EPWM2_MAX_CMPB;
    epwm2_info.EPwmMinCMPB = EPWM2_MIN_CMPB;

}

void InitEPwm3Example(void)
{

    // Setup TBCLK
    EPwm3Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN; // Count up/down and down
    EPwm3Regs.TBPRD = 242;           // Set timer period
    EPwm3Regs.TBCTL.bit.PHSEN = 0;        // Disable phase loading
    EPwm3Regs.TBPHS.bit.TBPHS = 0x0000;            // Phase is 0
    EPwm3Regs.TBCTR = 0x0000;                      // Clear counter
    EPwm3Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1;       // Clock ratio to SYSCLKOUT
    EPwm3Regs.TBCTL.bit.CLKDIV = TB_DIV1;

    // Setup shadow register load on ZERO
    EPwm3Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;
    EPwm3Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;
    EPwm3Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO;
    EPwm3Regs.CMPCTL.bit.LOADBMODE = CC_CTR_ZERO;

    // Set Compare values
    EPwm3Regs.CMPA.bit.CMPA = 0;   // Set compare A value
    EPwm3Regs.CMPB.bit.CMPB = 0;   // Set Compare B value
#if 0
    // Set Actions
    EPwm3Regs.AQCTLA.bit.PRD = AQ_SET;           // Set PWM3A on period
    EPwm3Regs.AQCTLA.bit.CBD = AQ_CLEAR;         // Clear PWM3A on event B, down
                                                 // count

    EPwm3Regs.AQCTLB.bit.PRD = AQ_CLEAR;         // Clear PWM3A on period
    EPwm3Regs.AQCTLB.bit.CAU = AQ_SET;           // Set PWM3A on event A, up
#else

    EPwm3Regs.AQCTLA.bit.ZRO =  AQ_SET;
    EPwm3Regs.AQCTLA2.bit.T1D = AQ_SET;         // peak current mode  different with EPWM2
    EPwm3Regs.AQCTLA.bit.PRD = AQ_CLEAR;

    EPwm3Regs.AQCTLB.bit.ZRO = AQ_SET;
    EPwm3Regs.AQCTLB.bit.PRD = AQ_CLEAR;
    EPwm3Regs.AQCTLB2.bit.T1U = AQ_CLEAR;       // for peak current mode different with EPWM2


#endif

    // dead time

    EPwm3Regs.DBCTL.bit.OUT_MODE = DB_FULL_ENABLE;
    EPwm3Regs.DBCTL.bit.IN_MODE  = DBA_RED_DBB_FED;
    EPwm3Regs.DBCTL.bit.POLSEL   = DB_ACTV_HIC;
    EPwm3Regs.DBCTL.bit.OUTSWAP  = 0x03;            // A: B-path  B: A-path


    EPwm3Regs.DBFED = 60;
    EPwm3Regs.DBRED = 60;                                                 // count


    EALLOW;

    EPwm3Regs.AQTSRCSEL.bit.T1SEL          = 3;// DCBEVT2 to be T1

#if 1
    //Configure EPWM2B to output low on  TRIP
    EPwm3Regs.TZCTL.bit.TZB                = TZ_FORCE_LO;
    EPwm3Regs.TZCTL.bit.TZA                = TZ_FORCE_LO;
    //Configure DCB to be TRIP4
    EPwm3Regs.TZDCSEL.bit.DCBEVT2          = TZ_DCBH_HI;
    EPwm3Regs.DCTRIPSEL.bit.DCBHCOMPSEL    = 0xF; //DCBHTRIPSEL
    EPwm3Regs.DCBHTRIPSEL.bit.TRIPINPUT4   = 1;
    //Configure DCB as CBC
    //EPwm3Regs.TZSEL.bit.DCBEVT2           = 1;
    //Configure DCB path to be unfiltered & async
    EPwm3Regs.DCBCTL.bit.EVT2SRCSEL        = DC_EVT1;
    EPwm3Regs.DCBCTL.bit.EVT2FRCSYNCSEL    = DC_EVT_ASYNC;

    EPwm3Regs.TZCLR.bit.CBC                = 1;
#endif

    EDIS;




    // Interrupt where we will change the Compare Values
    EPwm3Regs.ETSEL.bit.INTSEL = ET_CTR_ZERO;    // Select INT on Zero event
    EPwm3Regs.ETSEL.bit.INTEN = 1;               // Enable INT
    EPwm3Regs.ETPS.bit.INTPRD = ET_3RD;          // Generate INT on 3rd event

    // Information this example uses to keep track
    // of the direction the CMPA/CMPB values are
    // moving, the min and max allowed values and
    // a pointer to the correct ePWM registers
    epwm3_info.EPwm_CMPA_Direction = EPWM_CMP_UP;   // Start by increasing CMPA &
    epwm3_info.EPwm_CMPB_Direction = EPWM_CMP_DOWN; // decreasing CMPB
    epwm3_info.EPwmTimerIntCount = 0;               // Zero the interrupt counter
    epwm3_info.EPwmRegHandle = &EPwm3Regs;          // Set the pointer to the
                                                    // ePWM module
    epwm3_info.EPwmMaxCMPA = EPWM3_MAX_CMPA;        // Setup min/max CMPA/CMPB
                                                    // values
    epwm3_info.EPwmMinCMPA = EPWM3_MIN_CMPA;
    epwm3_info.EPwmMaxCMPB = EPWM3_MAX_CMPB;
    epwm3_info.EPwmMinCMPB = EPWM3_MIN_CMPB;

}

void InitEPwm4Example(void)
{

    // Setup TBCLK
    EPwm4Regs.TBCTL.bit.CTRMODE = TB_COUNT_UP; // Count up/down and down
    EPwm4Regs.TBPRD = 242-1;           // Set timer period
    EPwm4Regs.TBCTL.bit.PHSEN = 0;        // Disable phase loading
    EPwm4Regs.TBPHS.bit.TBPHS = 0x0000;            // Phase is 0
    EPwm4Regs.TBCTR = 0x0000;                      // Clear counter
    EPwm4Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1;       // Clock ratio to SYSCLKOUT
    EPwm4Regs.TBCTL.bit.CLKDIV = TB_DIV1;

    // Setup shadow register load on ZERO
    EPwm4Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;
    EPwm4Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;
    EPwm4Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO;
    EPwm4Regs.CMPCTL.bit.LOADBMODE = CC_CTR_ZERO;

    EPwm4Regs.HRPCTL.bit.PWMSYNCSELX = 4;  // CMPC  up
    EPwm4Regs.CMPC = 242 -10;

    EPwm4Regs.AQCTLA.bit.ZRO = AQ_CLEAR;
    EPwm4Regs.AQCTLA.bit.PRD = AQ_CLEAR;
    EPwm4Regs.AQCTLA.bit.CAU = AQ_CLEAR ;
    EPwm4Regs.AQCTLA2.bit.T1U = AQ_SET;
    EPwm4Regs.CMPA.bit.CMPA  = 210;

    EALLOW;
    EPwm4Regs.AQTSRCSEL.bit.T1SEL          = 3;// DCBEVT2 to be T1
    //Configure EPWM2B to output low on  TRIP
    //EPwm4Regs.TZCTL.bit.TZA               = TZ_FORCE_LO;
    //EPwm4Regs.TZCTL.bit.TZB               = TZ_FORCE_LO;
    //Configure DCB to be TRIP4
    EPwm4Regs.TZDCSEL.bit.DCBEVT2          = TZ_DCBH_HI;
    EPwm4Regs.DCTRIPSEL.bit.DCBHCOMPSEL    = 0xF; //DCBHTRIPSEL
    EPwm4Regs.DCBHTRIPSEL.bit.TRIPINPUT4   = 1;
    //Configure DCB as CBC
    EPwm4Regs.TZSEL.bit.DCBEVT2           = 0;
    //Configure DCB path to be unfiltered & async
    EPwm4Regs.DCBCTL.bit.EVT2SRCSEL        = DC_EVT2;
    EPwm4Regs.DCBCTL.bit.EVT2FRCSYNCSEL    = DC_EVT_ASYNC;

    EPwm4Regs.TZCLR.bit.CBC                = 1;

    EPwm4Regs.DCFCTL.bit.PULSESEL = 0x10;
    EPwm4Regs.DCFCTL.bit.SRCSEL   = 0x11;
    EPwm4Regs.DCFWINDOW  = 0x60;
    EPwm4Regs.DCFCTL.bit.BLANKE   = 1;
    EDIS;









}


void update_compare(EPWM_INFO *epwm_info)
{

    // Every 10'th interrupt, change the CMPA/CMPB values
    if(epwm_info->EPwmTimerIntCount == 10)
    {
        epwm_info->EPwmTimerIntCount = 0;

        // If we were increasing CMPA, check to see if
        // we reached the max value.  If not, increase CMPA
        // else, change directions and decrease CMPA
        if(epwm_info->EPwm_CMPA_Direction == EPWM_CMP_UP)
        {
            if(epwm_info->EPwmRegHandle->CMPA.bit.CMPA <
               epwm_info->EPwmMaxCMPA)
            {
                epwm_info->EPwmRegHandle->CMPA.bit.CMPA++;
            }
            else
            {
                epwm_info->EPwm_CMPA_Direction = EPWM_CMP_DOWN;
                epwm_info->EPwmRegHandle->CMPA.bit.CMPA--;
            }
        }

        // If we were decreasing CMPA, check to see if
        // we reached the min value.  If not, decrease CMPA
        // else, change directions and increase CMPA
        else
        {
            if(epwm_info->EPwmRegHandle->CMPA.bit.CMPA ==
               epwm_info->EPwmMinCMPA)
            {
                epwm_info->EPwm_CMPA_Direction = EPWM_CMP_UP;
                epwm_info->EPwmRegHandle->CMPA.bit.CMPA++;
            }
            else
            {
                epwm_info->EPwmRegHandle->CMPA.bit.CMPA--;
            }
        }

        // If we were increasing CMPB, check to see if
        // we reached the max value.  If not, increase CMPB
        // else, change directions and decrease CMPB
        if(epwm_info->EPwm_CMPB_Direction == EPWM_CMP_UP)
        {
            if(epwm_info->EPwmRegHandle->CMPB.bit.CMPB < epwm_info->EPwmMaxCMPB)
            {
                epwm_info->EPwmRegHandle->CMPB.bit.CMPB++;
            }
            else
            {
                epwm_info->EPwm_CMPB_Direction = EPWM_CMP_DOWN;
                epwm_info->EPwmRegHandle->CMPB.bit.CMPB--;
            }
        }

        // If we were decreasing CMPB, check to see if
        // we reached the min value.  If not, decrease CMPB
        // else, change directions and increase CMPB

        else
        {
            if(epwm_info->EPwmRegHandle->CMPB.bit.CMPB == epwm_info->EPwmMinCMPB)
            {
                epwm_info->EPwm_CMPB_Direction = EPWM_CMP_UP;
                epwm_info->EPwmRegHandle->CMPB.bit.CMPB++;
            }
            else
            {
            	epwm_info->EPwmRegHandle->CMPB.bit.CMPB--;
            }
        }
    }
    else
    {
    	epwm_info->EPwmTimerIntCount++;
    }

    return;
}




void InitCMPSS(void)
{
    EALLOW;
    //Enable CMPSS
    Cmpss1Regs.COMPCTL.bit.COMPDACE            = 1;
    //NEG signal comes from DAC
    Cmpss1Regs.COMPCTL.bit.COMPHSOURCE         = NEGIN_DAC;
    //Use VDDA as the reference for DAC
    Cmpss1Regs.COMPDACCTL.bit.SELREF           = REFERENCE_VDDA;
    //Set DAC to midpoint for arbitrary reference
    Cmpss1Regs.DACHVALS.bit.DACVAL             = 2048;
    //
    Cmpss1Regs.COMPDACCTL.bit.RAMPSOURCE = 3;   // PWMSYN4
    Cmpss1Regs.COMPDACCTL.bit.DACSOURCE  = 1;   // Ramp

    //
    Cmpss1Regs.RAMPMAXREFS = 16000;
    Cmpss1Regs.RAMPDECVALS = 10;

    // Configure CTRIPOUT path
    //Asynch output feeds CTRIPH and CTRIPOUTH
    Cmpss1Regs.COMPCTL.bit.CTRIPHSEL           = CTRIP_ASYNCH;
    Cmpss1Regs.COMPCTL.bit.CTRIPOUTHSEL        = CTRIP_ASYNCH;

    // Configure CTRIPOUTH output pin
    //Configure OUTPUTXBAR3 to be CTRIPOUT1H
    OutputXbarRegs.OUTPUT3MUX0TO15CFG.bit.MUX0 = 0;
    //Enable OUTPUTXBAR3 Mux for Output
    OutputXbarRegs.OUTPUT3MUXENABLE.bit.MUX0   = 1;
    EDIS;


    // setup ramp generation
    //



}
