/*
 * Copyright (c) 2020 Texas Instruments Incorporated - http://www.ti.com
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "board.h"

//*****************************************************************************
//
// Board Configurations
// Initializes the rest of the modules. 
// Call this function in your application if you wish to do all module 
// initialization.
// If you wish to not use some of the initializations, instead of the 
// Board_init use the individual Module_inits
//
//*****************************************************************************
void Board_init()
{
	EALLOW;

	PinMux_init();
	SYNC_init();
	ASYSCTL_init();
	ADC_init();
	CMPSS_init();
	CPUTIMER_init();
	ECAP_init();
	EPWM_init();
	GPIO_init();
	I2C_init();
	SCI_init();
	INTERRUPT_init();

	EDIS;
}

//*****************************************************************************
//
// PINMUX Configurations
//
//*****************************************************************************
void PinMux_init()
{
	//
	// PinMux for modules assigned to CPU1
	//
	
	//
	// ANALOG -> AdcPinMux Pinmux
	//
	// Analog PinMux for A0, B15, C15, DACA_OUT
	GPIO_setPinConfig(GPIO_231_GPIO231);
	// AIO -> Analog mode selected
	GPIO_setAnalogMode(231, GPIO_ANALOG_ENABLED);
	// Analog PinMux for A2, B6, C9, PGA1_INP, GPIO224
	GPIO_setPinConfig(GPIO_224_GPIO224);
	// AGPIO -> Analog mode selected
	GPIO_setAnalogMode(224, GPIO_ANALOG_ENABLED);
	// Analog PinMux for A8, B0, C11, PGA3_OUT
	GPIO_setPinConfig(GPIO_241_GPIO241);
	// AIO -> Analog mode selected
	GPIO_setAnalogMode(241, GPIO_ANALOG_ENABLED);
	// Analog PinMux for A10, B1, C10, GPIO230
	GPIO_setPinConfig(GPIO_230_GPIO230);
	// AGPIO -> Analog mode selected
	GPIO_setAnalogMode(230, GPIO_ANALOG_ENABLED);
	// Analog PinMux for A11, B10, C0, PGA2_OUT
	GPIO_setPinConfig(GPIO_237_GPIO237);
	// AIO -> Analog mode selected
	GPIO_setAnalogMode(237, GPIO_ANALOG_ENABLED);
	// Analog PinMux for A12, C1, E11, PGA3_INP
	GPIO_setPinConfig(GPIO_238_GPIO238);
	// AIO -> Analog mode selected
	GPIO_setAnalogMode(238, GPIO_ANALOG_ENABLED);
	// Analog PinMux for B2, C6, E12, GPIO226
	GPIO_setPinConfig(GPIO_226_GPIO226);
	// AGPIO -> Analog mode selected
	GPIO_setAnalogMode(226, GPIO_ANALOG_ENABLED);
	// Analog PinMux for A1, B7, D11, CMP1_DACL
	GPIO_setPinConfig(GPIO_232_GPIO232);
	// AIO -> Analog mode selected
	GPIO_setAnalogMode(232, GPIO_ANALOG_ENABLED);
	//
	// EPWM1 -> pwm_a Pinmux
	//
	GPIO_setPinConfig(pwm_a_EPWMA_PIN_CONFIG);
	GPIO_setPadConfig(pwm_a_EPWMA_GPIO, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(pwm_a_EPWMA_GPIO, GPIO_QUAL_SYNC);

	GPIO_setPinConfig(pwm_a_EPWMB_PIN_CONFIG);
	GPIO_setPadConfig(pwm_a_EPWMB_GPIO, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(pwm_a_EPWMB_GPIO, GPIO_QUAL_SYNC);

	//
	// EPWM2 -> pwm_b Pinmux
	//
	GPIO_setPinConfig(pwm_b_EPWMA_PIN_CONFIG);
	GPIO_setPadConfig(pwm_b_EPWMA_GPIO, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(pwm_b_EPWMA_GPIO, GPIO_QUAL_SYNC);

	GPIO_setPinConfig(pwm_b_EPWMB_PIN_CONFIG);
	GPIO_setPadConfig(pwm_b_EPWMB_GPIO, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(pwm_b_EPWMB_GPIO, GPIO_QUAL_SYNC);

	//
	// EPWM12 -> pwm_c Pinmux
	//
	GPIO_setPinConfig(pwm_c_EPWMA_PIN_CONFIG);
	GPIO_setPadConfig(pwm_c_EPWMA_GPIO, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(pwm_c_EPWMA_GPIO, GPIO_QUAL_SYNC);

	GPIO_setPinConfig(pwm_c_EPWMB_PIN_CONFIG);
	GPIO_setPadConfig(pwm_c_EPWMB_GPIO, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(pwm_c_EPWMB_GPIO, GPIO_QUAL_SYNC);

	//
	// EPWM8 -> pwm_break Pinmux
	//
	GPIO_setPinConfig(pwm_break_EPWMA_PIN_CONFIG);
	// AGPIO -> GPIO mode selected
	GPIO_setAnalogMode(24, GPIO_ANALOG_DISABLED);
	GPIO_setPadConfig(pwm_break_EPWMA_GPIO, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(pwm_break_EPWMA_GPIO, GPIO_QUAL_SYNC);

	GPIO_setPinConfig(pwm_break_EPWMB_PIN_CONFIG);
	GPIO_setPadConfig(pwm_break_EPWMB_GPIO, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(pwm_break_EPWMB_GPIO, GPIO_QUAL_SYNC);

	// GPIO33 -> rs485_ena Pinmux
	GPIO_setPinConfig(GPIO_33_GPIO33);
	// AGPIO -> GPIO mode selected
	GPIO_setAnalogMode(33, GPIO_ANALOG_DISABLED);
	// GPIO15 -> fan_ctrl Pinmux
	GPIO_setPinConfig(GPIO_15_GPIO15);
	// GPIO42 -> relay_ctrl Pinmux
	GPIO_setPinConfig(GPIO_42_GPIO42);
	// GPIO46 -> v2_ctrl Pinmux
	GPIO_setPinConfig(GPIO_46_GPIO46);
	// GPIO30 -> di1 Pinmux
	GPIO_setPinConfig(GPIO_30_GPIO30);
	// GPIO62 -> di2 Pinmux
	GPIO_setPinConfig(GPIO_62_GPIO62);
	// GPIO43 -> di3 Pinmux
	GPIO_setPinConfig(GPIO_43_GPIO43);
	// GPIO25 -> bus_ov Pinmux
	GPIO_setPinConfig(GPIO_25_GPIO25);
	// GPIO9 -> bus_uv Pinmux
	GPIO_setPinConfig(GPIO_9_GPIO9);
	//
	// I2CA -> i2c_a Pinmux
	//
	GPIO_setPinConfig(i2c_a_I2CSDA_PIN_CONFIG);
	GPIO_setPadConfig(i2c_a_I2CSDA_GPIO, GPIO_PIN_TYPE_STD | GPIO_PIN_TYPE_PULLUP);
	GPIO_setQualificationMode(i2c_a_I2CSDA_GPIO, GPIO_QUAL_ASYNC);

	GPIO_setPinConfig(i2c_a_I2CSCL_PIN_CONFIG);
	GPIO_setPadConfig(i2c_a_I2CSCL_GPIO, GPIO_PIN_TYPE_STD | GPIO_PIN_TYPE_PULLUP);
	GPIO_setQualificationMode(i2c_a_I2CSCL_GPIO, GPIO_QUAL_ASYNC);

	//
	// SCIA -> sci_a Pinmux
	//
	GPIO_setPinConfig(sci_a_SCIRX_PIN_CONFIG);
	// AGPIO -> GPIO mode selected
	GPIO_setAnalogMode(17, GPIO_ANALOG_DISABLED);
	GPIO_setPadConfig(sci_a_SCIRX_GPIO, GPIO_PIN_TYPE_STD | GPIO_PIN_TYPE_PULLUP);
	GPIO_setQualificationMode(sci_a_SCIRX_GPIO, GPIO_QUAL_ASYNC);

	GPIO_setPinConfig(sci_a_SCITX_PIN_CONFIG);
	// AGPIO -> GPIO mode selected
	GPIO_setAnalogMode(16, GPIO_ANALOG_DISABLED);
	GPIO_setPadConfig(sci_a_SCITX_GPIO, GPIO_PIN_TYPE_STD | GPIO_PIN_TYPE_PULLUP);
	GPIO_setQualificationMode(sci_a_SCITX_GPIO, GPIO_QUAL_ASYNC);

	//
	// SCIB -> sci_b Pinmux
	//
	GPIO_setPinConfig(sci_b_SCIRX_PIN_CONFIG);
	// AGPIO -> GPIO mode selected
	GPIO_setAnalogMode(13, GPIO_ANALOG_DISABLED);
	GPIO_setPadConfig(sci_b_SCIRX_GPIO, GPIO_PIN_TYPE_STD | GPIO_PIN_TYPE_PULLUP);
	GPIO_setQualificationMode(sci_b_SCIRX_GPIO, GPIO_QUAL_ASYNC);

	GPIO_setPinConfig(sci_b_SCITX_PIN_CONFIG);
	// AGPIO -> GPIO mode selected
	GPIO_setAnalogMode(12, GPIO_ANALOG_DISABLED);
	GPIO_setPadConfig(sci_b_SCITX_GPIO, GPIO_PIN_TYPE_STD | GPIO_PIN_TYPE_PULLUP);
	GPIO_setQualificationMode(sci_b_SCITX_GPIO, GPIO_QUAL_ASYNC);


}

//*****************************************************************************
//
// ADC Configurations
//
//*****************************************************************************
void ADC_init(){
	adc_a_init();
	adc_b_init();
	adc_c_init();
}

void adc_a_init(){
	//
	// ADC Initialization: Write ADC configurations and power up the ADC
	//
	// Set the analog voltage reference selection and ADC module's offset trims.
	// This function sets the analog voltage reference to internal (with the reference voltage of 1.65V or 2.5V) or external for ADC
	// which is same as ASysCtl APIs.
	//
	ADC_setVREF(adc_a_BASE, ADC_REFERENCE_INTERNAL, ADC_REFERENCE_2_5V);
	//
	// Configures the analog-to-digital converter module prescaler.
	//
	ADC_setPrescaler(adc_a_BASE, ADC_CLK_DIV_2_0);
	//
	// Sets the timing of the end-of-conversion pulse
	//
	ADC_setInterruptPulseMode(adc_a_BASE, ADC_PULSE_END_OF_ACQ_WIN);
	//
	// Sets the timing of early interrupt generation.
	//
	ADC_setInterruptCycleOffset(adc_a_BASE, 0U);
	//
	// Powers up the analog-to-digital converter core.
	//
	ADC_enableConverter(adc_a_BASE);
	//
	// Delay for 1ms to allow ADC time to power up
	//
	DEVICE_DELAY_US(5000);
	//
	// Enable alternate timings for DMA trigger
	//
	ADC_enableAltDMATiming(adc_a_BASE);
	//
	// SOC Configuration: Setup ADC EPWM channel and trigger settings
	//
	// Disables SOC burst mode.
	//
	ADC_disableBurstMode(adc_a_BASE);
	//
	// Sets the priority mode of the SOCs.
	//
	ADC_setSOCPriority(adc_a_BASE, ADC_PRI_ALL_ROUND_ROBIN);
	//
	// Start of Conversion 0 Configuration
	//
	//
	// Disables Sample Capacitor to reset after each conversion.
	//
	ADC_disableSampleCAPReset(adc_a_BASE, ADC_SOC_NUMBER0);
	//
	// Configures a start-of-conversion (SOC) in the ADC and its interrupt SOC trigger.
	// 	  	SOC number		: 0
	//	  	Trigger			: ADC_TRIGGER_SW_ONLY
	//	  	Channel			: ADC_CH_ADCIN0
	//	 	Sample Window	: 12 SYSCLK cycles
	//		Interrupt Trigger: ADC_INT_SOC_TRIGGER_NONE
	//
	ADC_setupSOC(adc_a_BASE, ADC_SOC_NUMBER0, ADC_TRIGGER_SW_ONLY, ADC_CH_ADCIN0, 12U);
	ADC_setInterruptSOCTrigger(adc_a_BASE, ADC_SOC_NUMBER0, ADC_INT_SOC_TRIGGER_NONE);
	//
	// Start of Conversion 1 Configuration
	//
	//
	// Disables Sample Capacitor to reset after each conversion.
	//
	ADC_disableSampleCAPReset(adc_a_BASE, ADC_SOC_NUMBER1);
	//
	// Configures a start-of-conversion (SOC) in the ADC and its interrupt SOC trigger.
	// 	  	SOC number		: 1
	//	  	Trigger			: ADC_TRIGGER_SW_ONLY
	//	  	Channel			: ADC_CH_ADCIN0
	//	 	Sample Window	: 12 SYSCLK cycles
	//		Interrupt Trigger: ADC_INT_SOC_TRIGGER_NONE
	//
	ADC_setupSOC(adc_a_BASE, ADC_SOC_NUMBER1, ADC_TRIGGER_SW_ONLY, ADC_CH_ADCIN0, 12U);
	ADC_setInterruptSOCTrigger(adc_a_BASE, ADC_SOC_NUMBER1, ADC_INT_SOC_TRIGGER_NONE);
	//
	// Start of Conversion 2 Configuration
	//
	//
	// Disables Sample Capacitor to reset after each conversion.
	//
	ADC_disableSampleCAPReset(adc_a_BASE, ADC_SOC_NUMBER2);
	//
	// Configures a start-of-conversion (SOC) in the ADC and its interrupt SOC trigger.
	// 	  	SOC number		: 2
	//	  	Trigger			: ADC_TRIGGER_SW_ONLY
	//	  	Channel			: ADC_CH_ADCIN0
	//	 	Sample Window	: 12 SYSCLK cycles
	//		Interrupt Trigger: ADC_INT_SOC_TRIGGER_NONE
	//
	ADC_setupSOC(adc_a_BASE, ADC_SOC_NUMBER2, ADC_TRIGGER_SW_ONLY, ADC_CH_ADCIN0, 12U);
	ADC_setInterruptSOCTrigger(adc_a_BASE, ADC_SOC_NUMBER2, ADC_INT_SOC_TRIGGER_NONE);
}

void adc_b_init(){
	//
	// ADC Initialization: Write ADC configurations and power up the ADC
	//
	// Set the analog voltage reference selection and ADC module's offset trims.
	// This function sets the analog voltage reference to internal (with the reference voltage of 1.65V or 2.5V) or external for ADC
	// which is same as ASysCtl APIs.
	//
	ADC_setVREF(adc_b_BASE, ADC_REFERENCE_INTERNAL, ADC_REFERENCE_2_5V);
	//
	// Configures the analog-to-digital converter module prescaler.
	//
	ADC_setPrescaler(adc_b_BASE, ADC_CLK_DIV_2_0);
	//
	// Sets the timing of the end-of-conversion pulse
	//
	ADC_setInterruptPulseMode(adc_b_BASE, ADC_PULSE_END_OF_ACQ_WIN);
	//
	// Sets the timing of early interrupt generation.
	//
	ADC_setInterruptCycleOffset(adc_b_BASE, 0U);
	//
	// Powers up the analog-to-digital converter core.
	//
	ADC_enableConverter(adc_b_BASE);
	//
	// Delay for 1ms to allow ADC time to power up
	//
	DEVICE_DELAY_US(5000);
	//
	// Enable alternate timings for DMA trigger
	//
	ADC_enableAltDMATiming(adc_b_BASE);
	//
	// SOC Configuration: Setup ADC EPWM channel and trigger settings
	//
	// Disables SOC burst mode.
	//
	ADC_disableBurstMode(adc_b_BASE);
	//
	// Sets the priority mode of the SOCs.
	//
	ADC_setSOCPriority(adc_b_BASE, ADC_PRI_ALL_ROUND_ROBIN);
	//
	// Start of Conversion 0 Configuration
	//
	//
	// Disables Sample Capacitor to reset after each conversion.
	//
	ADC_disableSampleCAPReset(adc_b_BASE, ADC_SOC_NUMBER0);
	//
	// Configures a start-of-conversion (SOC) in the ADC and its interrupt SOC trigger.
	// 	  	SOC number		: 0
	//	  	Trigger			: ADC_TRIGGER_SW_ONLY
	//	  	Channel			: ADC_CH_ADCIN0
	//	 	Sample Window	: 12 SYSCLK cycles
	//		Interrupt Trigger: ADC_INT_SOC_TRIGGER_NONE
	//
	ADC_setupSOC(adc_b_BASE, ADC_SOC_NUMBER0, ADC_TRIGGER_SW_ONLY, ADC_CH_ADCIN0, 12U);
	ADC_setInterruptSOCTrigger(adc_b_BASE, ADC_SOC_NUMBER0, ADC_INT_SOC_TRIGGER_NONE);
	//
	// Start of Conversion 1 Configuration
	//
	//
	// Disables Sample Capacitor to reset after each conversion.
	//
	ADC_disableSampleCAPReset(adc_b_BASE, ADC_SOC_NUMBER1);
	//
	// Configures a start-of-conversion (SOC) in the ADC and its interrupt SOC trigger.
	// 	  	SOC number		: 1
	//	  	Trigger			: ADC_TRIGGER_SW_ONLY
	//	  	Channel			: ADC_CH_ADCIN0
	//	 	Sample Window	: 12 SYSCLK cycles
	//		Interrupt Trigger: ADC_INT_SOC_TRIGGER_NONE
	//
	ADC_setupSOC(adc_b_BASE, ADC_SOC_NUMBER1, ADC_TRIGGER_SW_ONLY, ADC_CH_ADCIN0, 12U);
	ADC_setInterruptSOCTrigger(adc_b_BASE, ADC_SOC_NUMBER1, ADC_INT_SOC_TRIGGER_NONE);
	//
	// Start of Conversion 2 Configuration
	//
	//
	// Disables Sample Capacitor to reset after each conversion.
	//
	ADC_disableSampleCAPReset(adc_b_BASE, ADC_SOC_NUMBER2);
	//
	// Configures a start-of-conversion (SOC) in the ADC and its interrupt SOC trigger.
	// 	  	SOC number		: 2
	//	  	Trigger			: ADC_TRIGGER_SW_ONLY
	//	  	Channel			: ADC_CH_ADCIN0
	//	 	Sample Window	: 12 SYSCLK cycles
	//		Interrupt Trigger: ADC_INT_SOC_TRIGGER_NONE
	//
	ADC_setupSOC(adc_b_BASE, ADC_SOC_NUMBER2, ADC_TRIGGER_SW_ONLY, ADC_CH_ADCIN0, 12U);
	ADC_setInterruptSOCTrigger(adc_b_BASE, ADC_SOC_NUMBER2, ADC_INT_SOC_TRIGGER_NONE);
}

void adc_c_init(){
	//
	// ADC Initialization: Write ADC configurations and power up the ADC
	//
	// Set the analog voltage reference selection and ADC module's offset trims.
	// This function sets the analog voltage reference to internal (with the reference voltage of 1.65V or 2.5V) or external for ADC
	// which is same as ASysCtl APIs.
	//
	ADC_setVREF(adc_c_BASE, ADC_REFERENCE_INTERNAL, ADC_REFERENCE_2_5V);
	//
	// Configures the analog-to-digital converter module prescaler.
	//
	ADC_setPrescaler(adc_c_BASE, ADC_CLK_DIV_2_0);
	//
	// Sets the timing of the end-of-conversion pulse
	//
	ADC_setInterruptPulseMode(adc_c_BASE, ADC_PULSE_END_OF_ACQ_WIN);
	//
	// Sets the timing of early interrupt generation.
	//
	ADC_setInterruptCycleOffset(adc_c_BASE, 0U);
	//
	// Powers up the analog-to-digital converter core.
	//
	ADC_enableConverter(adc_c_BASE);
	//
	// Delay for 1ms to allow ADC time to power up
	//
	DEVICE_DELAY_US(5000);
	//
	// Enable alternate timings for DMA trigger
	//
	ADC_enableAltDMATiming(adc_c_BASE);
	//
	// SOC Configuration: Setup ADC EPWM channel and trigger settings
	//
	// Disables SOC burst mode.
	//
	ADC_disableBurstMode(adc_c_BASE);
	//
	// Sets the priority mode of the SOCs.
	//
	ADC_setSOCPriority(adc_c_BASE, ADC_PRI_ALL_ROUND_ROBIN);
	//
	// Start of Conversion 0 Configuration
	//
	//
	// Disables Sample Capacitor to reset after each conversion.
	//
	ADC_disableSampleCAPReset(adc_c_BASE, ADC_SOC_NUMBER0);
	//
	// Configures a start-of-conversion (SOC) in the ADC and its interrupt SOC trigger.
	// 	  	SOC number		: 0
	//	  	Trigger			: ADC_TRIGGER_SW_ONLY
	//	  	Channel			: ADC_CH_ADCIN0
	//	 	Sample Window	: 12 SYSCLK cycles
	//		Interrupt Trigger: ADC_INT_SOC_TRIGGER_NONE
	//
	ADC_setupSOC(adc_c_BASE, ADC_SOC_NUMBER0, ADC_TRIGGER_SW_ONLY, ADC_CH_ADCIN0, 12U);
	ADC_setInterruptSOCTrigger(adc_c_BASE, ADC_SOC_NUMBER0, ADC_INT_SOC_TRIGGER_NONE);
	//
	// Start of Conversion 1 Configuration
	//
	//
	// Disables Sample Capacitor to reset after each conversion.
	//
	ADC_disableSampleCAPReset(adc_c_BASE, ADC_SOC_NUMBER1);
	//
	// Configures a start-of-conversion (SOC) in the ADC and its interrupt SOC trigger.
	// 	  	SOC number		: 1
	//	  	Trigger			: ADC_TRIGGER_SW_ONLY
	//	  	Channel			: ADC_CH_ADCIN0
	//	 	Sample Window	: 12 SYSCLK cycles
	//		Interrupt Trigger: ADC_INT_SOC_TRIGGER_NONE
	//
	ADC_setupSOC(adc_c_BASE, ADC_SOC_NUMBER1, ADC_TRIGGER_SW_ONLY, ADC_CH_ADCIN0, 12U);
	ADC_setInterruptSOCTrigger(adc_c_BASE, ADC_SOC_NUMBER1, ADC_INT_SOC_TRIGGER_NONE);
}


//*****************************************************************************
//
// ASYSCTL Configurations
//
//*****************************************************************************
void ASYSCTL_init(){
	//
	// asysctl initialization
	//
	// Disables the temperature sensor output to the ADC.
	//
	ASysCtl_disableTemperatureSensor();
	//
	// Set the analog voltage reference selection to internal.
	//
	ASysCtl_setAnalogReferenceInternal( ASYSCTL_ANAREF_INTREF_ADCA | ASYSCTL_ANAREF_INTREF_ADCB | ASYSCTL_ANAREF_INTREF_ADCC | ASYSCTL_ANAREF_INTREF_ADCD | ASYSCTL_ANAREF_INTREF_ADCE );

	//
	// Set the internal analog voltage reference selection to 2.5V.
	//
	ASysCtl_setAnalogReference2P5( ASYSCTL_ANAREF_ADCA | ASYSCTL_ANAREF_ADCB | ASYSCTL_ANAREF_ADCC | ASYSCTL_ANAREF_ADCD | ASYSCTL_ANAREF_ADCE );
}

//*****************************************************************************
//
// CMPSS Configurations
//
//*****************************************************************************
void CMPSS_init(){
	cmp1_init();
}

void cmp1_init(){
    //
    // Select the value for CMP1HPMXSEL.
    //
    ASysCtl_selectCMPHPMux(ASYSCTL_CMPHPMUX_SELECT_1,0U);
    //
    // Select the value for CMP1LPMXSEL.
    //
    ASysCtl_selectCMPLPMux(ASYSCTL_CMPLPMUX_SELECT_1,0U);
    //
    // Sets the configuration for the high comparator.
    //
    CMPSS_configHighComparator(cmp1_BASE,(CMPSS_INSRC_DAC));
    //
    // Sets the configuration for the low comparator.
    //
    CMPSS_configLowComparator(cmp1_BASE,(CMPSS_INSRC_DAC));
    //
    // Sets the configuration for the internal comparator DACs.
    //
    CMPSS_configDACHigh(cmp1_BASE,(CMPSS_DACVAL_SYSCLK | CMPSS_DACSRC_SHDW));
    CMPSS_configDACLow(cmp1_BASE, CMPSS_DACSRC_SHDW);
    //
    // Sets the value of the internal DAC of the high comparator.
    //
    CMPSS_setDACValueHigh(cmp1_BASE,2200U);
    //
    // Sets the value of the internal DAC of the low comparator.
    //
    CMPSS_setDACValueLow(cmp1_BASE,2200U);
    //
    //  Configures the digital filter of the high comparator.
    //
    CMPSS_configFilterHigh(cmp1_BASE, 499U, 19U, 15U);
    //
    // Configures the digital filter of the low comparator.
    //
    CMPSS_configFilterLow(cmp1_BASE, 499U, 19U, 15U);
    //
    // Sets the output signal configuration for the high comparator.
    //
    CMPSS_configOutputsHigh(cmp1_BASE,(CMPSS_TRIPOUT_ASYNC_COMP | CMPSS_TRIP_ASYNC_COMP));
    //
    // Sets the output signal configuration for the low comparator.
    //
    CMPSS_configOutputsLow(cmp1_BASE,(CMPSS_TRIPOUT_ASYNC_COMP | CMPSS_TRIP_ASYNC_COMP));
    //
    // Sets the comparator hysteresis settings.
    //
    CMPSS_setHysteresis(cmp1_BASE,0U);
    //
    // Configures the comparator subsystem's high ramp generator.
    //
    CMPSS_configRampHigh(cmp1_BASE, CMPSS_RAMP_DIR_DOWN, 0U,0U,0U,1U,true);
    //
    // Configures the comparator subsystem's low ramp generator.
    //
    CMPSS_configRampLow(cmp1_BASE, CMPSS_RAMP_DIR_DOWN, 0U,0U,0U,1U,true);
    //
    // Configures the high comparator's ramp generator clock divider
    //
    CMPSS_setRampClockDividerHigh(cmp1_BASE, CMPSS_RAMP_CLOCK_DIV1);
    //
    // Configures the low comparator's ramp generator clock divider
    //
    CMPSS_setRampClockDividerLow(cmp1_BASE, CMPSS_RAMP_CLOCK_DIV1);
    //
    // Disables reset of HIGH comparator digital filter output latch on PWMSYNC
    //
    CMPSS_disableLatchResetOnPWMSYNCHigh(cmp1_BASE);
    //
    // Disables reset of LOW comparator digital filter output latch on PWMSYNC
    //
    CMPSS_disableLatchResetOnPWMSYNCLow(cmp1_BASE);
    //
    // Sets the ePWM module blanking signal that holds trip in reset.
    //
    CMPSS_configBlanking(cmp1_BASE,1U);
    //
    // Disables an ePWM blanking signal from holding trip in reset.
    //
    CMPSS_disableBlanking(cmp1_BASE);
    //
    // Configures whether or not the digital filter latches are reset by PWMSYNC
    //
    CMPSS_configLatchOnPWMSYNC(cmp1_BASE,false,false);
    //
    // Enables the CMPSS module.
    //
    CMPSS_enableModule(cmp1_BASE);
    //
    // Delay for CMPSS DAC to power up.
    //
    DEVICE_DELAY_US(500);
}

//*****************************************************************************
//
// CPUTIMER Configurations
//
//*****************************************************************************
void CPUTIMER_init(){
	cpu_timer1_init();
}

void cpu_timer1_init(){
	CPUTimer_setEmulationMode(cpu_timer1_BASE, CPUTIMER_EMULATIONMODE_STOPAFTERNEXTDECREMENT);
	CPUTimer_setPreScaler(cpu_timer1_BASE, 150U);
	CPUTimer_setPeriod(cpu_timer1_BASE, 1000U);
	CPUTimer_enableInterrupt(cpu_timer1_BASE);
	CPUTimer_stopTimer(cpu_timer1_BASE);

	CPUTimer_reloadTimerCounter(cpu_timer1_BASE);
	CPUTimer_startTimer(cpu_timer1_BASE);
}

//*****************************************************************************
//
// ECAP Configurations
//
//*****************************************************************************
void ECAP_init(){
	cap1_init();
}

void cap1_init(){
	//
	// Disables time stamp capture.
	//
	ECAP_disableTimeStampCapture(cap1_BASE);
	//
	// Stops Time stamp counter.
	//
	ECAP_stopCounter(cap1_BASE);
	//
	// Sets eCAP in Capture mode.
	//
	ECAP_enableCaptureMode(cap1_BASE);
	//
	// Sets the capture mode.
	//
	ECAP_setCaptureMode(cap1_BASE,ECAP_CONTINUOUS_CAPTURE_MODE,ECAP_EVENT_1);
	//
	// Sets the Capture event prescaler.
	//
	ECAP_setEventPrescaler(cap1_BASE, 0U);
	//
	// Sets the Capture event polarity.
	//
	ECAP_setEventPolarity(cap1_BASE,ECAP_EVENT_1,ECAP_EVNT_RISING_EDGE);
	ECAP_setEventPolarity(cap1_BASE,ECAP_EVENT_2,ECAP_EVNT_RISING_EDGE);
	ECAP_setEventPolarity(cap1_BASE,ECAP_EVENT_3,ECAP_EVNT_RISING_EDGE);
	ECAP_setEventPolarity(cap1_BASE,ECAP_EVENT_4,ECAP_EVNT_RISING_EDGE);
	//
	// Configure counter reset on events
	//
	ECAP_disableCounterResetOnEvent(cap1_BASE,ECAP_EVENT_1);
	ECAP_disableCounterResetOnEvent(cap1_BASE,ECAP_EVENT_2);
	ECAP_disableCounterResetOnEvent(cap1_BASE,ECAP_EVENT_3);
	ECAP_disableCounterResetOnEvent(cap1_BASE,ECAP_EVENT_4);
	//
	// Select eCAP input.
	//
	ECAP_selectECAPInput(cap1_BASE,ECAP_INPUT_CMPSS1_CTRIP_HIGH_OR_LOW);
	//
	// Sets a phase shift value count.
	//
	ECAP_setPhaseShiftCount(cap1_BASE,0U);
	//
	// Disable counter loading with phase shift value.
	//
	ECAP_disableLoadCounter(cap1_BASE);
	//
	// Configures Sync out signal mode.
	//
	ECAP_setSyncOutMode(cap1_BASE,ECAP_SYNC_OUT_SYNCI);
	//
	// Configures emulation mode.
	//
	ECAP_setEmulationMode(cap1_BASE,ECAP_EMULATION_STOP);
	//
	// Set up the source for sync-in pulse..
	//
	ECAP_setSyncInPulseSource(cap1_BASE,ECAP_SYNC_IN_PULSE_SRC_DISABLE);
	//
	// Starts Time stamp counter for cap1.
	//
	ECAP_startCounter(cap1_BASE);
	//
	// Enables time stamp capture for cap1.
	//
	ECAP_enableTimeStampCapture(cap1_BASE);
	//
	// Re-arms the eCAP module for cap1.
	//
	ECAP_reArm(cap1_BASE);

}

//*****************************************************************************
//
// EPWM Configurations
//
//*****************************************************************************
void EPWM_init(){
    EPWM_setClockPrescaler(pwm_a_BASE, EPWM_CLOCK_DIVIDER_1, EPWM_HSCLOCK_DIVIDER_2);	
    EPWM_setTimeBasePeriod(pwm_a_BASE, 2000);	
    EPWM_setTimeBaseCounter(pwm_a_BASE, 0);	
    EPWM_setTimeBaseCounterMode(pwm_a_BASE, EPWM_COUNTER_MODE_UP_DOWN);	
    EPWM_disablePhaseShiftLoad(pwm_a_BASE);	
    EPWM_setPhaseShift(pwm_a_BASE, 0);	
    EPWM_setSyncInPulseSource(pwm_a_BASE, EPWM_SYNC_IN_PULSE_SRC_DISABLE);	
    EPWM_setCounterCompareValue(pwm_a_BASE, EPWM_COUNTER_COMPARE_A, 50);	
    EPWM_setCounterCompareShadowLoadMode(pwm_a_BASE, EPWM_COUNTER_COMPARE_A, EPWM_COMP_LOAD_ON_CNTR_ZERO);	
    EPWM_setCounterCompareValue(pwm_a_BASE, EPWM_COUNTER_COMPARE_B, 1950);	
    EPWM_setCounterCompareShadowLoadMode(pwm_a_BASE, EPWM_COUNTER_COMPARE_B, EPWM_COMP_LOAD_ON_CNTR_ZERO);	
    EPWM_disableActionQualifierShadowLoadMode(pwm_a_BASE, EPWM_ACTION_QUALIFIER_A);	
    EPWM_setActionQualifierShadowLoadMode(pwm_a_BASE, EPWM_ACTION_QUALIFIER_A, EPWM_AQ_LOAD_ON_CNTR_ZERO);	
    EPWM_setActionQualifierAction(pwm_a_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);	
    EPWM_setActionQualifierAction(pwm_a_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_PERIOD);	
    EPWM_setActionQualifierAction(pwm_a_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);	
    EPWM_setActionQualifierAction(pwm_a_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);	
    EPWM_setActionQualifierAction(pwm_a_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPB);	
    EPWM_setActionQualifierAction(pwm_a_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB);	
    EPWM_disableActionQualifierShadowLoadMode(pwm_a_BASE, EPWM_ACTION_QUALIFIER_B);	
    EPWM_setActionQualifierShadowLoadMode(pwm_a_BASE, EPWM_ACTION_QUALIFIER_B, EPWM_AQ_LOAD_ON_CNTR_ZERO);	
    EPWM_setActionQualifierAction(pwm_a_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);	
    EPWM_setActionQualifierAction(pwm_a_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_PERIOD);	
    EPWM_setActionQualifierAction(pwm_a_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);	
    EPWM_setActionQualifierAction(pwm_a_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);	
    EPWM_setActionQualifierAction(pwm_a_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPB);	
    EPWM_setActionQualifierAction(pwm_a_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB);	
    EPWM_setDeadBandDelayMode(pwm_a_BASE, EPWM_DB_RED, true);	
    EPWM_setRisingEdgeDelayCountShadowLoadMode(pwm_a_BASE, EPWM_RED_LOAD_ON_CNTR_ZERO);	
    EPWM_disableRisingEdgeDelayCountShadowLoadMode(pwm_a_BASE);	
    EPWM_setRisingEdgeDelayCount(pwm_a_BASE, 150);	
    EPWM_setDeadBandDelayMode(pwm_a_BASE, EPWM_DB_FED, true);	
    EPWM_setFallingEdgeDelayCountShadowLoadMode(pwm_a_BASE, EPWM_FED_LOAD_ON_CNTR_ZERO);	
    EPWM_disableFallingEdgeDelayCountShadowLoadMode(pwm_a_BASE);	
    EPWM_setFallingEdgeDelayCount(pwm_a_BASE, 150);	
    EPWM_enableTripZoneSignals(pwm_a_BASE, EPWM_TZ_SIGNAL_OSHT1);	
    EPWM_setClockPrescaler(pwm_b_BASE, EPWM_CLOCK_DIVIDER_1, EPWM_HSCLOCK_DIVIDER_2);	
    EPWM_setTimeBasePeriod(pwm_b_BASE, 2000);	
    EPWM_setTimeBaseCounter(pwm_b_BASE, 0);	
    EPWM_setTimeBaseCounterMode(pwm_b_BASE, EPWM_COUNTER_MODE_UP_DOWN);	
    EPWM_disablePhaseShiftLoad(pwm_b_BASE);	
    EPWM_setPhaseShift(pwm_b_BASE, 0);	
    EPWM_setCounterCompareValue(pwm_b_BASE, EPWM_COUNTER_COMPARE_A, 50);	
    EPWM_setCounterCompareShadowLoadMode(pwm_b_BASE, EPWM_COUNTER_COMPARE_A, EPWM_COMP_LOAD_ON_CNTR_ZERO);	
    EPWM_setCounterCompareValue(pwm_b_BASE, EPWM_COUNTER_COMPARE_B, 1950);	
    EPWM_setCounterCompareShadowLoadMode(pwm_b_BASE, EPWM_COUNTER_COMPARE_B, EPWM_COMP_LOAD_ON_CNTR_ZERO);	
    EPWM_disableActionQualifierShadowLoadMode(pwm_b_BASE, EPWM_ACTION_QUALIFIER_A);	
    EPWM_setActionQualifierShadowLoadMode(pwm_b_BASE, EPWM_ACTION_QUALIFIER_A, EPWM_AQ_LOAD_ON_CNTR_ZERO);	
    EPWM_setActionQualifierAction(pwm_b_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);	
    EPWM_setActionQualifierAction(pwm_b_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_PERIOD);	
    EPWM_setActionQualifierAction(pwm_b_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);	
    EPWM_setActionQualifierAction(pwm_b_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);	
    EPWM_setActionQualifierAction(pwm_b_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPB);	
    EPWM_setActionQualifierAction(pwm_b_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB);	
    EPWM_disableActionQualifierShadowLoadMode(pwm_b_BASE, EPWM_ACTION_QUALIFIER_B);	
    EPWM_setActionQualifierShadowLoadMode(pwm_b_BASE, EPWM_ACTION_QUALIFIER_B, EPWM_AQ_LOAD_ON_CNTR_ZERO);	
    EPWM_setActionQualifierAction(pwm_b_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);	
    EPWM_setActionQualifierAction(pwm_b_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_PERIOD);	
    EPWM_setActionQualifierAction(pwm_b_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);	
    EPWM_setActionQualifierAction(pwm_b_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);	
    EPWM_setActionQualifierAction(pwm_b_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPB);	
    EPWM_setActionQualifierAction(pwm_b_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB);	
    EPWM_setDeadBandDelayMode(pwm_b_BASE, EPWM_DB_RED, true);	
    EPWM_setRisingEdgeDelayCountShadowLoadMode(pwm_b_BASE, EPWM_RED_LOAD_ON_CNTR_ZERO);	
    EPWM_disableRisingEdgeDelayCountShadowLoadMode(pwm_b_BASE);	
    EPWM_setRisingEdgeDelayCount(pwm_b_BASE, 150);	
    EPWM_setDeadBandDelayMode(pwm_b_BASE, EPWM_DB_FED, true);	
    EPWM_setFallingEdgeDelayCountShadowLoadMode(pwm_b_BASE, EPWM_FED_LOAD_ON_CNTR_ZERO);	
    EPWM_disableFallingEdgeDelayCountShadowLoadMode(pwm_b_BASE);	
    EPWM_setFallingEdgeDelayCount(pwm_b_BASE, 150);	
    EPWM_enableTripZoneSignals(pwm_b_BASE, EPWM_TZ_SIGNAL_OSHT1);	
    EPWM_setClockPrescaler(pwm_c_BASE, EPWM_CLOCK_DIVIDER_1, EPWM_HSCLOCK_DIVIDER_2);	
    EPWM_setTimeBasePeriod(pwm_c_BASE, 2000);	
    EPWM_setTimeBaseCounter(pwm_c_BASE, 0);	
    EPWM_setTimeBaseCounterMode(pwm_c_BASE, EPWM_COUNTER_MODE_STOP_FREEZE);	
    EPWM_disablePhaseShiftLoad(pwm_c_BASE);	
    EPWM_setPhaseShift(pwm_c_BASE, 0);	
    EPWM_setCounterCompareValue(pwm_c_BASE, EPWM_COUNTER_COMPARE_A, 50);	
    EPWM_setCounterCompareShadowLoadMode(pwm_c_BASE, EPWM_COUNTER_COMPARE_A, EPWM_COMP_LOAD_ON_CNTR_ZERO);	
    EPWM_setCounterCompareValue(pwm_c_BASE, EPWM_COUNTER_COMPARE_B, 1950);	
    EPWM_setCounterCompareShadowLoadMode(pwm_c_BASE, EPWM_COUNTER_COMPARE_B, EPWM_COMP_LOAD_ON_CNTR_ZERO);	
    EPWM_disableActionQualifierShadowLoadMode(pwm_c_BASE, EPWM_ACTION_QUALIFIER_A);	
    EPWM_setActionQualifierShadowLoadMode(pwm_c_BASE, EPWM_ACTION_QUALIFIER_A, EPWM_AQ_LOAD_ON_CNTR_ZERO);	
    EPWM_setActionQualifierAction(pwm_c_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);	
    EPWM_setActionQualifierAction(pwm_c_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_PERIOD);	
    EPWM_setActionQualifierAction(pwm_c_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_HIGH, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);	
    EPWM_setActionQualifierAction(pwm_c_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_LOW, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);	
    EPWM_setActionQualifierAction(pwm_c_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPB);	
    EPWM_setActionQualifierAction(pwm_c_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB);	
    EPWM_disableActionQualifierShadowLoadMode(pwm_c_BASE, EPWM_ACTION_QUALIFIER_B);	
    EPWM_setActionQualifierShadowLoadMode(pwm_c_BASE, EPWM_ACTION_QUALIFIER_B, EPWM_AQ_LOAD_ON_CNTR_ZERO);	
    EPWM_setActionQualifierAction(pwm_c_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);	
    EPWM_setActionQualifierAction(pwm_c_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_PERIOD);	
    EPWM_setActionQualifierAction(pwm_c_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);	
    EPWM_setActionQualifierAction(pwm_c_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);	
    EPWM_setActionQualifierAction(pwm_c_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPB);	
    EPWM_setActionQualifierAction(pwm_c_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB);	
    EPWM_setDeadBandDelayMode(pwm_c_BASE, EPWM_DB_RED, true);	
    EPWM_setRisingEdgeDelayCountShadowLoadMode(pwm_c_BASE, EPWM_RED_LOAD_ON_CNTR_ZERO);	
    EPWM_disableRisingEdgeDelayCountShadowLoadMode(pwm_c_BASE);	
    EPWM_setRisingEdgeDelayCount(pwm_c_BASE, 150);	
    EPWM_setDeadBandDelayMode(pwm_c_BASE, EPWM_DB_FED, true);	
    EPWM_setFallingEdgeDelayCountShadowLoadMode(pwm_c_BASE, EPWM_FED_LOAD_ON_CNTR_ZERO);	
    EPWM_disableFallingEdgeDelayCountShadowLoadMode(pwm_c_BASE);	
    EPWM_setFallingEdgeDelayCount(pwm_c_BASE, 150);	
    EPWM_enableTripZoneSignals(pwm_c_BASE, EPWM_TZ_SIGNAL_OSHT1);	
    EPWM_setClockPrescaler(pwm_break_BASE, EPWM_CLOCK_DIVIDER_1, EPWM_HSCLOCK_DIVIDER_2);	
    EPWM_setTimeBasePeriod(pwm_break_BASE, 20000);	
    EPWM_setTimeBaseCounter(pwm_break_BASE, 0);	
    EPWM_setTimeBaseCounterMode(pwm_break_BASE, EPWM_COUNTER_MODE_UP_DOWN);	
    EPWM_disablePhaseShiftLoad(pwm_break_BASE);	
    EPWM_setPhaseShift(pwm_break_BASE, 0);	
    EPWM_setCounterCompareValue(pwm_break_BASE, EPWM_COUNTER_COMPARE_A, 0);	
    EPWM_setCounterCompareShadowLoadMode(pwm_break_BASE, EPWM_COUNTER_COMPARE_A, EPWM_COMP_LOAD_ON_CNTR_ZERO);	
    EPWM_setCounterCompareValue(pwm_break_BASE, EPWM_COUNTER_COMPARE_B, 0);	
    EPWM_setCounterCompareShadowLoadMode(pwm_break_BASE, EPWM_COUNTER_COMPARE_B, EPWM_COMP_LOAD_ON_CNTR_ZERO);	
    EPWM_disableActionQualifierShadowLoadMode(pwm_break_BASE, EPWM_ACTION_QUALIFIER_A);	
    EPWM_setActionQualifierShadowLoadMode(pwm_break_BASE, EPWM_ACTION_QUALIFIER_A, EPWM_AQ_LOAD_ON_CNTR_ZERO);	
    EPWM_setActionQualifierAction(pwm_break_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);	
    EPWM_setActionQualifierAction(pwm_break_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_PERIOD);	
    EPWM_setActionQualifierAction(pwm_break_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);	
    EPWM_setActionQualifierAction(pwm_break_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);	
    EPWM_setActionQualifierAction(pwm_break_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPB);	
    EPWM_setActionQualifierAction(pwm_break_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB);	
    EPWM_disableActionQualifierShadowLoadMode(pwm_break_BASE, EPWM_ACTION_QUALIFIER_B);	
    EPWM_setActionQualifierShadowLoadMode(pwm_break_BASE, EPWM_ACTION_QUALIFIER_B, EPWM_AQ_LOAD_ON_CNTR_ZERO);	
    EPWM_setActionQualifierAction(pwm_break_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);	
    EPWM_setActionQualifierAction(pwm_break_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_PERIOD);	
    EPWM_setActionQualifierAction(pwm_break_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);	
    EPWM_setActionQualifierAction(pwm_break_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);	
    EPWM_setActionQualifierAction(pwm_break_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPB);	
    EPWM_setActionQualifierAction(pwm_break_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB);	
    EPWM_setRisingEdgeDelayCountShadowLoadMode(pwm_break_BASE, EPWM_RED_LOAD_ON_CNTR_ZERO);	
    EPWM_disableRisingEdgeDelayCountShadowLoadMode(pwm_break_BASE);	
    EPWM_setFallingEdgeDelayCountShadowLoadMode(pwm_break_BASE, EPWM_FED_LOAD_ON_CNTR_ZERO);	
    EPWM_disableFallingEdgeDelayCountShadowLoadMode(pwm_break_BASE);	
}

//*****************************************************************************
//
// GPIO Configurations
//
//*****************************************************************************
void GPIO_init(){
	rs485_ena_init();
	fan_ctrl_init();
	relay_ctrl_init();
	v2_ctrl_init();
	di1_init();
	di2_init();
	di3_init();
	bus_ov_init();
	bus_uv_init();
}

void rs485_ena_init(){
	GPIO_writePin(rs485_ena, 0);
	GPIO_setPadConfig(rs485_ena, GPIO_PIN_TYPE_STD | GPIO_PIN_TYPE_PULLUP);
	GPIO_setQualificationMode(rs485_ena, GPIO_QUAL_SYNC);
	GPIO_setDirectionMode(rs485_ena, GPIO_DIR_MODE_OUT);
	GPIO_setControllerCore(rs485_ena, GPIO_CORE_CPU1);
}
void fan_ctrl_init(){
	GPIO_writePin(fan_ctrl, 0);
	GPIO_setPadConfig(fan_ctrl, GPIO_PIN_TYPE_STD | GPIO_PIN_TYPE_PULLUP);
	GPIO_setQualificationMode(fan_ctrl, GPIO_QUAL_SYNC);
	GPIO_setDirectionMode(fan_ctrl, GPIO_DIR_MODE_OUT);
	GPIO_setControllerCore(fan_ctrl, GPIO_CORE_CPU1);
}
void relay_ctrl_init(){
	GPIO_writePin(relay_ctrl, 0);
	GPIO_setPadConfig(relay_ctrl, GPIO_PIN_TYPE_STD | GPIO_PIN_TYPE_PULLUP);
	GPIO_setQualificationMode(relay_ctrl, GPIO_QUAL_SYNC);
	GPIO_setDirectionMode(relay_ctrl, GPIO_DIR_MODE_OUT);
	GPIO_setControllerCore(relay_ctrl, GPIO_CORE_CPU1);
}
void v2_ctrl_init(){
	GPIO_writePin(v2_ctrl, 0);
	GPIO_setPadConfig(v2_ctrl, GPIO_PIN_TYPE_STD | GPIO_PIN_TYPE_PULLUP);
	GPIO_setQualificationMode(v2_ctrl, GPIO_QUAL_SYNC);
	GPIO_setDirectionMode(v2_ctrl, GPIO_DIR_MODE_OUT);
	GPIO_setControllerCore(v2_ctrl, GPIO_CORE_CPU1);
}
void di1_init(){
	GPIO_setPadConfig(di1, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(di1, GPIO_QUAL_SYNC);
	GPIO_setDirectionMode(di1, GPIO_DIR_MODE_IN);
	GPIO_setControllerCore(di1, GPIO_CORE_CPU1);
}
void di2_init(){
	GPIO_setPadConfig(di2, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(di2, GPIO_QUAL_SYNC);
	GPIO_setDirectionMode(di2, GPIO_DIR_MODE_IN);
	GPIO_setControllerCore(di2, GPIO_CORE_CPU1);
}
void di3_init(){
	GPIO_setPadConfig(di3, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(di3, GPIO_QUAL_SYNC);
	GPIO_setDirectionMode(di3, GPIO_DIR_MODE_IN);
	GPIO_setControllerCore(di3, GPIO_CORE_CPU1);
}
void bus_ov_init(){
	GPIO_setPadConfig(bus_ov, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(bus_ov, GPIO_QUAL_SYNC);
	GPIO_setDirectionMode(bus_ov, GPIO_DIR_MODE_IN);
	GPIO_setControllerCore(bus_ov, GPIO_CORE_CPU1);
}
void bus_uv_init(){
	GPIO_setPadConfig(bus_uv, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(bus_uv, GPIO_QUAL_SYNC);
	GPIO_setDirectionMode(bus_uv, GPIO_DIR_MODE_IN);
	GPIO_setControllerCore(bus_uv, GPIO_CORE_CPU1);
}

//*****************************************************************************
//
// I2C Configurations
//
//*****************************************************************************
void I2C_init(){
	i2c_a_init();
}

void i2c_a_init(){
	I2C_disableModule(i2c_a_BASE);
	I2C_initController(i2c_a_BASE, DEVICE_SYSCLK_FREQ, i2c_a_BITRATE, I2C_DUTYCYCLE_33);
	I2C_setConfig(i2c_a_BASE, I2C_CONTROLLER_SEND_MODE);
	I2C_disableLoopback(i2c_a_BASE);
	I2C_setOwnAddress(i2c_a_BASE, i2c_a_OWN_ADDRESS);
	I2C_setTargetAddress(i2c_a_BASE, i2c_a_TARGET_ADDRESS);
	I2C_setBitCount(i2c_a_BASE, I2C_BITCOUNT_1);
	I2C_setDataCount(i2c_a_BASE, 1);
	I2C_setAddressMode(i2c_a_BASE, I2C_ADDR_MODE_7BITS);
	I2C_disableExtendedAutomaticClkStretchingMode(i2c_a_BASE);
	I2C_enableFIFO(i2c_a_BASE);
	I2C_setFIFOInterruptLevel(i2c_a_BASE, I2C_FIFO_TXEMPTY, I2C_FIFO_RXEMPTY);
	I2C_setEmulationMode(i2c_a_BASE, I2C_EMULATION_STOP_SCL_LOW);
	I2C_enableModule(i2c_a_BASE);
}

//*****************************************************************************
//
// INTERRUPT Configurations
//
//*****************************************************************************
void INTERRUPT_init(){
	
	// Interrupt Settings for INT_cpu_timer1
	// ISR need to be defined for the registered interrupts
	Interrupt_register(INT_cpu_timer1, &INT_cpu_timer1_ISR);
	Interrupt_enable(INT_cpu_timer1);
}
//*****************************************************************************
//
// SCI Configurations
//
//*****************************************************************************
void SCI_init(){
	sci_a_init();
	sci_b_init();
}

void sci_a_init(){
	SCI_clearInterruptStatus(sci_a_BASE, SCI_INT_RXFF | SCI_INT_TXFF | SCI_INT_FE | SCI_INT_OE | SCI_INT_PE | SCI_INT_RXERR | SCI_INT_RXRDY_BRKDT | SCI_INT_TXRDY);
	SCI_clearOverflowStatus(sci_a_BASE);
	SCI_resetTxFIFO(sci_a_BASE);
	SCI_resetRxFIFO(sci_a_BASE);
	SCI_resetChannels(sci_a_BASE);
	SCI_setConfig(sci_a_BASE, DEVICE_LSPCLK_FREQ, sci_a_BAUDRATE, (SCI_CONFIG_WLEN_8|SCI_CONFIG_STOP_ONE|SCI_CONFIG_PAR_NONE));
	SCI_disableLoopback(sci_a_BASE);
	SCI_performSoftwareReset(sci_a_BASE);
	SCI_setFIFOInterruptLevel(sci_a_BASE, SCI_FIFO_TX0, SCI_FIFO_RX0);
	SCI_enableFIFO(sci_a_BASE);
	SCI_enableModule(sci_a_BASE);
}
void sci_b_init(){
	SCI_clearInterruptStatus(sci_b_BASE, SCI_INT_RXFF | SCI_INT_TXFF | SCI_INT_FE | SCI_INT_OE | SCI_INT_PE | SCI_INT_RXERR | SCI_INT_RXRDY_BRKDT | SCI_INT_TXRDY);
	SCI_clearOverflowStatus(sci_b_BASE);
	SCI_resetTxFIFO(sci_b_BASE);
	SCI_resetRxFIFO(sci_b_BASE);
	SCI_resetChannels(sci_b_BASE);
	SCI_setConfig(sci_b_BASE, DEVICE_LSPCLK_FREQ, sci_b_BAUDRATE, (SCI_CONFIG_WLEN_8|SCI_CONFIG_STOP_ONE|SCI_CONFIG_PAR_NONE));
	SCI_disableLoopback(sci_b_BASE);
	SCI_performSoftwareReset(sci_b_BASE);
	SCI_setFIFOInterruptLevel(sci_b_BASE, SCI_FIFO_TX0, SCI_FIFO_RX0);
	SCI_enableFIFO(sci_b_BASE);
	SCI_enableModule(sci_b_BASE);
}

//*****************************************************************************
//
// SYNC Scheme Configurations
//
//*****************************************************************************
void SYNC_init(){
	SysCtl_setSyncOutputConfig(SYSCTL_SYNC_OUT_SRC_EPWM1SYNCOUT);
	//
	// SOCA
	//
	SysCtl_enableExtADCSOCSource(0);
	//
	// SOCB
	//
	SysCtl_enableExtADCSOCSource(0);
}
