//
//  Code from manual "TMS320C28x Assembly Language Tools",
//    https://www.ti.com/lit/pdf/SPRU513
//    Appendix C - CRC Reference Implementation
//  Modified to use ISO exact-width integer types and always refer to
//  program memory as (uint16_t *)
//
//
/*****************************************************************************/
/* Reference implementation of a CRC calculation function */
/* */
/* gen_crc is the interface function which should be called from the */
/* application. There is also a stand-alone test mode that can be used */
/* if _RUN_MAIN is defined. */
/*****************************************************************************/
/*---------------------------------------------------------------------------*/
/* This file does NOT implement a general-purpose CRC function. */
/* Specifically, it does not handle parameterization by initial value, bit */
/* reflection, or final XOR value. This implementation is intended only to */
/* implement the CRC functions used by the linker for C28x CRC tables. The */
/* algorithms used by the linker are selected to match the CRC algorithms in */
/* the PRIME and IEEE 802.15.4-2006 standards, which use the polynomials */
/* supported by the C28x VCU hardware. To understand CRCs in general, */
/* especially what other parameters exist, see: */
/* */
/* "A Painless Guide To CRC Error Detection Algorithms" likely at: */
/* http://www.ross.net/crc/download/crc_v3.txt */
/* Author : Ross Williams (ross@guest.adelaide.edu.au.). */
/* Date : 3 June 1993. */
/* Status : Public domain (C code). */
/*---------------------------------------------------------------------------*/
#include  <stdio.h>
#include  <stdlib.h>
#include  <limits.h>
#include  <stdint.h>

#ifdef __TMS320C28XX__
#include  <crc_tbl.h>
#else
#endif  //  __TMS320C28XX__

#include  "ref_crc.h"

#ifdef __TMS320C28XX__
#pragma  SET_DATA_SECTION("ramgs1")

//
//  This changes everything: if CRC code is not in a separate section,
//  and included in tested memory, the CRC check fails!
//
//#pragma  SET_CODE_SECTION(".crc_code")

#else
#define CRC32_PRIME	    0	/* Poly = 0x04c11db7 */ /* DEFAULT ALGORITHM */
#define CRC16_802_15_4  1	/* Poly = 0x00001021 */
#define CRC16_ALT	    2	/* Poly = 0x00008005 */
#define CRC8_PRIME	    3	/* Poly = 0x00000007 */
#endif

typedef struct crc_config_t
{
    uint16_t    id;
    uint16_t    degree;
    uint32_t    poly;
} crc_config_t;

const crc_config_t crc_config[] = {
    {CRC32_PRIME,       32u,    0x04c11db7u},
    {CRC16_802_15_4,    16u,    0x1021u},
    {CRC16_ALT,         16u,    0x8005u},
    {CRC8_PRIME,        8u,     0x07u}
};

uint32_t    crc_table[256] = { 0 };

const crc_config_t *find_config(int id)
{
    size_t i;
    for (i = 0u; i < sizeof(crc_config) / sizeof(*crc_config); i++)
    {
        if (crc_config[i].id == id)
        {
            return &crc_config[i];
        }
    }
    //fprintf(stderr, "invalid config id %d\n", id);
    exit(EXIT_FAILURE);
    return NULL;
}

/*---------------------------------------------------------------------------*/
/* Table-driven version */
/*---------------------------------------------------------------------------*/
uint32_t generate_mask(uint16_t degree)
{
    uint32_t half = (1ul << (degree / 2)) - 1;
    return half << (degree / 2) | half;
} 

void generate_crc_table(const crc_config_t *config)
{
    uint16_t    i, j;
    uint32_t    bit, crc;
    uint32_t    high_bit = (1ul << (config->degree - 1u));
    uint32_t    mask = generate_mask(config->degree);
    for (i = 0u; i < 256u; i++)
    {
        crc = (uint32_t)i << (config->degree - 8u);
        for (j = 0u; j < 8u; j++)
        {
            bit = crc & high_bit;
            crc <<= 1u;
            if (bit)
            {
                crc ^= config->poly;
            }
        }
        crc_table[i] = crc & mask;
    }
}

/*****************************************************************************/
/* gen_crc - Return the CRC value for the data using the given CRC algorithm */
/* int id : identifies the CRC algorithm */
/* char *data : the data */
/* size_t len : the size of the data */
/*****************************************************************************/
uint32_t gen_crc(uint16_t id, const uint16_t *data, size_t len)
{
    /*-----------------------------------------------------------------------*/
    /* Note: this is not a general-purpose CRC function. It does not handle */
    /* parameterization by initial value, bit reflection, or final XOR */
    /* value. This CRC function is specialized to the CRC algorithms in the */
    /* linker used for C28x CRC tables. */
    /*-----------------------------------------------------------------------*/
    /*-----------------------------------------------------------------------*/
    /* This CRC function is not intended to be optimal; it is written such */
    /* that it works and generates the same result on all 8-bit and 16-bit */
    /* targets, including C28x, other TI DSPs, and typical desktops. */
    /*-----------------------------------------------------------------------*/
    const crc_config_t *config = find_config(id);
    uint32_t crc = 0;
    uint32_t mask = generate_mask(config->degree);
    size_t i;
    generate_crc_table(config);
    for (i = 0; i < len; i++)
    {
        uint16_t datum = data[i];
        uint16_t n;
        n = 0;
        do {
/*----------------------------------------------------------------*/
/* For 16-bit machines, we need to feed the octets in an */
/* arbitrary order. For C2000, the arbitrary order we choose is */
/* to feed the LEAST significant octet of char 0 first. The */
/* first octet fed to the CRC is the LEAST-significant octet of */
/* char 0; the second octet is the MOST-significant octet of char */
/* 0. See the "Special Note regarding 16-bit char" in the */
/* Assembly Language Tools User's Guide. */
/*----------------------------------------------------------------*/
#if  __TMS320C28XX__
            /*----------------------------------------------------------------*/
            /* Using __byte is not necessary; we use it here to illustrate */
            /* how it relates to octet order. */
            /*----------------------------------------------------------------*/
            uint32_t octet = (uint16_t)__byte((int *)&datum, n) & 0x0FFu;
#else
            uint32_t octet = ((datum >> (8 * n)) & 0xff);
#endif
            uint32_t term1 = (crc << 8);
            uint16_t idx = ((crc >> (config->degree - 8u)) & 0xffu) ^ octet;
            crc = term1 ^ crc_table[idx];
            
            n++;
        } while (n <= 1);
    }
    return crc & mask;
}


#ifdef __TMS320C28XX__
extern CRC_TABLE _code_crctable;


uint16_t ref_check_CRC(void)
{
    CRC_TABLE   *tp = &_code_crctable;

    uint16_t    i;
    uint16_t    ret_val = 1;
    uint32_t    my_crc;
    // printf("\n\tTABLE INFO: rec size=%d, num_rec=%d.",
    //        tp->rec_size, tp->num_recs);
    for (i = 0; i < tp->num_recs; i++)
    {
        CRC_RECORD crc_rec = tp->recs[i];
        /**************************************************/
        /* COMPUTE CRC OF DATA STARTING AT crc_rec.addr */
        /* FOR crc_rec.size UNITS. USE */
        /* crc_rec.crc_alg_ID to select algorithm. */
        /* COMPARE COMPUTED VALUE TO crc_rec.crc_value. */
        /**************************************************/
        my_crc = gen_crc(crc_rec.crc_alg_ID, (uint16_t *)crc_rec.addr,
                         crc_rec.size);
        // printf("\n\tCRC record: page=%x, alg=%x, addr = %lx, size=%lx, "
        //        "\n\t\tcrc=%lx, my_crc=%lx.",
        //        crc_rec.page_id, crc_rec.crc_alg_ID,
        //        crc_rec.addr, crc_rec.size, crc_rec.crc_value, my_crc);
        if (my_crc == crc_rec.crc_value)
        {
            // printf("\n\tCRCs match for record %d.\n", i);
        }
        else
        {
            ret_val = 0;
            // printf("\n\tCRCs DO NOT match for record %d.\n", i);
        }
    }
    
    return ret_val;
}
#endif  // __TMS320C28XX__




#if defined(_RUN_MAIN) || defined(_WIN32)
/*****************************************************************************/
/* main - If requested, compute the CRC of test data using each algorithm. */
/*****************************************************************************/
int main(void)
{
#if CHAR_BIT == 16
        const unsigned char data[] = { 'a',
                                       'b',
                                       'c',
                                       'd' };
#elif CHAR_BIT == 8
        /*-----------------------------------------------------------------------*/
        /* This represents "abcd" as it would appear in C2000 memory if we view */
        /* C2000 memory as octets, least-significant octet first; see "a special */
        /* note regarding 16-bit char" in Assembly Language Tools User's Guide. */
        /*-----------------------------------------------------------------------*/
        const unsigned char data[] = { 'a',
                                       0,
                                       'b',
                                       0,
                                       'c',
                                       0,
                                       'd',
                                       0 };
#endif
    /* CRC_8_PRIME: 0x70 */
    /* CRC_16_802: 0x1bd3 */
    /* CRC_32_PRIME: 0x4beab53b */
    const unsigned char *p = (const unsigned char *)data;
    unsigned long crc;
    crc = gen_crc(CRC32_PRIME, p, sizeof data);
    printf("CRC_32_PRIME: %08lx\n", crc);
    crc = gen_crc(CRC8_PRIME, p, sizeof data);
    printf("CRC_8_PRIME: %02lx\n", crc);
    crc = gen_crc(CRC16_802_15_4, p, sizeof data);
    printf("CRC16_802_15_4: %04lx\n", crc);
    return 0;
}
#endif

