//###########################################################################
// FILE:   flash_m3.c
// TITLE:  flash API example.
//###########################################################################
// $TI Release: F28M36x Support Library v210 $
// $Release Date: Wed Feb 15 16:20:49 CST 2017 $
// $Copyright: Copyright (C) 2012-2017 Texas Instruments Incorporated -
//             http://www.ti.com/ ALL RIGHTS RESERVED $
//###########################################################################

#include <string.h>
#include <stdbool.h>

#include "inc/hw_ints.h"
#include "inc/hw_memmap.h"
#include "inc/hw_nvic.h"
#include "inc/hw_types.h"
#include "inc/hw_sysctl.h"
#include "inc/hw_ipc.h"
#include "inc/hw_flash.h"
#include "driverlib/debug.h"
#include "driverlib/ipc.h"
#include "driverlib/interrupt.h"
#include "driverlib/sysctl.h"
#include "driverlib/flash.h"
#include "driverlib/device.h"
#include "driverlib/dcsm.h"
#include "flash_programming_m3.h"

//*****************************************************************************
// FILE Flash API include file
//*****************************************************************************
#include "F021_Concerto_Cortex.h"

#include "uart.h"
#include "flash.h"
#include "system_task.h"
#include "hardware_def.h"


#define I2C_SLV_ADDR_EXIO2 0x71 // Endereo 7 bits

//*****************************************************************************
//! \addtogroup dual_example_list
//! <h1>Flash API (flash)</h1>
//!
//! This example illustrates the usage of Flash API to erase and program the
//! flash.
//! Flash API functions are executed from RAM.  See linker command file for
//! details on loading the Flash API in Flash and executing it from RAM.
//!
//! \b To \b use \b the \b Flash \b API \n
//! 1.  If passwords are programmed in Flash, users are supposed to unlock the
//!     flash using the DCSMUnlockZone1CSM and DCSMUnlockZone2CSM functions with
//!     appropriate key if executing Flash API from unsecured memory.  See
//!     security chapter in TRM for details \n
//! 2.  Users should configure the desired system frequency \n
//! 3.  Users should configure waitstates for FSM operations based on desired
//!     system frequency using the formula: RWAIT = (SYSCLK/(2*24MHz))-1 \n
//! 4.  Users should include F021_CONCERTO_Cortex.h in the application \n
//! 5.  Users should include Flash API library to the project \n
//! 6.  Watchdog is disabled in this project.  If users want to enable and
//!     service the watchdog, users have to include the watchdog service code
//!     in Fapi_UserDefinedFunctions.c file \n
//! 7.  This example uses F28M35x_FlashAPIexample_Dual_m3.h file\n
//!
//!
//! Steps to convert code to run out of Flash instead of RAM are as follows: \n
//! 1. Change the command linker file to Flash linker file (i.e. F28M35x.cmd)\n
//! 2. Make sure initialized sections are mapped to Flash. \n
//! 3. Make sure ResetISR() or branch to starting point of your application is
//! mapped to "BOOT_TO_FLASH" address of 0x00200030. (0x13FFF0 on C28) \n
//! 4. Add security files /driverlib/dcsm_z1_secvalues.asm and/or
//! driverlib/dcsm_z2_secvalues.asm to program Z1 and Z2 security values. \n
//! 5. For best performance from the flash, modify the waitstates
//! and enable the flash pipeline as shown in this example. \n
//! Note: any code that manipulates the flash waitstate and pipeline
//! control must be run from RAM. Thus these functions are located
//! in their own memory section called ramfuncs. \n
//! 6. Code that must execute faster than Flash allows can also be
//! run from RAM (i.e. interrupt service routines). \n
//! 7. To copy code from Flash to RAM during runtime, set up .cmd file
//! appropriately and call MemCopy() function (in memcopy.c in utils) \n
//! 8. If using the IntRegister()/IntUnRegister() driverlib functions, replace
//! with IntRAMRegister()/IntRAMUnregister() functions which copy the Flash
//! interrupt vector table into RAM .vtable location. \n
//
//*****************************************************************************

//Lengths of sectors
//#define Bzero_16KSector_u32length   0x1000
//#define Bzero_32KSector_u32length   0x2000
//#define Bzero_64KSector_u32length   0x4000
//#define Bzero_128KSector_u32length  0x8000

//Data/Program Buffer used for testing the flash API functions
//#define  BYTES_IN_FLASH_BUFFER    0xFF   // Programming data buffer, bytes
//uint8    Buffer[BYTES_IN_FLASH_BUFFER + 1];

//*****************************************************************************
// Flags that contain the current value of the interrupt indicator as displayed
// on the CSTN display.
//*****************************************************************************
//unsigned long g_ulFlags;

// Structure CSMPSWKEY is defined in dcsm.h
//Users should modify this Zone1 key if the password locations in Flash are
// not all 0xFFFFFFFFs
CSMPSWDKEY Zone1KEY = {0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF};
//CSMPSWDKEY Zone1KEY = {0x200000, 0x200004, 0x200008, 0x20000C};

//Users should modify this Zone2 key if the password locations in Flash are
// not all 0xFFFFFFFFs
CSMPSWDKEY Zone2KEY = {0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF};
//CSMPSWDKEY Zone2KEY = {0x2FFFF0, 0x2FFFF4, 0x2FFFF8, 0x2FFFFC};

// These are defined by the linker (see device linker command file)
extern unsigned long RamfuncsLoadStart;
extern unsigned long RamfuncsRunStart;
extern unsigned long RamfuncsLoadSize;

volatile bool ExIo1Sts = 0;
volatile bool ExIo2Sts = 0;
static uint8_t StsExpIO1 = 0;
static uint8_t StsExpIO2 = 0;

uint8_t data_exio[10];

volatile bool LedStsMode = 0;

//*****************************************************************************
// The error routine that is called if the driver library encounters an error.
//*****************************************************************************
#ifdef DEBUG
void
__error__(char *pcFilename, unsigned long ulLine)
{
}

#endif

//*****************************************************************************
// For this example, if an error is found just stop here
//*****************************************************************************
//#pragma CODE_SECTION(Example_Error,ramFuncSection);
//void Example_Error(Fapi_StatusType status)
//{
//    //  Error code will be in the status parameter
//        for(;;);
//}


//*****************************************************************************
//  For this example, once we are done just stop here
//*****************************************************************************
#pragma CODE_SECTION(Example_Done,ramFuncSection);
void Example_Done(void)
{
    for(;;);
}

//*****************************************************************************
// This is an example code demonstrating F021 Flash API usage.  This code is 
// all in Flash.
//*****************************************************************************
uint8_t status, erase, write = 0;
int main(void)
{
    // Disable Protection
    HWREG(SYSCTL_MWRALLOW) =  0xA5A5A5A5;

    // Setup main clock tree for 75MHz - M3 and 150MHz - C28x
    SysCtlClockConfigSet(SYSCTL_SYSDIV_1 | SYSCTL_M3SSDIV_2 | SYSCTL_USE_PLL |
                             (SYSCTL_SPLLIMULT_M & 0x0F));
    //  Unlock Zone1 and Zone2 CSM.
    //

    //  If the API functions are going to run in unsecured RAM
    //  then the CSM must be unlocked in order for the flash
    //  API functions to access the flash.
    //  If the flash API functions are executed from secure memory
    //  then this step is not required.
    DCSMUnlockZone1CSM(&Zone1KEY);
    DCSMUnlockZone2CSM(&Zone2KEY);

    //  Copy API Functions and required user interface functions to RAM.
    //
    //  The flash API functions MUST be run out of RAM memory.
    //  If the Flash API is already in RAM then this step
    //  can be skipped.
    //  DO NOT run the Flash API from Flash
    memcpy(&RamfuncsRunStart, &RamfuncsLoadStart, (size_t)&RamfuncsLoadSize);

    //  Call Flash Initialization to setup flash waitstates
    //  This function must reside in RAM
    FlashInit();

    //  Send boot command to allow the C28 application to begin execution
    //  IPC command is sent to C28 while C28 boot ROM is executing and in IDLE state.
    //  During debug, if not executing C28 code via boot ROM, comment out the below
    //  line.
    //IPCMtoCBootControlSystem(CBROM_MTOC_BOOTMODE_BOOT_FROM_FLASH);

    // Disable clock supply for the watchdog modules
    SysCtlPeripheralDisable(SYSCTL_PERIPH_WDOG1);
    SysCtlPeripheralDisable(SYSCTL_PERIPH_WDOG0);

    initialize_uart();

    // Gain flash pump semaphore
    // This function must reside in RAM
    FlashGainPump();

    //  Jump to RAM and call the Flash API functions
    //Example_CallFlashAPI();

    //initialize_uart();

    IntMasterEnable();

    for(;;) {
        task_check();
    }
}
