#include "driverlib.h"

#define VECT_TABLE_LEN ((0xEBE - 0xD00)/2)
#define ISR_STACK_SIZE 0x100

typedef int func(void);

uint32_t swVectorTable[VECT_TABLE_LEN];
#pragma LOCATION(isrStack, 0xA000)
uint16_t isrStack[ISR_STACK_SIZE];
volatile uint16_t savedSP;
volatile uint16_t intNesting = 0;

void remapVectorTable();
__interrupt void commonISR(void);

void remapVectorTable()
{
    EALLOW;
    uint16_t i;
    for(i=0; i<VECT_TABLE_LEN; i++)
    {
        // Do not remap the follows vectors :
        // - Reset vector (ID=0)
        // - The following 12 vectors, as they are not used
        // - Timer2 (ID = 14) because it is used for RTOS interrupt
        // - NMI vector (ID = 18)
        // TODO: Add check for 14. removing for the sake of this example
        if((i <= 12) || (i == 14) || (i == 18))
            continue;

        swVectorTable[i] = HWREG(PIEVECTTABLE_BASE + i*2);
        HWREG(PIEVECTTABLE_BASE + i*2) = (uint32_t)&commonISR;

    }
    EDIS;
}

__interrupt void commonISR(void)
{
    if(intNesting == 0)
    {
        // Save SP
        asm(" MOVL    XAR4, #savedSP");
        asm(" MOVL    XAR6, #0");
        asm(" MOV     AR6, SP");
        asm(" MOVL    *XAR4, XAR6");

        // SP = isrStack
        asm(" MOVL    XAR4, #isrStack");
        asm(" MOV     @SP, AR4");
    }

    intNesting++;

    uint16_t pieVect;

    //
    // Calculate the vector ID. If the vector is in the lower PIE, it's the
    // offset of the vector that was fetched (bits 7:1 of PIECTRL.PIEVECT)
    // divided by two.
    //
    pieVect = HWREGH(PIECTRL_BASE + PIE_O_CTRL) & 0xFFFE;


    (*(func *)(swVectorTable[(pieVect - PIEVECTTABLE_BASE)/2]))();

    DINT;

    // Note : We need to make sure the following code is executed with interrupts disabled.
    // DINT above may be removed if it is done in individual ISRs.

    intNesting--;
    if(intNesting == 0)
    {
        // Restore SP
        asm(" MOVL    XAR0, #savedSP");
        asm(" MOVL    XAR0, *XAR0");
        asm(" MOV     @SP, AR0");
    }
}
