#ifndef _PHY_RX_H
#define _PHY_RX_H

/******************************************************************************
* FILE PURPOSE: Public header file module G3 PHY Rx Manager
*******************************************************************************
*
* FILE NAME: phy_rx.h
*
* DESCRIPTION:
*       This file contains public definitions, data structures and function
*       prototypes for PHY Rx manager.
*
*  Copyright (c) 2009 Texas Instruments Inc.
*  All Rights Reserved This program is the confidential and proprietary
*  product of Texas Instruments Inc.  Any Unauthorized use, reproduction or
*  transfer of this program is strictly prohibited.
*
* HISTORY:
*
* 02/27/2009 S.Yim    Adapted from PRIME
*
*
* LIST OF FUNCTION PROTOTYPES:
*
******************************************************************************/
#include <typedefs.h>
#include <phy.h>
#include <g3_phy.h>

/*****************************************************************************/
/* Definitions                                                               */
/*****************************************************************************/
/* Set types */
typedef enum
{
  PHY_RX_SET_AGC       = 0,
  PHY_RX_SET_PPDU_INFO = 1,
  PHY_RX_SET_TONEMASK  = 2,
  PHY_RX_SET_BLKILV    = 3,
  PHY_RX_SET_COH       = 4,
  PHY_RX_SET_BANDSEL   = 5,        // Set band (Cenelec or FCC)
  PHY_RX_NUM_SET_TYPES = 6

}PHY_rxSetType_t;

/* Get types */
typedef enum
{
  PHY_RX_GET_AGC       = 0,
  PHY_RX_GET_CD        = 1,
  PHY_RX_GET_NL        = 2,
  PHY_RX_GET_SNR       = 3,
  PHY_RX_GET_SB_SNR    = 4,
  PHY_RX_GET_RQ        = 5,
  PHY_RX_GET_TONEMASK  = 6,
  PHY_RX_GET_TONEMAP_INFO = 7,
  PHY_RX_GET_COH       = 8,
  PHY_RX_GET_BANDSEL   = 9,

  PHY_RX_NUM_GET_TYPES = 10

}PHY_rxGetType_t;

/* AAGC */
#define PHY_RX_AAGC_MODE_AUTO    0x0000
#define PHY_RX_AAGC_MODE_MANUAL  0x0001

/* coherent moduation */
#define PHY_RX_COH_MODE_ON       0x0001
#define PHY_RX_COH_MODE_OFF      0x0000

/* Carrier detect */
#define PHY_RX_CD_NOT_DETECTED      0x0000   // parm 0
#define PHY_RX_CD_DETECTED          0x0001
#define PHY_RX_PRMB_LEN_UNKNOWN          1   
#define PHY_RX_PRMB_LEN_KNOWN            0   

/* rx proc type */
typedef enum
{
  PHY_RX_PROC_SYMB = 0,
  PHY_RX_PROC_BIT = 1

}PHY_rxProc_t; 

/*****************************************************************************/
/* Data Structures                                                           */
/*****************************************************************************/
/* Rx PPDU information data structure */
typedef struct
{
  UINT16 id;       // ppdu ID
  UINT16 length;   // number of (MAC_H + payload) bytes)      
  UINT16 level;    // received level
                   // 0: <= 70 dBuV
                   // 1: <= 72 dBuV
                   // 15: < 98 dBuV
  UINT16 lqi;      // link quality indicator (SNR)
  UINT16 mcs;      // modulation coding scheme:
                   // 0: DBPSK
                   // 1: DQPSK
                   // 2: D8PSK
                   // 3: not used
                   // 4: DBPSK + FEC
                   // 5: DQPSK + FEC
                   // 6: D8PSK + FEC
  UINT16 *data_p;  // pointer to data (MAC_H + payload)
  SINT16 *sb_snr_p;  // pointer to subband SNR for the current packet
  UINT32 time;     // time when preamble is received    

}PHY_rxPpdu_t;

/* PHY RX statistics data structure */
typedef struct
{
  UINT16 phyCrcIncorrectCount; // # of bursts received with CRC errors
  UINT16 phyCrcFailCount;      // # of bursts received with correct CRC but invalid protocol header
  UINT16 phyRxDropCount;       // # of received PPDU drop owing buffer overrun
  UINT16 phyRxQueueLen;        // # of concurrent MPDUs Rx buffer can hold
  UINT32 phyRxProcDelay;       // time in us from PPDU received from AFE to MPDU available to MAC
  SINT16 phyAgcMinGain;        // minimum gain for AGC <0dB
  UINT16 phyAgcStepValue;      // distance between steps in dB < 6dB 
  UINT16 phyAgcStepNum;        // number of steps so that phyAgcMinGain+((phAgcStepNum - 1)*phyAgcStepValue) >= 21dB

  UINT32 phyRxPpduCount;       // # of packets received since switched on
}PHY_rx_stat_t;

/* Set AAGC data structure */
typedef struct
{
  UINT16 mode;                 // 0=auto, 1=manual
  UINT16 step;                 // If manual, AAGC gain step = 0 to N-1 in AAGC gain_step_dB

}PHY_rxAagcSet_t;

/* Set PPDU info data structure */
typedef struct
{
  UINT16 ppduId;               // PPDU ID that was received from PHY
  UINT16 nodeId;               // node ID for the PPDU received

}PHY_rxPpduSet_t;

/* Get AAGC info data structure */
typedef struct
{
  UINT16 mode;                 // 0=auto, 1=manual
  UINT16 step;                 // Current AAGC gain setting = 0 to N-1 in AAGC gain_step_dB

}PHY_rxAagcGet_t;

/* Get carrier detect data structure */
typedef struct
{
  UINT16 busy;                 // 1 = busy
                               // 0 = idle
}PHY_rxCdGet_t;

/* Set data structure */
typedef struct
{
  UINT16 nTones_start;
  UINT16 toneMask[G3_PHY_TONEMASK_WORDS]; // leave this as 6 for now even though it starts from second word
}PHY_rxToneMask_t;

typedef union
{
//  UINT16          toneMask[G3_PHY_TONEMASK_WORDS];
  PHY_rxToneMask_t toneMask;
  PHY_rxAagcSet_t aagc;
  PHY_rxPpduSet_t ppdu;
  UINT16          roboMode;
  UINT16          blkIlvMode;
  UINT16          cohMod;
  UINT16          bandsel;
}PHY_rxSetData_t;

/* Get data structure */
typedef union
{
  PHY_rxAagcGet_t aagc;
  PHY_rxCdGet_t   cd;
  UINT16          nl;          // noise level 0 to 15 (50 to 92 dBuV in 3 dB step)
  UINT16          snr;         // snr 0 tp 7 (0 to 18 dB in 3 dB step)
  UINT16          sb_snr[G3_PHY_MAX_NUM_SUBBANDS]; // sub band SNR
  UINT16          rq;          // receive quality (0 to 8 from bad to good)
  UINT16          roboMode;    // ROBO mode (1), PRIME mode (0)
//  UINT16          toneMask[G3_PHY_TONEMASK_WORDS];
  PHY_rxToneMask_t toneMask;
  UINT16          toneMapInfo; // received tone map information
  UINT16          cohMod;
  UINT16          bandsel;
}PHY_rxGetData_t;

/*****************************************************************************/
/* External variables                                                        */
/*****************************************************************************/


/*****************************************************************************/
/* Function Prototypes                                                       */
/*****************************************************************************/
PHY_status_t   PHY_rxInit(void);
PHY_status_t   PHY_rxStart(UINT16 timeOut, PHY_cbFunc_t cb_p);
PHY_status_t   PHY_rxSuspend(PHY_cbFunc_t cb_p);
PHY_status_t   PHY_rxResume(void);
PHY_status_t   PHY_rxPpduStart(PHY_cbFunc_t cb_p);
PHY_status_t   PHY_rxPpduStop(void);
PHY_status_t   PHY_rxPpduRelease(PHY_rxPpdu_t *ppdu_p);
PHY_status_t   PHY_rxSet(UINT16 setType, PHY_rxSetData_t *setData_p);
PHY_status_t   PHY_rxGet(UINT16 getType, PHY_rxGetData_t *getData_p);
PHY_status_t   PHY_rxGetStat(PHY_rx_stat_t *getData_p);
PHY_status_t   PHY_rxSmRun(PHY_rxProc_t procType);
PHY_status_t   PHY_rxBitStartIndicate(PHY_cbFunc_t cb_p);
PHY_status_t   PHY_rxPhyTimingIndicate(PHY_cbFunc_t cb_p);

#endif // _PHY_RX_H



