#ifndef _PHY_TEST_H
#define _PHY_TEST_H

/******************************************************************************
* FILE PURPOSE: Public header file module PRIME PHY diagnostic
*******************************************************************************
*
* FILE NAME: phy_test.h
*
* DESCRIPTION:
*       This file contains public definitions, data structures and function
*       prototypes for PRIME PHY diagnostic.
*
*  Copyright (c) 2009 Texas Instruments Inc.
*  All Rights Reserved This program is the confidential and proprietary
*  product of Texas Instruments Inc.  Any Unauthorized use, reproduction or
*  transfer of this program is strictly prohibited.
*
* HISTORY:
*
* 09/22/2009 S.Yim    Written
*
* LIST OF FUNCTION PROTOTYPES:
*
******************************************************************************/
//#include <project.h>
#include <phy.h>

/*****************************************************************************/
/* Definitions                                                               */
/*****************************************************************************/
/* PHY Test data pattern */
typedef enum
{
  PHY_TEST_DATA_NONE          = 0,   // no test data      
  PHY_TEST_DATA_OCTET         = 1,   // test data with fixed octet
  PHY_TEST_DATA_RAMP          = 2,   // test data with ramp pattern
  PHY_TEST_IOT_TC1            = 3,   // IOT test case 1
  PHY_TEST_IOT_TC2            = 4,   // IOT test case 2
  PHY_TEST_IOT_TC3            = 5,   // IOT test case 3
  PHY_TEST_IOT_TC4            = 6,   // IOT test case 4
  PHY_TEST_IOT_TC5            = 7    // IOT test case 5
}PHY_testPattern_t;

#define PHY_RX_NUM_MCS                   8

/* Tx Test Mode for diagnostic */
#define PHY_TX_TEST_MODE_ON           0x0001
#define PHY_TX_TEST_MCS_SWEEP         0x0002
#define PHY_TX_TEST_PPDU_LEN_SWEEP    0x0004
#define PHY_TX_TEST_PKT               0x0008
#define PHY_TX_TEST_CONT              0x0010

#define PHY_TX_TEST_ALL_SWEEP         (PHY_TX_TEST_MCS_SWEEP | PHY_TX_TEST_PPDU_LEN_SWEEP)

// IOT test defines
#define PHY_NUM_IOT_TCS		 5     // total number of IOT test cases
#define PHY_G3_IOT_MSG_LEN   24    // 48 byte of IOT message
#define PHY_G3_IOT_RAMP_LEN  67    // 133 byte of IOT ramp data




/*****************************************************************************/
/* Data Structures                                                           */
/*****************************************************************************/
// data structure for the IOT test set
typedef struct
{
  UINT16   pdc;               // PDC field
  UINT16   dt;                // DT field
  UINT16   mod;               // modulation
  UINT16   toneMap;           // tone map field
  UINT16   ppdu_len;          // length of the ppdu
  UINT16  *pattern_ptr;       // pointer to the data pattern
}PHY_iot_tst_t;

/* PHY Rx BERT */
typedef struct
{
  UINT32 bitErrCnt[PHY_RX_NUM_MCS];   // bit error count
  UINT32 bitTotCnt[PHY_RX_NUM_MCS];   // bit received count
  UINT32 bitTotCntErrorFree[PHY_RX_NUM_MCS];  // error free bit count
  UINT16 ppduErrCnt[PHY_RX_NUM_MCS];  // pkt error count
  UINT16 ppduTotCnt[PHY_RX_NUM_MCS];  // pkt received count

  UINT32 phyRawDataRate;              // phy raw data rate based on MCS
  UINT32 phyActualRxBits;             // phy actual rx bits; 
  UINT16 missingPktCnt;               // missing packet count
  UINT16 lastMCS;                     // latest MCS

}PHY_rxBert_t;

typedef struct
{
  UINT16         flags;                       // 1 = test mode on, count bit error
  UINT16         dataPattern;                 // see PHY_testPattern_t in phy.h
  UINT16         octet;                       // octet value when test pattern is octet

  PHY_rxBert_t   bert;
 
}PHY_rxTestMode_t;

/* tx test mode msg */
typedef struct
{
  UINT16 flags;       // see test mode above
  UINT16 ppduLength;  // Number of bytes per PPDU
  UINT16 numPpdu;     // Number of PPDU per setting
  UINT16 mcs;         // Modulation (DBPSK/DQPSK/D8PSK)
  UINT16 txLevel;     // transmit level
  UINT16 dataPattern; // data pattern (none/ramp/octet)
  UINT16 octet;       // fixed data value
  UINT16 ppduDone;
  UINT16 ppduCnt;
  UINT16 zcDelay;     // zcDelay[14:0]:zero crossing delay in 100 us.  zcDelay[15]: zc delay enable
  UINT16 ppduTime;
  UINT32 toneMap;
}PHY_txTestMode_t;

typedef struct
{
  PHY_txTestMode_t testMode;
  UINT16           type;
}PHY_testMsg_t;

/* Callback functions */
typedef void (*PHY_cbTestMsg_t)(UINT16 type, PHY_txTestMode_t *test_p);

/*****************************************************************************/
/* External variables                                                        */
/*****************************************************************************/
extern const PHY_iot_tst_t  PHY_iot_tst_tbl[PHY_NUM_IOT_TCS];
extern const UINT16 g3_iot_msg[PHY_G3_IOT_MSG_LEN];
extern const UINT16 g3_iot_ramp[PHY_G3_IOT_RAMP_LEN];


/*****************************************************************************/
/* Function Prototypes                                                       */
/*****************************************************************************/
PHY_status_t PHY_rxSetTestMode(PHY_rxTestMode_t *test_p);
PHY_status_t PHY_rxGetTestMode(PHY_rxTestMode_t *test_p);
PHY_status_t PHY_rxGetTestBert(PHY_rxBert_t *bert_p);

PHY_status_t PHY_txSetTestMode(PHY_txTestMode_t *test_p);
PHY_status_t PHY_txGetTestMode(PHY_txTestMode_t *test_p);

PHY_status_t PHY_rxGetTestRssi(SINT16 *rssi_p);

void PHY_txTestRun(void);

#endif // _PHY_TEST_H
