/*
 * i2c_library.c
 *
 *  Created on: 07.03.2018
 *      Author: Dawid Balcerzak
 */
#include "driverlib.h"
#include "device.h"
#include "i2c_library.h"

// Internal functions prototypes - unnecessary to be seen for user
static I2C_Master_t InitializeFunctionPointers(void);
static void I2C_lib_MasterRegAccessReadyHandler(I2C_Master_t *i2c);
static void I2C_lib_MasterWriteHandler(I2C_Master_t *i2c);
static void I2C_lib_MasterReadHandler(I2C_Master_t *i2c);
static void I2C_lib_MasterTransactionFinished(I2C_Master_t *i2c, I2CM_RESULT_t result);
static void I2C_lib_MasterNoAckHandler(I2C_Master_t *i2c);
static void I2C_lib_MasterStopConditionHandler(I2C_Master_t *i2c);
//************************************************************************************

// Internal functions
static I2C_Master_t InitializeFunctionPointers(void)
{
    I2C_Master_t i2c;
    i2c.SendStopCondition = I2C_sendStopCondition;
    i2c.SendStartCondition = I2C_sendStartCondition;
    i2c.SetSlaveAddress = I2C_setSlaveAddress;
    i2c.SetDataCount = I2C_setDataCount;
    i2c.SetConfig = I2C_setConfig;
    i2c.PutData = I2C_putData;
    i2c.GetData = I2C_getData;
    i2c.GetRxFIFOStatus = I2C_getRxFIFOStatus;
    i2c.GetInterruptSource = I2C_getInterruptSource;
    return i2c;
}
static void I2C_lib_MasterRegAccessReadyHandler(I2C_Master_t *i2c)
{
    if(i2c->bytesToRead)
        if(i2c->GetRxFIFOStatus(i2c->I2C_BASE))//if(I2C_getRxFIFOStatus(i2c->I2C_BASE))
            i2c->SendStopCondition(i2c->I2C_BASE);//I2C_sendStopCondition(i2c->I2C_BASE);
        else
        {
            i2c->SetDataCount(i2c->I2C_BASE, i2c->bytesToRead);//I2C_setDataCount(i2c->I2C_BASE, i2c->bytesToRead);
            i2c->SetConfig(i2c->I2C_BASE, I2C_MASTER_RECEIVE_MODE);//I2C_setConfig(i2c->I2C_BASE, I2C_MASTER_RECEIVE_MODE);
            i2c->SendStartCondition(i2c->I2C_BASE);//I2C_sendStartCondition(i2c->I2C_BASE);
        }
    else
        i2c->SendStopCondition(i2c->I2C_BASE);//I2C_sendStopCondition(i2c->I2C_BASE);
}
static void I2C_lib_MasterWriteHandler(I2C_Master_t *i2c)
{
    i2c->PutData(i2c->I2C_BASE, i2c->writeData[i2c->bytesWritten]);//I2C_putData(i2c->I2C_BASE, i2c->writeData[i2c->bytesWritten]);
    ++i2c->bytesWritten;
}
static void I2C_lib_MasterReadHandler(I2C_Master_t *i2c)
{
    i2c->readData[i2c->bytesRead] = i2c->GetData(i2c->I2C_BASE);//I2C_getData(i2c->I2C_BASE);
    i2c->bytesRead++;
}
static void I2C_lib_MasterTransactionFinished(I2C_Master_t *i2c, I2CM_RESULT_t result)
{
    i2c->result = result;
    i2c->status = I2CM_STATUS_READY;
}
static void I2C_lib_MasterNoAckHandler(I2C_Master_t *i2c)
{
    i2c->result = I2CM_RESULT_NACK_RECEIVED;
    i2c->SendStopCondition(i2c->I2C_BASE);//I2C_sendStopCondition(i2c->I2C_BASE);
}
static void I2C_lib_MasterStopConditionHandler(I2C_Master_t *i2c)
{
    if(i2c->result == I2CM_RESULT_UNKNOWN)
        I2C_lib_MasterTransactionFinished(i2c, I2CM_RESULT_OK);
    else
        I2C_lib_MasterTransactionFinished(i2c, i2c->result);
}
/*void TWI_MasterArbitrationLostBusErrorHandler(I2C_Master_t *i2c)
{
    i2c->result = TWIM_RESULT_ARBITRATION_LOST;
    i2c->status = I2CM_STATUS_READY;
    i2c->SendStopCondition(i2c->I2C_BASE);//I2C_sendStopCondition(i2c->I2C_BASE);
}*/
//************************************************************************************

// Global functions
void I2C_lib_init(uint32_t I2C_BASE, I2C_Master_t *i2c)
{
    *i2c = InitializeFunctionPointers();

    // zapisanie w strukturze adresu uywanego moduu I2C
    i2c->I2C_BASE = I2CA_BASE;

    // Must put I2C into reset before configuring it
    I2C_disableModule(I2C_BASE);

    // I2C configuration.
    I2C_initMaster(I2C_BASE, DEVICE_SYSCLK_FREQ, 400000, I2C_DUTYCYCLE_50);
    I2C_setBitCount(I2C_BASE, I2C_BITCOUNT_8);
    I2C_setEmulationMode(I2C_BASE, I2C_EMULATION_FREE_RUN);

    // Enable stop condition and register-access-ready interrupts
    I2C_enableInterrupt(I2C_BASE,   I2C_INT_STOP_CONDITION |
                                    I2C_INT_REG_ACCESS_RDY |
                                    I2C_INT_RX_DATA_RDY |
                                    I2C_INT_TX_DATA_RDY |
                                    I2C_INT_NO_ACK
                                    );

    // FIFO configuration
    I2C_enableFIFO(I2C_BASE);
    I2C_clearInterruptStatus(I2C_BASE, I2C_INT_RXFF | I2C_INT_TXFF);

    // Configuration complete. Enable the module.
    I2C_enableModule(I2C_BASE);
}
// \brief I2C write and/or read transaction.
//  This function is a TWI Master write and/or read transaction. The function
//  can be used to both write and/or read bytes to/from the TWI Slave in one
//  transaction.
//  \param i2c            The I2C_Master_t struct instance.
//  \param address        The slave address.
//  \param writeData      Pointer to data to write.
//  \param bytesToWrite   Number of bytes to write.
//  \param bytesToRead    Number of bytes to read.
void I2C_lib_MasterWriteRead(I2C_Master_t *i2c,
                             uint16_t/*uint8_t*/ address,
                             uint16_t/*uint8_t*/ *writeData,
                         uint16_t bytesToWrite,
                         uint16_t bytesToRead)
{
    // Parameter sanity check.
    if (bytesToWrite > I2CM_WRITE_BUFFER_SIZE)
        I2C_lib_MasterTransactionFinished(i2c, I2CM_RESULT_BUFFER_OVERFLOW);
    if (bytesToRead > I2CM_READ_BUFFER_SIZE)
        I2C_lib_MasterTransactionFinished(i2c, I2CM_RESULT_BUFFER_OVERFLOW);

    // Initiate transaction if bus is ready.
    if (i2c->status == I2CM_STATUS_READY)
    {
        i2c->status = I2CM_STATUS_BUSY;
        i2c->result = I2CM_RESULT_UNKNOWN;

        i2c->address = address;
        i2c->SetSlaveAddress(i2c->I2C_BASE, i2c->address);//I2C_setSlaveAddress(i2c->I2C_BASE, i2c->address);

        // Fill write data buffer
        uint16_t/*uint8_t*/ bufferIndex;
        for (bufferIndex = 0; bufferIndex < bytesToWrite; bufferIndex++)
            i2c->writeData[bufferIndex] = writeData[bufferIndex];

        i2c->bytesToWrite = bytesToWrite;
        i2c->bytesToRead = bytesToRead;
        i2c->bytesWritten = 0;
        i2c->bytesRead = 0;

        // If write command, send the START condition + Address + 'R/_W = 0'
        if (i2c->bytesToWrite > 0)
        {
            i2c->SetDataCount(i2c->I2C_BASE, i2c->bytesToWrite);//I2C_setDataCount(i2c->I2C_BASE, i2c->bytesToWrite);
            i2c->SetConfig(i2c->I2C_BASE, I2C_MASTER_SEND_MODE);//I2C_setConfig(i2c->I2C_BASE, I2C_MASTER_SEND_MODE);
            i2c->SendStartCondition(i2c->I2C_BASE);//I2C_sendStartCondition(i2c->I2C_BASE);
        }

        // If read command, send the START condition + Address + 'R/_W = 1'
        else if (i2c->bytesToRead > 0)
        {
            i2c->SetDataCount(i2c->I2C_BASE, i2c->bytesToRead);//I2C_setDataCount(i2c->I2C_BASE, i2c->bytesToRead);
            i2c->SetConfig(i2c->I2C_BASE, I2C_MASTER_RECEIVE_MODE);//I2C_setConfig(i2c->I2C_BASE, I2C_MASTER_RECEIVE_MODE);
            i2c->SendStartCondition(i2c->I2C_BASE);//I2C_sendStartCondition(i2c->I2C_BASE);
        }
    } else
        ;//I2C_lib_MasterTransactionFinished(i2c, I2CM_RESULT_FAIL);
}
void I2C_lib_InterruptHandler(I2C_Master_t *i2c)
{
    I2C_InterruptSource intSource = i2c->GetInterruptSource(i2c->I2C_BASE);//I2C_getInterruptSource(i2c->I2C_BASE);

    switch(intSource)
    {
        case I2C_INTSRC_TX_DATA_RDY:
            I2C_lib_MasterWriteHandler(i2c);
            break;
        case I2C_INTSRC_RX_DATA_RDY:
            I2C_lib_MasterReadHandler(i2c);
            break;
        case I2C_INTSRC_REG_ACCESS_RDY:
            I2C_lib_MasterRegAccessReadyHandler(i2c);
            break;
        case I2C_INTSRC_NO_ACK:
            I2C_lib_MasterNoAckHandler(i2c);
            break;
        case I2C_INTSRC_STOP_CONDITION:
            I2C_lib_MasterStopConditionHandler(i2c);
            break;
    }

    /*if(intSource == I2C_INTSRC_TX_DATA_RDY)
        I2C_lib_MasterWriteHandler(i2c);
    else if(intSource == I2C_INTSRC_RX_DATA_RDY)
        I2C_lib_MasterReadHandler(i2c);
    else if(intSource == I2C_INTSRC_REG_ACCESS_RDY)
        I2C_lib_MasterRegAccessReadyHandler(i2c);
    else if(intSource == I2C_INTSRC_NO_ACK)
        I2C_lib_MasterNoAckHandler(i2c);
    else if(intSource == I2C_INTSRC_STOP_CONDITION)
        I2C_lib_MasterStopConditionHandler(i2c);*/
}
//************************************************************************************
