//==============================================================================================//
//              AC3PH.H                                                                         //
//=============================================================================================c//

#ifndef AC3PH_H_
#define AC3PH_H_

#include "utils.h"

#include "control.h"
#include "error.h"

//==============================================================================================//
//              MACRO FLAGS                                                                     //
//=============================================================================================c//

//
// PLL IN-LOCK FLAGS
//

#define AC3PH_PLL_INLOCK_NO     0
#define AC3PH_PLL_INLOCK_YES    1

//==============================================================================================//
//              STRUCTURE DECLARATIONS                                                          //
//=============================================================================================c//

// Phase measurements in 3-phase AC system
typedef struct AC3PH_abcn_STRUCT
{
    fp32_t an;
    fp32_t bn;
    fp32_t cn;
} AC3PH_abcn_S;

// Line measurements in 3-phase AC system
typedef struct AC3PH_abc_STRUCT
{
    fp32_t ab;
    fp32_t bc;
    fp32_t ca;
} AC3PH_abc_S;

// Stationary reference frame (SRF)
typedef struct AC3PH_ab_STRUCT
{
    fp32_t alpha;
    fp32_t beta;
} AC3PH_ab_S;

// Rotating reference frame (RRF)
typedef struct AC3PH_dq_STRUCT
{
    fp32_t d;
    fp32_t q;
} AC3PH_dq_S;

// Phase-locked loop algorithm variables
typedef struct AC3PH_pll_STRUCT
{
    fp32_t WN;              // TODO explain what is this
    fp32_t ZETA;            // TODO explain what is this
    fp32_t NORM;            // Normalization coefficient for q component
    fp32_t W_NOM;           // Nominal frequency (rad/s)
    fp32_t Q_MAX;           // q-component limit to detect in-lock
    uint32_t TMR;           // In-lock software timer limit
    fp32_t TS;              // Sample time (s)
    UTILS_angle_S phase;    // Estimated angle
    CTRL_pi_S pi_w_err;     // Frequency error estimator
    fp32_t w_est;           // Estimated frequency
    uint32_t timer;         // PLL in-lock software timer
    uint16_t in_lock;       // PLL in-lock flag
} AC3PH_pll_S;

//==============================================================================================//
//              FORWARD AND INVERSE CLARKE AND PARK TRANSFORMATIONS                             //
//=============================================================================================c//

UTILS_INLINE void AC3PH_fwdClarkeLine(volatile AC3PH_abc_S *abc, volatile AC3PH_ab_S *ab);
UTILS_INLINE void AC3PH_fwdClarkePhase(volatile AC3PH_abcn_S *abcn, volatile AC3PH_ab_S *ab);
UTILS_INLINE void AC3PH_fwdPark(volatile AC3PH_ab_S *ab, volatile AC3PH_dq_S *dq, volatile UTILS_angle_S *theta);
UTILS_INLINE void AC3PH_invClarkeLine(volatile AC3PH_ab_S *ab, volatile AC3PH_abc_S *abc);
UTILS_INLINE void AC3PH_invClarkePhase(volatile AC3PH_ab_S *ab, volatile AC3PH_abcn_S *abcn);
UTILS_INLINE void AC3PH_invPark(volatile AC3PH_dq_S *dq, volatile AC3PH_ab_S *ab, volatile UTILS_angle_S *theta);

UTILS_INLINE void AC3PH_fwdClarkeLine(volatile AC3PH_abc_S *abc, volatile AC3PH_ab_S *ab)
{
    ab->alpha = UTILS_2BY3*abc->ab + UTILS_1BY3*abc->bc;
    ab->beta  = UTILS_SQRT3BY3*abc->bc;
}

UTILS_INLINE void AC3PH_fwdClarkePhase(volatile AC3PH_abcn_S *abcn, volatile AC3PH_ab_S *ab)
{
    ab->alpha = abcn->an;
    ab->beta  = UTILS_SQRT3BY3*(abcn->bn-abcn->cn);
}

UTILS_INLINE void AC3PH_fwdPark(volatile AC3PH_ab_S *ab, volatile AC3PH_dq_S *dq, volatile UTILS_angle_S *theta)
{
    dq->d = +ab->alpha*theta->_cos + ab->beta*theta->_sin;
    dq->q = -ab->alpha*theta->_sin + ab->beta*theta->_cos;
}

UTILS_INLINE void AC3PH_invClarkeLine(volatile AC3PH_ab_S *ab, volatile AC3PH_abc_S *abc)
{
    abc->ab = UTILS_3BY2*ab->alpha - UTILS_SQRT3BY2*ab->beta;
    abc->bc = UTILS_SQRT3*ab->beta;
    abc->ca = -(abc->ab+abc->bc);
}

UTILS_INLINE void AC3PH_invClarkePhase(volatile AC3PH_ab_S *ab, volatile AC3PH_abcn_S *abcn)
{
    abcn->an = ab->alpha;
    abcn->bn = -UTILS_1BY2*ab->alpha + UTILS_SQRT3BY2*ab->beta;
    abcn->cn = -(abcn->an+abcn->bn);
}

UTILS_INLINE void AC3PH_invPark(volatile AC3PH_dq_S *dq, volatile AC3PH_ab_S *ab, volatile UTILS_angle_S *theta)
{
    ab->alpha = dq->d*theta->_cos - dq->q*theta->_sin;
    ab->beta  = dq->d*theta->_sin + dq->q*theta->_cos;
}

//==============================================================================================//
//              PHASE-LOCKED LOOP                                                               //
//=============================================================================================c//

_Bool AC3PH_pllInit(volatile AC3PH_pll_S *pll);
void AC3PH_pllReset(volatile AC3PH_pll_S *pll);
UTILS_INLINE void AC3PH_pllUpdate(fp32_t q, volatile AC3PH_pll_S *pll);

UTILS_INLINE void AC3PH_pllUpdate(fp32_t q, volatile AC3PH_pll_S *pll)
{
    // Normalize q component
    // [should be in -1..+1 interval]
    fp32_t q_norm = q / pll->NORM;

    // Frequency estimation
    pll->w_est = pll->W_NOM + CTRL_piUpdate(q_norm, &pll->pi_w_err);

    // Update phase angle
    // [it is sufficient to use 1st order approximation]
    pll->phase.theta += (pll->w_est * pll->TS);

    // Limit phase angle to 0..2PI interval
    if (pll->phase.theta < 0)           { pll->phase.theta += UTILS_2PI; }
    if (pll->phase.theta > UTILS_2PI)   { pll->phase.theta -= UTILS_2PI; }

    // Reset in-lock variables if q component is not within limits
    if ((q_norm < -pll->Q_MAX) || (q_norm > +pll->Q_MAX))
    {
        pll->timer = 0;
        pll->in_lock = AC3PH_PLL_INLOCK_NO;
    }

    // Update in-lock variables if q component is within limits
    else
    {
        // Reset PLL in-lock flag if in-lock timer is still running
        if (pll->timer < pll->TMR)
        {
            pll->timer++;
            pll->in_lock = AC3PH_PLL_INLOCK_NO;
        }

        // Set PLL in-lock flag if in-lock timer is not running
        else // (pll->timer >= pll->TMR)
        {
            pll->in_lock = AC3PH_PLL_INLOCK_YES;
        }
    }
}

#endif /* AC3PH_H_ */

//==============================================================================================//
//              END OF FILE                                                                     //
//=============================================================================================c//
