//==============================================================================================//
//              BUCKBOOST.C                                                                     //
//=============================================================================================c//

#ifndef buckboost_C_
#define buckboost_C_

#include "buckboost.h"

#include "mcu.h"

// Buck-boost handler
volatile BUB_pack_S *BUB;

//==============================================================================================//
//              ADCA1 AND EPWM2 INTERRUPT FUNCTION DEFINITIONS                                  //
//=============================================================================================c//

// CAUTION: You must update this table if you add new ISR commands
const uint32_t BUB_COMMAND_TABLE[5][3] =
{
    { SYS_RELAY_CONTACTOR_DC_DISCONNECT, SYS_RELAY_DISCHARGE_OUTPUT_OFF, MCU_EPWM_OUTPUT_DISABLE },  // 0 DEFAULT_STATE
    { SYS_RELAY_CONTACTOR_DC_DISCONNECT, SYS_RELAY_DISCHARGE_OUTPUT_ON,  MCU_EPWM_OUTPUT_DISABLE },  // 1 DISCHARGE_START
    { SYS_RELAY_CONTACTOR_DC_DISCONNECT, SYS_RELAY_DISCHARGE_OUTPUT_OFF, MCU_EPWM_OUTPUT_DISABLE },  // 2 DISCHARGE_STOP
    { SYS_RELAY_CONTACTOR_DC_DISCONNECT, SYS_RELAY_DISCHARGE_OUTPUT_OFF, MCU_EPWM_OUTPUT_ENABLE  },  // 3 BUCKBOOST_START
    { SYS_RELAY_CONTACTOR_DC_CONNECT,    SYS_RELAY_DISCHARGE_OUTPUT_OFF, MCU_EPWM_OUTPUT_ENABLE  }   // 4 CONNECT_BATTERY
};

interrupt void BUB_isrAdca1(void)
{
    //
    // GET DIGITAL MEASUREMENTS
    //

    // Inductor (L2) current
    BUB->raw->i_L2  = (fp32_t) AdcaResultRegs.ADCRESULT13;  // A5

    // Battery voltage, and remote sense voltage
    BUB->raw->v_bat = (fp32_t) AdcaResultRegs.ADCRESULT14;  // A2
    BUB->raw->v_rem = (fp32_t) AdcaResultRegs.ADCRESULT15;  // A3

    // Clear EOC trigger flag
    // TODO check if this flag is cleared automatically when new conversion starts!
    AdcaRegs.ADCINTFLGCLR.bit.ADCINT1 = 1;

    //
    // RETURN FROM INTERRUPT SERVICE ROUTINE
    //

    // Reset interrupt flags and return
//  AdcaRegs.ADCINTFLGCLR.bit.ADCINT1 = 1;  // [already cleared in this ISR]
    PieCtrlRegs.PIEACK.all = PIEACK_GROUP1;
    return;
}

interrupt void BUB_isrEPwm2(void)
{
    //
    // GET STATE-MACHINE AND ERROR HANDLERS
    //

    // Pointer to state machine structure
    volatile CTRL_sm_S *sm = &BUB->state_machine;

    // Pointer to error structure
    volatile ERR_pack_S *err = BUB->error;

    // Pointer to stopwatch structure
    volatile SWC_tictoc_S *swc = &BUB->stopwatch;

#ifdef PLECS
    printf("BUCK-BOOST: %2d : %2d\n", sm->state, sm->substate);
#endif //#ifdef PLECS

    //
    // GET DIGITAL MEASUREMENTS
    //

    // Wait until measurements triggered on TBCTR=0x00 of ePWM2 are ready, but do not wait forever
    // as we will stuck in an infinite loop and block resources for other ISRs with lower or same
    // priority level. The ADC flag will be latched on the EOC12 event.
    while (AdcaRegs.ADCINTFLG.bit.ADCINT2 == 0)
    {
        // Do not wait forever, check timeout condition
        if (EPwm2Regs.TBCTR > CONST_TMO_ADC_MEASUREMENTS)
        {
            // Timeout has occurred, raise error flag
            ERR_FIELD_SET(TIMEOUT, ADC_EPWM2);

            // Do not wait for measurements any more
            break;
        }
    }

    // Get measurements from SOC channels if there is no timeout error
    if (ERR_FIELD_GET(TIMEOUT, ADC_EPWM2) == ERR_CLEAR)
    {
        // Inductor (L1) current, and output current
        BUB->raw->i_L1  = (fp32_t) AdcaResultRegs.ADCRESULT10;  // A4
        BUB->raw->i_out = (fp32_t) AdcaResultRegs.ADCRESULT11;  // A1

        // Output voltage
        BUB->raw->v_out = (fp32_t) AdcaResultRegs.ADCRESULT12;  // A0

        // Clear EOC trigger flag
        AdcaRegs.ADCINTFLGCLR.bit.ADCINT2 = 1;
    }

#ifndef CONST_CALIBRATION_BUFFER_SIZE

    // BUCK-BOOST CONTROL ALGORITHM

    // START or STOP the buck-boost state machine.
    // The buck-boost state machine can be started only after inverter reaches the STARTED state.
    sm->command = (BUB->inv_state_machine->state == CTRL_SM_STATE_STARTED) ?
            BUB->usr_command : CTRL_SM_COMMAND_STOP;
    CTRL_smErrorStopStart(sm, ERR_FIELD_GET(CONFIG, GLOBAL));

    // Buck-boost control algorithm
    SWC_start(swc);
    BUB_doControl();
    SWC_stop(swc);

    // Update buck-boost duty cycles
    MCU_epwmSetDutyCycle(SYS_PWM_BUB_L1, BUB->duty->d1);
    MCU_epwmSetDutyCycle(SYS_PWM_BUB_L2, BUB->duty->d2);

    //
    // UPDATE RELAY AND PWM ENABLE STATES
    //

    // Update relay and PWM output-enable states, but only if the ISR command from the
    // state machine has changed since the last interrupt call
    if (BUB->isr_command != BUB_ISR_COMMAND)
    {
        // Update new buck-boost ISR command
        BUB->isr_command = BUB_ISR_COMMAND;

        // Update relay state
        GPIO_WritePin(SYS_RELAY_CONTACTOR_DC,     BUB_COMMAND_TABLE[BUB->isr_command][0]);
        GPIO_WritePin(SYS_RELAY_DISCHARGE_OUTPUT, BUB_COMMAND_TABLE[BUB->isr_command][1]);

        // Enable or disable PWM generators
        MCU_epwmEnableOutputs(SYS_PWM_BUB_L1, BUB_COMMAND_TABLE[BUB->isr_command][2]);
        MCU_epwmEnableOutputs(SYS_PWM_BUB_L2, BUB_COMMAND_TABLE[BUB->isr_command][2]);
    }

#endif //#ifndef CONST_CALIBRATION_BUFFER_SIZE

    //
    // RETURN FROM INTERRUPT SERVICE ROUTINE
    //

    // Reset interrupt flags and return
    EPwm2Regs.ETCLR.bit.INT = 1;
    PieCtrlRegs.PIEACK.all = PIEACK_GROUP3;
}

//==============================================================================================//
//              INITIALIZATION FUNCTION DEFINITION                                              //
//=============================================================================================c//

void BUB_init(volatile BUB_pack_S *BUB_PACK)
{
    // Initialize handler
    BUB = BUB_PACK;

    // Pointer to error structure
    volatile ERR_pack_S *err = BUB->error;

    // Initialize PI controllers
    ERR_FIELD_GET(INIT, PI_V_OUT_TOP) = CTRL_piInit(&BUB->pi_v_out_top);
    ERR_FIELD_GET(INIT, PI_I_OUT)     = CTRL_piInit(&BUB->pi_i_out);
    ERR_FIELD_GET(INIT, PI_V_OUT)     = CTRL_piInit(&BUB->pi_v_out);
    ERR_FIELD_GET(INIT, PI_I_L1)      = CTRL_piInit(&BUB->pi_i_L1);
    ERR_FIELD_GET(INIT, PI_I_L2)      = CTRL_piInit(&BUB->pi_i_L2);

    // Initialize rate-limiter filters
    ERR_FIELD_GET(INIT, RLIM_V_OUT_REF) = DSP_rlimInit(&BUB->rlim_v_out_ref);
    ERR_FIELD_GET(INIT, RLIM_I_OUT_REF) = DSP_rlimInit(&BUB->rlim_i_out_ref);

    // Reset calibration structures
    ERR_FIELD_GET(INIT, STAT_I_L1)  = DSP_statInit(&BUB->stat_i_L1);
    ERR_FIELD_GET(INIT, STAT_I_L2)  = DSP_statInit(&BUB->stat_i_L2);
    ERR_FIELD_GET(INIT, STAT_I_OUT) = DSP_statInit(&BUB->stat_i_out);

    // Initialize stopwatch
    SWC_init(
        &BUB->stopwatch,
        (uint32_t *) &CpuTimer2.RegsAddr->TIM.all,
        (uint32_t) CpuTimer2.RegsAddr->PRD.all,
        (fp32_t) CpuTimer2.PeriodInUSec
    );
}

//==============================================================================================//
//              CONTROL FUNCTION DEFINITION                                                     //
//=============================================================================================c//

void BUB_doControl(void)
{
#ifndef DO_NOT_PROGRAM_CONTORL_FUNCTIONS
    //
    // GET STATE MACHINE AND ERROR HANDLERS
    //

    // Pointer to buck-boost state machine
    volatile CTRL_sm_S *sm = &BUB->state_machine;

    // Pointer to error structure
    volatile ERR_pack_S *err = BUB->error;

    //
    // CALCULATE PHYSICAL MEASUREMENTS
    //

    // Digital measurements are converted here from raw format (0..4095) to their actual physical
    // values. Note that some measurements have an inherent offset, i.e., 2048 is converted to 0
    // for grid and secondary voltages, and for secondary, two inductor and output currents.
    // These measurements need to be calibrated before we can calculate their physical values.

    // Output, battery, and remote sense voltages
    BUB->meas->v_out = UTILS_ADC_NORM(BUB->raw->v_out, CONST_MEAS_V_OUT, 0);
    BUB->meas->v_bat = UTILS_ADC_NORM(BUB->raw->v_bat, CONST_MEAS_V_BAT, 0);
    BUB->meas->v_rem = UTILS_ADC_NORM(BUB->raw->v_rem, CONST_MEAS_V_REM, 0);

    // Scale measurements only after the measurements calibration is done
    if (BUB_CAL_FLAG == CONST_MEAS_CALIBRATED_YES)
    {
        // Inductor currents
        BUB->meas->i_L1  = UTILS_ADC_NORM(BUB->raw->i_L1,  CONST_MEAS_I_L1,  BUB->stat_i_L1.mean );
        BUB->meas->i_L2  = UTILS_ADC_NORM(BUB->raw->i_L2,  CONST_MEAS_I_L2,  BUB->stat_i_L2.mean );

        // Output current
        BUB->meas->i_out = UTILS_ADC_NORM(BUB->raw->i_out, CONST_MEAS_I_OUT, BUB->stat_i_out.mean);
    }

    //
    // SOFTWARE PROTECTION
    //

    // TODO comment
    // TODO think about including additional protection signals (state, substate, ISR command etc.)

    // Output overvoltage protection
    if (BUB->meas->v_out > CONST_PROT_V_OUT_HIGH)
    {
        ERR_FIELD_SET(MEAS_HIGH, V_OUT);
    }

    // Battery overvoltage protection
    if (BUB->meas->v_bat > CONST_PROT_V_BAT_HIGH)
    {
        ERR_FIELD_SET(MEAS_HIGH, V_BAT);
    }

    // Remote-sense overvoltage protection
    if (BUB->meas->v_rem > CONST_PROT_V_REM_HIGH)
    {
        ERR_FIELD_SET(MEAS_HIGH, V_REM);
    }

    //
    // STATE MACHINE
    //

    // TODO ADD COMMENTS FOR STATE MACHINE

    // STATE MACHINE
    switch (sm->state)
    {
        // State machine is STOPPED,
        // immediately break the control algorithm.
        case CTRL_SM_STATE_STOPPED :
            return;

        // State machine is STARTED,
        // copy the user-defined references, and proceed with the control algorithm.
        case CTRL_SM_STATE_STARTED :
            // Set output voltage and current references
            BUB->v_out_ref = BUB->v_out_usr;
            BUB->i_out_ref = BUB->i_out_usr;

            break;

        // Reset buck-boost to default state
        case CTRL_SM_STATE_RESET :
            switch (sm->substate)
            {
                // RESET TO DEFAULT STATE
                case 0 :
                    // Set default state for buck-boost relays and PWM generators
                    BUB_ISR_COMMAND = BUB_ISR_COMMAND_DEFAULT_STATE;

                    // Reset measurements that need to be calibrated
                    BUB->meas->i_L1  = 0.0f;
                    BUB->meas->i_L2  = 0.0f;
                    BUB->meas->i_out = 0.0f;

                    // Reset buck-boost duty cycles
                    BUB->duty->d1 = 0.0f;
                    BUB->duty->d2 = 0.0f;

                    // Reset measurements calibration flag
                    BUB_CAL_FLAG = CONST_MEAS_CALIBRATED_NO;

                    // There is no CRITICAL ERROR in the system,
                    // state machine can start with the DC link discharge routine,
                    // and after that go to STARTING or STOPPING procedure.
                    if (ERR_FIELD_GET(CONFIG, CRITICAL) == ERR_CLEAR)
                    {
                        CTRL_smGotoState(sm, BUB_SM_STATE_DISCHARGE_OUTPUT);
                        return;
                    }

                    // There is CRITICAL ERROR in the system,
                    // state machine should be STOPPED without going into the discharge routine.
                    else
                    {
                        CTRL_smGotoState(sm, CTRL_SM_STATE_STOPPED);
                        return;
                    }

                // UNEXPECTED SUBSTATE
                default :
                    ERR_FIELD_SET(CONFIG, UNEXPECTED_SUBSTATE);
                    return;
            }

        // When PWM generators are enabled, duty cycle is set to 0%, which means that high-side
        // transistors are OFF, and low-side transistors are ON. In that case, output capacitors
        // are short-circuited via inductors and low-side transistors, and that is why the output
        // capacitors need to be completely discharged before enabling buck-boost PWM generators.
        case BUB_SM_STATE_DISCHARGE_OUTPUT :
            switch (sm->substate)
            {
                // START OR SKIP DISCHARGE ROUTINE
                // If output capacitors are already discharged, skip the whole discharge routine.
                // In this way we will avoid making sounds that occur on the relay transition.
                case 0 :
                    // Skip the discharge routine if output is already discharged
                    if (BUB->meas->v_out < CONST_BUB_DISCHARGE_V_OUT)
                    {
                        CTRL_smGotoState(sm, CTRL_SM_STATE_START_OR_STOP);
                        return;
                    }

                    // Start the discharge routine
                    else // (BUB->meas->v_out >= CONST_BUB_DISCHARGE_V_OUT)
                    {
                        // Connect the output discharge resistor
                        BUB_ISR_COMMAND = BUB_ISR_COMMAND_DISCHARGE_START;

                        CTRL_smGotoNextSubstate(sm);
                        return;
                    }

                // WAIT UNTIL OUTPUT IS DISCHARGED
                case 1 :
                    CTRL_SM_GOTO_NEXT_SUBSTATE_CONDITION(
                        BUB->meas->v_out < CONST_BUB_DISCHARGE_V_OUT,
                        CONST_TMO_BUB_DISCHARGE_OUTPUT,
                        TIMEOUT, DISCHARGE_OUTPUT
                    );

                    // Set critical error flag if output cannot be discharged. Output discharge
                    // resistor will be disconnected in the CTRL_SM_STATE_RESET state.
                    if (ERR_FIELD_GET(TIMEOUT, DISCHARGE_OUTPUT) == ERR_SET)
                    {
                        ERR_FIELD_SET(CONFIG, CRITICAL);
                    }

                    return;

                // STOP THE DISCHARGE ROUTINE
                case 2 :
                    // Disconnect the output discharge resistor
                    BUB_ISR_COMMAND = BUB_ISR_COMMAND_DISCHARGE_STOP;

                    CTRL_smGotoNextSubstate(sm);
                    return;

                // WAIT UNTIL DISCHARGE RESISTOR DISCONNECTS
                case 3 :
                    CTRL_smGotoNextSubstateDelay(sm, CONST_TMR_BUB_RELAY);
                    return;

                // DISCHARGE ROUTINE IS DONE
                case 4 :
                    CTRL_smGotoState(sm, CTRL_SM_STATE_START_OR_STOP);
                    return;

                // UNEXPECTED SUBSTATE
                default :
                    ERR_FIELD_SET(CONFIG, UNEXPECTED_SUBSTATE);
                    return;
            }

        // START or STOP the state machine
        case CTRL_SM_STATE_START_OR_STOP :
            switch (sm->substate)
            {
                // START OR STOP THE STATE MACHINE
                case 0 :
                    // State machine is in the STARTING mode,
                    // we need to calibrate measurements before we can start with the soft-start.
                    if (sm->mode == CTRL_SM_MODE_STARTING)
                    {
                        CTRL_smGotoState(sm, BUB_SM_STATE_CALIBRATION);
                        return;
                    }

                    // State machine is in the STOPPING mode,
                    // go straight to the STOPPED state.
                    else // CTRL_SM_MODE_STOPPING
                    {
                        CTRL_smGotoState(sm, CTRL_SM_STATE_STOPPED);
                        return;
                    }

                // UNEXPECTED SUBSTATE
                default :
                    ERR_FIELD_SET(CONFIG, UNEXPECTED_SUBSTATE);
                    return;
            }

        // Current measurements can be calibrated only while buck-boost is not in operation, i.e.,
        // while all currents are 0 A.
        case BUB_SM_STATE_CALIBRATION :
            switch (sm->substate)
            {
                // RESET CALIBRATION STRUCTURES
                case 0 :
                    // Reset calibration structures
                    //  Inductor currents
                    DSP_statReset(&BUB->stat_i_L1);
                    DSP_statReset(&BUB->stat_i_L2);

                    // Reset calibration structure
                    //  Output current
                    DSP_statReset(&BUB->stat_i_out);

                    CTRL_smGotoNextSubstate(sm);
                    return;

                // CALIBRATE MEASUREMENTS
                case 1 :
                    // Update statistics
                    //  Inductor currents
                    DSP_statUpdate(0, (int16_t) BUB->raw->i_L1,  &BUB->stat_i_L1 );
                    DSP_statUpdate(0, (int16_t) BUB->raw->i_L2,  &BUB->stat_i_L2 );

                    // Update statistics
                    //  Output current
                    DSP_statUpdate(0, (int16_t) BUB->raw->i_out, &BUB->stat_i_out);

                    // Ensure that we collect enough calibration samples
                    CTRL_smGotoNextSubstateDelay(sm, CONST_TMR_BUB_CALIBRATION);
                    return;

                // CHECK MEASUREMENT OFFSET
                case 2 :
                    // Check offset
                    //  Inductor (L1) current
                    ERR_CHECK_OFFSET(BUB->stat_i_L1.mean,  CONST_PROT_OFFSET, I_L1 );
                    ERR_CHECK_OFFSET(BUB->stat_i_L2.mean,  CONST_PROT_OFFSET, I_L2 );

                    // Check offset
                    //  Output current
                    ERR_CHECK_OFFSET(BUB->stat_i_out.mean, CONST_PROT_OFFSET, I_OUT);

                    CTRL_smGotoNextSubstate(sm);
                    return;

                // CHECK RMS VALUE
                case 3 :
                    // We do not need to check RMS value for current measurements
                    CTRL_smGotoNextSubstate(sm);
                    return;

                // CALIBRATION IS DONE
                case 4 :
                    // All measurements have been successfully calibrated,
                    // set the measurements calibration flag
                    BUB_CAL_FLAG = CONST_MEAS_CALIBRATED_YES;

                    CTRL_smGotoState(sm, BUB_SM_STATE_SOFT_START);
                    return;

                // UNEXPECTED SUBSTATE
                default :
                    ERR_FIELD_SET(CONFIG, UNEXPECTED_SUBSTATE);
                    return;
            }

        // TODO EXPLAIN WHAT IS THIS STATE
        case BUB_SM_STATE_SOFT_START :
            switch (sm->substate)
            {
                // RESET CONTROL VARIABLES
                case 0 :
                    // Reset all PI controllers
                    CTRL_piReset(&BUB->pi_v_out_top);
                    CTRL_piReset(&BUB->pi_i_out);
                    CTRL_piReset(&BUB->pi_v_out);
                    CTRL_piReset(&BUB->pi_i_L1);
                    CTRL_piReset(&BUB->pi_i_L2);

                    // Reset rate limiters
                    DSP_rlimReset(&BUB->rlim_v_out_ref);
                    DSP_rlimReset(&BUB->rlim_i_out_ref);

                    // Reset automatic-startup software timer
                    BUB->auto_timer = 0;

                    // Reset buck-boost duty cycles
                    BUB->duty->d1 = 0.0f;
                    BUB->duty->d2 = 0.0f;

                    // Enable buck-boost PWM generators. Note that the output capacitors must be
                    // completely discharged before enabling buck-boost PWM generators, as the
                    // low transistors will conduct since duty cycle is set to 0%, i.e., output
                    // capacitors are short-circuited via inductors and low-side transistors.
                    BUB_ISR_COMMAND = BUB_ISR_COMMAND_BUCKBOOST_START;

                    CTRL_smGotoNextSubstate(sm);
                    return;

                // AUTOMATIC STARTUP PROCEDURE
                // Buck-boost will try to raise the voltage on the output capacitor bank to match
                // the battery voltage. The DC contactor is connected only when the output voltage
                // is equal to the battery voltage over some (predefined) time period.
                case 1 :
                    // Track the battery voltage
                    BUB->v_out_ref = BUB->meas->v_bat;
                    BUB->i_out_ref = CONST_BUB_SOFT_START_I_OUT;

                    // Calculate output voltage tracking error
                    BUB->auto_err = BUB->v_out_ref - BUB->meas->v_out;

                    // Reset or update automatic-startup software timer
                    BUB->auto_timer = ( (BUB->auto_err < -CONST_BUB_SOFT_START_V_ERR) ||
                                        (BUB->auto_err > +CONST_BUB_SOFT_START_V_ERR) ) ?
                                    0 : (BUB->auto_timer+1);

                    // Wait until the automatic startup is done
                    CTRL_SM_GOTO_NEXT_SUBSTATE_CONDITION(
                        BUB->auto_timer >= CONST_TMR_BUB_SOFT_START,
                        CONST_TMO_BUB_SOFT_START,
                        TIMEOUT, SOFT_START
                    );
                    break;

                // CONNECT THE DC CONTACTOR
                case 2 :
                    // Automatic startup is done, we can connect the battery
                    BUB_ISR_COMMAND = BUB_ISR_COMMAND_CONNECT_BATTERY;

                    // TODO should we reset the references? (due to rate limiters)
                    // TODO should we set the current reference to 0 A?

                    CTRL_smGotoNextSubstate(sm);
                    break;

                // WAIT FOR DC CONTACTOR TO CONNECT
                case 3 :
                    CTRL_smGotoNextSubstateDelay(sm, CONST_TMR_BUB_CONTACTOR);
                    break;

                // AUTOMATIC STARTUP IS DONE
                case 4 :
                    CTRL_smGotoState(sm, CTRL_SM_STATE_STARTED);
                    break;

                // UNEXPECTED SUBSTATE
                default :
                    ERR_FIELD_SET(CONFIG, UNEXPECTED_SUBSTATE);
                    return;
            }

            break;

        // UNEXPECTED STATE
        default :
            ERR_FIELD_SET(CONFIG, UNEXPECTED_STATE);
            return;
    }

    //
    // RATE LIMITER ON VOLTAGE AND CURRENT REFERENCES
    //

    // Apply rate limiter on the output voltage reference
    BUB->v_out_ref_rlim = DSP_rlimUpdate(BUB->v_out_ref, &BUB->rlim_v_out_ref);

    // Apply rate limiter on the output current reference
    BUB->i_out_ref_rlim = DSP_rlimUpdate(BUB->i_out_ref, &BUB->rlim_i_out_ref);

    //
    // SATURATION LIMITS FOR THE TOP LEVEL OUTPUT VOLTAGE CONTROLLER
    //

    // Current reference is positive (battery charge)
    if (BUB->i_out_ref_rlim >= 0)
    {
        BUB->pi_v_out_top.ULO = 0.0f;
        BUB->pi_v_out_top.UHI = BUB->i_out_ref_rlim;
    }

    // Current reference is negative (battery discharge)
    else // (i_out_ref_rlim < 0)
    {
        BUB->pi_v_out_top.ULO = BUB->i_out_ref_rlim;
        BUB->pi_v_out_top.UHI = 0.0f;
    }

    // Automatic startup current can also be negative
    // TODO explain why we set this!
    if (sm->state == BUB_SM_STATE_SOFT_START)
    {
        BUB->pi_v_out_top.ULO = -BUB->i_out_ref_rlim;
    }

    //
    // CONTROL ROUTINE
    //

    // Output voltage controller (top level)
    fp32_t i_out_pi = CTRL_piUpdate((BUB->v_out_ref_rlim - BUB->meas->v_out), &BUB->pi_v_out_top);

    // Output current controller
    fp32_t v_out_pi = CTRL_piUpdate((i_out_pi - BUB->meas->i_out), &BUB->pi_i_out);

    // Output voltage controller
    fp32_t i_L_pi = CTRL_piUpdate((v_out_pi - BUB->meas->v_out), &BUB->pi_v_out);

    // Inductor current controller
    BUB->duty->d1 = CTRL_piUpdate(i_L_pi - BUB->meas->i_L1, &BUB->pi_i_L1);
    BUB->duty->d2 = CTRL_piUpdate(i_L_pi - BUB->meas->i_L2, &BUB->pi_i_L2);
#endif
}

#endif /* BUCKBOOST_C_ */

//==============================================================================================//
//              END OF FILE                                                                     //
//=============================================================================================c//
