//==============================================================================================//
//              BUCKBOOST.H                                                                     //
//=============================================================================================c//

#ifndef BUCKBOOST_H_
#define BUCKBOOST_H_

#include "utils.h"

#include "system.h"
#include "control.h"
#include "dsp.h"
#include "error.h"

#include "constants.h"

#include "stopwatch.h"


//==============================================================================================//
//              MACRO FLAGS                                                                     //
//=============================================================================================c//

// These macro constants are used not to accidentally mix state machine flags
#define BUB_ISR_COMMAND     sm->flag_1
#define BUB_CAL_FLAG        sm->flag_2

//
// BUCK-BOOST STATE-MACHINE STATE FLAGS
//

// It must be noted that lower 3 bytes for the state-machine state flags are reserved, which means
// that state flags should start from at least 8. There are 4 predefined state-machine state flags
// (see control.h for more details):
//      CTRL_SM_STATE_RESET             0
//      CTRL_SM_STATE_START_OR_STOP     1
//      CTRL_SM_STATE_STARTED           2
//      CTRL_SM_STATE_STOPPED           3

#define BUB_SM_STATE_DISCHARGE_OUTPUT               8
#define BUB_SM_STATE_CALIBRATION                    9
#define BUB_SM_STATE_SOFT_START                     10

//
// BUCK-BOOST COMMANDS TO ISR
//

// CAUTION: Update BUB_COMMAND_TABLE if you add new commands!

#define BUB_ISR_COMMAND_DEFAULT_STATE               0
#define BUB_ISR_COMMAND_DISCHARGE_START             1
#define BUB_ISR_COMMAND_DISCHARGE_STOP              2
#define BUB_ISR_COMMAND_BUCKBOOST_START             3
#define BUB_ISR_COMMAND_CONNECT_BATTERY             4

//==============================================================================================//
//              HANDLER DECLARATION                                                             //
//=============================================================================================c//

typedef struct BUB_pack_STRUCT
{
    // Buck-boost state machine
    CTRL_sm_S state_machine;

    // State-machine commands interface
    uint32_t usr_command;           // User command for the state machine
    uint32_t isr_command;           // ISR command from the state machine

    // Output voltage reference
    fp32_t v_out_usr;               // User-defined voltage command (V)
    fp32_t v_out_ref;               // Output voltage command (V)
    fp32_t v_out_ref_rlim;          // Rate-limited v_out_ref variable

    // Output current reference
    fp32_t i_out_usr;               // User-defined current command (A)
    fp32_t i_out_ref;               // Output current command (A)
    fp32_t i_out_ref_rlim;          // Rate-limited i_out_ref variable

    // Automatic startup variables
    fp32_t auto_err;                // Battery voltage tracking error
    uint32_t auto_timer;            // Software timer

    // Digital PI controllers
    CTRL_pi_S pi_v_out_top;         // Output voltage PI controller (top level)
    CTRL_pi_S pi_i_out;             // Output current PI controller
    CTRL_pi_S pi_v_out;             // Output voltage PI controller
    CTRL_pi_S pi_i_L1;              // Inductor (L1) current PI controller
    CTRL_pi_S pi_i_L2;              // Inductor (L2) current PI controller

    // Rate-limiter filters
    DSP_rlim_S rlim_v_out_ref;
    DSP_rlim_S rlim_i_out_ref;

    // Measurements calibration structures
    DSP_stat_S stat_i_L1;
    DSP_stat_S stat_i_L2;
    DSP_stat_S stat_i_out;

    // Stopwatch to measure time
    SWC_tictoc_S stopwatch;

    // POINTERS TO OTHER STRUCTURES

    // Pointer to measurement structures
    volatile SYS_meas_S *raw;       // Structure with raw measurements
    volatile SYS_meas_S *meas;      // Structure with physical values

    // Pointer to duty-cycle structure
    volatile SYS_pwm2_S *duty;

    // Pointer to error structure
    volatile ERR_pack_S *error;

    // Pointer to inverter state machine
    volatile CTRL_sm_S *inv_state_machine;

} BUB_pack_S;

//==============================================================================================//
//              FUNCTION DECLARATIONS                                                           //
//=============================================================================================c//

interrupt void BUB_isrAdca1(void);
interrupt void BUB_isrEPwm2(void);

void BUB_init(volatile BUB_pack_S *BUB_PACK);

void BUB_doControl(void);

#endif /* BUCKBOOST_H_ */

//==============================================================================================//
//              END OF FILE                                                                     //
//=============================================================================================c//
