//==============================================================================================//
//              CONSTANTS.H                                                                     //
//=============================================================================================c//

#ifndef CONSTANTS_H_
#define CONSTANTS_H_

#include "utils.h"

// This macro will remove INV_doControl and BUB_doControl functions in inveter.c and buckboost.c
// files, so that the total program size (.text) does not exceed 0x2000. If program size exceeds
// 0x2000, for some reason debug mode does not work anymore.
#define DO_NOT_PROGRAM_CONTORL_FUNCTIONS

//==============================================================================================//
//              MEASUREMENTS CALIBRATION                                                        //
//=============================================================================================c//

// Measurements calibration buffer size. If this macro is defined, only ADC measurements will be
// enabled, i.e., inverter and buck-boost control functions are never going to be triggered, but
// the measurements will be fetched from ADC cores as they finish with conversions. Calibration
// routine is implemented in the inverter controller interrupt service routine (ePWM4 ISR).
// Be careful with the calibration buffer size, as too large number (i.e., too large buffer) could
// occupy entire RAM. However, computational time does not depend on the buffer size, and it takes
// around 1.5us to recalculate mean and standard deviation once new measurement becomes available.
// The calibration should be done in debug mode, in order to be able to change the calibration
// channel in the calibration structure without re-programming the device.
// Comment out this macro to disable calibration routine, i.e., to enable controllers.
//#define CONST_CALIBRATION_BUFFER_SIZE   (400)

//==============================================================================================//
//              SYSTEM CONSTANTS                                                                //
//=============================================================================================c//

// System frequency
#define CONST_SYSTEM_FREQ_HZ        ((fp32_t) 200.e6)

// Task scheduler clock period
#define CONST_SCHEDULER_CLOCK_US    ((fp32_t) 10.e3)

// Maximum value of the CPU tick counter in the task scheduler
#define CONST_CPU_TICK_MAX          ((uint32_t) 10000)

// ADC sample window duration given as a number of system clock ticks
// [1 tick = 5ns at 200MHz -> 20 ticks = 100ns]
#define CONST_ADC_SAMPLE_WINDOW     ((uint32_t) 20)

// UART communication speed in bits per second
#define CONST_SCI_BAUDRATE_BPS      ((fp32_t) 9600)

//==============================================================================================//
//              INVERTER CONSTANTS                                                              //
//=============================================================================================c//

// Inverter controller sample time
#define CONST_INV_TS                ((fp32_t) 50.e-6)   // (s)

// Inverter PWM frequency and dead-time
#define CONST_INV_FREQUENCY_HZ      ((fp32_t) 20.e3)    // (Hz)
#define CONST_INV_DEADTIME_US       ((fp32_t) 0.4)      // (us)

// Nominal DC link voltage level
#define CONST_INV_V_DC_NOM          ((fp32_t) 160)      // (V)

// Equivalent inductance in dq frame is used for feed-forward control to compensate the dq-frame
// model nonlinearity. This inductance is related to the inductor between transformer and DC link.
#define CONST_INV_L_DQ              ((fp32_t) 1.e-3)    // (H)

// Detected voltage sector will not be accepted until measurements of all three voltages are
// outside of the (-V_EPS,+V_EPS) range, in order not to have a wrong sector reading due to the
// measurements noise.
#define CONST_INV_V_EPS             ((fp32_t) 5)        // (V)

// Voltage level at which DC link is considered to be completely discharged. If you set too low
// voltage level, it might take too long to reach that discharge level.
#define CONST_INV_DISCHARGE_V_DC    ((fp32_t) 2)        // (V)

// If DC link cannot reach this voltage level during passive precharge, then something is wrong
// with the AC grid. Passive precharge is when DC link is directly connected to the AC grid via
// precharge resistors and bridge diodes, i.e., there is no PI control involved.
// We end the PRECHARGE_PASSIVE_I state when this voltage level is reached, and continue with
// CALIBRATION_II state, during which the secondary (line) voltage measurements are calibrated.
// Once calibration routine is done, we will have exact information to which voltage level must
// DC link be charged before disconnecting precharge resistors.
#define CONST_INV_PASSIVE_I_V_DC    ((fp32_t) 110)      // (V)

// Minimum DC link voltage at which we can disconnect the precharge resistors given relative to
// the secondary (line) voltage amplitude. E.g., 0.95 means that DC link needs to reach at least
// 95% of the secondary (line) voltage amplitude before precharge resistors can be disconnected.
// Voltage amplitude is estimated via PLL as d component in the dq frame. It must be noted that
// if the precharge resistors are disconnected too soon, there will be large voltage difference
// between the secondary voltage and the DC link voltage, which will cause a large current to
// flow to the DC link through the bridge diodes, which could permanently damage the diodes.
#define CONST_INV_PASSIVE_II_V_DC   ((fp32_t) 0.95)     // (0..1)

// Active DC link precharge is when we slowly ramp-up the DC link voltage to its nominal voltage
// level by using PI controllers, i.e., by controlling the power transistors. Ramp-up time is
// determined by the precharge current, i.e., ramp-up is faster for higher precharge currents.
#define CONST_INV_ACTIVE_I_D        ((fp32_t) 2)        // (A)

//==============================================================================================//
//              BUCK-BOOST CONSTANTS                                                            //
//=============================================================================================c//

// Buck-boost controller sample time
#define CONST_BUB_TS                ((fp32_t) 40.e-6)   // (s)

// Buck-boost PWM frequency and dead-time
#define CONST_BUB_FREQUENCY_HZ      ((fp32_t) 25.e3)    // (Hz)
#define CONST_BUB_DEADTIME_US       ((fp32_t) 0.4)      // (us)

// Voltage level at which output is considered to be completely discharged. If you set too low
// voltage level, it might take too long to reach that discharge level.
#define CONST_BUB_DISCHARGE_V_OUT   ((fp32_t) 0.5)      // (V)

// Automatic startup current to control the output voltage to the battery voltage level
#define CONST_BUB_SOFT_START_I_OUT  ((fp32_t) 5)        // (A)

// Maximum allowed battery voltage tracking error during automatic startup
#define CONST_BUB_SOFT_START_V_ERR  ((fp32_t) 0.5)      // (V)

//==============================================================================================//
//              PI CONTROLLERS, FILTERS ETC.                                                    //
//=============================================================================================c//

//
// INVERTER PI CONTROLLERS
//

// DC link voltage PI controller
// [outputs command for PI_I_INV_D]
#define CONST_PI_V_DC_KP            ((fp32_t) 7.2)
#define CONST_PI_V_DC_TI            ((fp32_t) 6.4e-3)
#define CONST_PI_V_DC_ULO           ((fp32_t) -50.0)    // (A)
#define CONST_PI_V_DC_UHI           ((fp32_t) +50.0)    // (A)

// Inverter current PI controllers
// [outputs Vdq command]
#define CONST_PI_I_INV_DQ_KP        ((fp32_t) 3.98)
#define CONST_PI_I_INV_DQ_TI        ((fp32_t) 10.e-3)
#define CONST_PI_I_INV_DQ_ULO       ((fp32_t) -25.0)    // (V)
#define CONST_PI_I_INV_DQ_UHI       ((fp32_t) +25.0)    // (V)

//
// BUCK-BOOST PI CONTROLLERS
//

// Output voltage (top level) PI controller
// [outputs command for PI_I_OUT]
#define CONST_PI_V_OUT_TOP_KP       ((fp32_t) 4.0)
#define CONST_PI_V_OUT_TOP_TI       ((fp32_t) 10.e-3)
// ULO and UHI are dynamically changed

// Output current PI controller
// [outputs command for PI_V_OUT]
#define CONST_PI_I_OUT_KP           ((fp32_t) 0.1)
#define CONST_PI_I_OUT_TI           ((fp32_t) 2.e-3)
#define CONST_PI_I_OUT_ULO          ((fp32_t) 0.0)      // (V)
#define CONST_PI_I_OUT_UHI          ((fp32_t) 115.0)    // (V)

// Output voltage PI controller
// [outputs command for two PI_I_L controllers]
#define CONST_PI_V_OUT_KP           ((fp32_t) 5.1786)
#define CONST_PI_V_OUT_TI           ((fp32_t) 2.e-3)
#define CONST_PI_V_OUT_ULO          ((fp32_t) -25.0)    // (A)
#define CONST_PI_V_OUT_UHI          ((fp32_t) +25.0)    // (A)

// Inductor current PI controller
// [outputs duty cycle]
#define CONST_PI_I_L_KP             ((fp32_t) 8.3245e-3)
#define CONST_PI_I_L_TI             ((fp32_t) 0.5e-3)
#define CONST_PI_I_L_ULO            ((fp32_t) 0.0)      // (-)
#define CONST_PI_I_L_UHI            ((fp32_t) 0.95)     // (-)

//
// INVERTER FILTERS
//

// In inverter controller we filter 4 variables:
//  - d and q components of the grid voltage, and
//  - d and q components of the secondary voltage.
// We will use moving-average or second-order Butterworth filter. If CONST_MAV_BUFFER_SIZE macro
// is defined, the moving-average filters are used, otherwise, the Butterworth filters are used.
// Note that the memory requirements are substantial for the moving-average filters, and that is
// why we do not allow moving-average filters in case of MEASUREMENTS CALIBRATION program mode.

// Fundamental moving-average filter frequency is calculated as the sampling frequency divided by
// the number of samples in circular buffer. E.g., for 20 kHz sampling frequency and buffer size
// of 400, the fundamental filter frequency is 50 Hz. The moving-average filter will filter-out
// all frequencies that are whole multiplier of the fundamental frequency.
// We do not allow moving-average filters if MCU is programmed for MEASUREMENTS CALIBRATION, in
// order to save some RAM.
#ifndef CONST_CALIBRATION_BUFFER_SIZE
#define CONST_MAV_BUFFER_SIZE       (400)
#endif //#ifndef CONST_CALIBRATION_BUFFER_SIZE

// Second-order Butterworth filter
#define CONST_BW2_K                 ((fp32_t) 1.0)
#define CONST_BW2_WC                ((fp32_t) UTILS_2PI*50.0)

//
// BUCK-BOOST FILTERS
//

// Output voltage and current reference rate-limiter filters
#define CONST_RLIM_V_OUT_REF        ((fp32_t) 110)      // (V/s)
#define CONST_RLIM_I_OUT_REF        ((fp32_t)  50)      // (A/s)

//
// PLL ALGORITHM
//

// PLL closed-loop system parameters are used to calculate PI controller parameters
#define CONST_PLL_WN                ((fp32_t) UTILS_2PI*6.5)
#define CONST_PLL_ZETA              ((fp32_t) UTILS_SQRT2BY2)
#define CONST_PLL_W_NOM             ((fp32_t) UTILS_2PI*50.0)
#define CONST_PLL_Q_MAX             ((fp32_t) 0.05)

// PLL uses PI controller to estimate frequency error, whereas the controller input is q component
// from Park transformation. Note that PI controller gain depends on the voltage amplitude, and
// that is why we use normalized PI controller parameters, i.e., we assume that the q component is
// in range (-1.0,+1.0). These parameters define the normalization coefficients for q component of
// the grid (phase) and secondary (phase) voltages. See inverter.c file for explanation why is PLL
// used for secondary (phase) voltage, instead of secondary (line) voltage.
// [230*sqrt(2)=325,27 ; (230*sqrt(2))*(70/172)/sqrt(3)=76,43]
#define CONST_PLL_NORM_GRID         ((fp32_t) 325)      // (V)
#define CONST_PLL_NORM_SEC          ((fp32_t)  76)      // (V)

//==============================================================================================//
//              MEASUREMENTS TRANSFORMATION CONSTANTS                                           //
//=============================================================================================c//

// TODO calibrate these measurement constants

// Measurements calibration macro flags
#define CONST_MEAS_CALIBRATED_NO        0
#define CONST_MEAS_CALIBRATED_YES       1

// Grid (phase) voltages
#define CONST_MEAS_V_GRID_AN    ((fp32_t) (391.14 / 2048.0))    // (V/-)
#define CONST_MEAS_V_GRID_BN    ((fp32_t) (391.14 / 2048.0))    // (V/-)
#define CONST_MEAS_V_GRID_CN    ((fp32_t) (391.14 / 2048.0))    // (V/-)

// Secondary (line) voltages
#define CONST_MEAS_V_SEC_AB     ((fp32_t) (195.30 / 2048.0))    // (V/-)
#define CONST_MEAS_V_SEC_BC     ((fp32_t) (195.30 / 2048.0))    // (V/-)
#define CONST_MEAS_V_SEC_CA     ((fp32_t) (195.30 / 2048.0))    // (V/-)

// Secondary (phase) currents
#define CONST_MEAS_I_SEC_AN     ((fp32_t) ( 64.32 / 2048.0))    // (A/-)
#define CONST_MEAS_I_SEC_BN     ((fp32_t) ( 64.32 / 2048.0))    // (A/-)
#define CONST_MEAS_I_SEC_CN     ((fp32_t) ( 64.32 / 2048.0))    // (A/-)

// DC voltages
#define CONST_MEAS_V_DC         ((fp32_t) (180.00 / 4096.0))    // (V/-)
#define CONST_MEAS_V_OUT        ((fp32_t) (134.03 / 4096.0))    // (V/-)
#define CONST_MEAS_V_BAT        ((fp32_t) (134.03 / 4096.0))    // (V/-)
#define CONST_MEAS_V_REM        ((fp32_t) (134.03 / 4096.0))    // (V/-)

// DC currents
#define CONST_MEAS_I_L1         ((fp32_t) ( 32.17 / 2048.0))    // (A/-)
#define CONST_MEAS_I_L2         ((fp32_t) ( 32.17 / 2048.0))    // (A/-)
#define CONST_MEAS_I_OUT        ((fp32_t) ( 64.32 / 2048.0))    // (A/-)

//==============================================================================================//
//              SOFTWARE TIMER AND TIMEOUT CONSTANTS                                            //
//=============================================================================================c//

// These constants are used as a base to calculate software timer constants for any given time.
// Constants are given as a number of ticks within 1ms for specified time base, e.g., for 40us
// time base there are 25 ticks within 1ms.
#define CONST_TMR_40US_1MS      25UL    // 25 ticks within 1ms
#define CONST_TMR_50US_1MS      20UL    // 20 ticks within 1ms

// Calculate software timer constant for a given time and time base
#define CONST_CALC_TMR_MS(TIME_MS, TIME_BASE)           \
    ((uint32_t) TIME_MS * CONST_TMR_##TIME_BASE##_1MS)

//
// INVERTER TIMER AND TIMEOUT CONSTANTS
//

// TODO explain why CONST_TMR_INV_CALIBRATION should be exactly one sine wave period

#define CONST_TMR_INV_SETTLE                    CONST_CALC_TMR_MS( 100UL, 50US)
#define CONST_TMR_INV_RELAY                     CONST_CALC_TMR_MS(  10UL, 50US)
#define CONST_TMR_INV_CONTACTOR                 CONST_CALC_TMR_MS(  50UL, 50US)
#define CONST_TMR_INV_CALIBRATION               CONST_CALC_TMR_MS(  20UL, 50US)
#define CONST_TMR_INV_PLL_LOCK                  CONST_CALC_TMR_MS( 100UL, 50US)
#define CONST_TMR_INV_PI_SETTLE                 CONST_CALC_TMR_MS( 100UL, 50US)

#define CONST_TMO_INV_DISCHARGE_DC_LINK         CONST_CALC_TMR_MS(8000UL, 50US)
#define CONST_TMO_INV_PRECHARGE_PASSIVE_I       CONST_CALC_TMR_MS(5000UL, 50US)
#define CONST_TMO_INV_PRECHARGE_PASSIVE_II      CONST_CALC_TMR_MS(1500UL, 50US)
#define CONST_TMO_INV_PRECHARGE_ACTIVE          CONST_CALC_TMR_MS(2000UL, 50US)
#define CONST_TMO_INV_PHASE_ORDER               CONST_CALC_TMR_MS(  10UL, 50US)
#define CONST_TMO_INV_PLL_LOCK                  CONST_CALC_TMR_MS( 500UL, 50US)

//
// BUCK-BOOST TIMER AND TIMEOUT CONSTANTS
//

#define CONST_TMR_BUB_SETTLE                    CONST_CALC_TMR_MS( 100UL, 40US)
#define CONST_TMR_BUB_RELAY                     CONST_CALC_TMR_MS(  10UL, 40US)
#define CONST_TMR_BUB_CONTACTOR                 CONST_CALC_TMR_MS(  50UL, 40US)
#define CONST_TMR_BUB_CALIBRATION               CONST_CALC_TMR_MS(  20UL, 40US)
#define CONST_TMR_BUB_SOFT_START                CONST_CALC_TMR_MS( 100UL, 40US)

#define CONST_TMO_BUB_DISCHARGE_OUTPUT          CONST_CALC_TMR_MS(1500UL, 40US)
#define CONST_TMO_BUB_SOFT_START                CONST_CALC_TMR_MS(2000UL, 40US)

//
// DIGITAL MEASUREMENTS TIMEOUT
//

// ADC is triggered at the same time as the ePWM ISR. We wait in an infinite while loop inside the
// ePWM ISR until all digital measurements become ready. In case of an error, it could happen that
// digital measurements never become ready, which will cause the CPU to wait for measurements in
// the while loop indefinitely. That is why we introduce a condition to break the infinite loop,
// i.e., to stop waiting for digital measurements to become available. When ePWM timer counter
// reaches certain value, we will no longer wait for digital  measurements to become ready, and
// the system error flag will be raised. Constant is given as a number of ePWM clock ticks.
// [1 tick = 10ns at 100MHz -> 200 ticks = 2000ns]
#define CONST_TMO_ADC_MEASUREMENTS              ((uint32_t) 200)

//==============================================================================================//
//              PROTECTION LIMITS                                                               //
//=============================================================================================c//

// Coldplate temperature limits in degrees Celsius
#define CONST_PROT_TEMP_LOW                 ((fp32_t)  5)
#define CONST_PROT_TEMP_HIGH                ((fp32_t) 80)

// Allowed measurements offset range. This is used for grid (phase) and secondary (line) voltages;
// and for secondary (phase), two inductor, and output currents. Ideally, offset should be 2048.
// Range between 2000 and 2100 is within 2.5% of the expected (ideal) offset.
#define CONST_PROT_OFFSET_LOW               ((fp32_t) 2000)
#define CONST_PROT_OFFSET_HIGH              ((fp32_t) 2100)

// Grid (phase) RMS voltage (+-10% of 230 V)
#define CONST_PROT_V_GRID_RMS_LOW           ((fp32_t) 207)
#define CONST_PROT_V_GRID_RMS_HIGH          ((fp32_t) 253)

// Secondary (line) RMS voltage (+-10% of 93.6 V)
#define CONST_PROT_V_SEC_RMS_LOW            ((fp32_t)  84)
#define CONST_PROT_V_SEC_RMS_HIGH           ((fp32_t) 103)

// DC link voltage limits (+-12.5% of 160 V)
#define CONST_PROT_V_DC_LOW                 ((fp32_t) 142)
#define CONST_PROT_V_DC_HIGH                ((fp32_t) 180)

// Output, battery, and remote-sense voltage limits
#define CONST_PROT_V_OUT_HIGH               ((fp32_t) 120)
#define CONST_PROT_V_BAT_HIGH               ((fp32_t) 120)
#define CONST_PROT_V_REM_HIGH               ((fp32_t) 120)

// Grid frequency limits (+-2% of 50 Hz)
#define CONST_PROT_AC_FREQUENCY_LOW         ((fp32_t) UTILS_2PI*49.0)
#define CONST_PROT_AC_FREQUENCY_HIGH        ((fp32_t) UTILS_2PI*51.0)

//==============================================================================================//
//              REDEFINE SOME CONSTANTS FOR FASTER PLECS SIMULATIONS                            //
//=============================================================================================c//

#ifdef PLECS

//
// MEASUREMENTS CALIBRATION CANNOT BE USED IN PLECS SIMULATIONS
//

#ifdef CONST_CALIBRATION_BUFFER_SIZE
#error "Measurements calibration cannot be used in PLECS simulations. Comment-out CONST_CALIBRATION_BUFFER_SIZE macro."
#endif

//
// INVERTER CONSTANTS
//

#undef  CONST_INV_ACTIVE_I_D

#define CONST_INV_ACTIVE_I_D    ((fp32_t) 15)

//
// BUCK-BOOST CONSTANTS
//

#undef  CONST_RLIM_V_OUT_REF
#undef  CONST_RLIM_I_OUT_REF

#define CONST_RLIM_V_OUT_REF    ((fp32_t) 500)
#define CONST_RLIM_I_OUT_REF    ((fp32_t) 200)

//
// MEASUREMENTS TRANSFORMATION CONSTANTS
//

// In PLECS simulations we use ideal analog filters (i.e., filter gains)

#undef  CONST_MEAS_V_GRID_AN
#undef  CONST_MEAS_V_GRID_BN
#undef  CONST_MEAS_V_GRID_CN
#undef  CONST_MEAS_V_SEC_AB
#undef  CONST_MEAS_V_SEC_BC
#undef  CONST_MEAS_V_SEC_CA
#undef  CONST_MEAS_I_SEC_AN
#undef  CONST_MEAS_I_SEC_BN
#undef  CONST_MEAS_I_SEC_CN
#undef  CONST_MEAS_V_DC
#undef  CONST_MEAS_V_OUT
#undef  CONST_MEAS_V_BAT
#undef  CONST_MEAS_V_REM
#undef  CONST_MEAS_I_L1
#undef  CONST_MEAS_I_L2
#undef  CONST_MEAS_I_OUT

#define CONST_MEAS_V_GRID_AN    ((fp32_t) (391.14 / 2048.0))
#define CONST_MEAS_V_GRID_BN    ((fp32_t) (391.14 / 2048.0))
#define CONST_MEAS_V_GRID_CN    ((fp32_t) (391.14 / 2048.0))
#define CONST_MEAS_V_SEC_AB     ((fp32_t) (195.30 / 2048.0))
#define CONST_MEAS_V_SEC_BC     ((fp32_t) (195.30 / 2048.0))
#define CONST_MEAS_V_SEC_CA     ((fp32_t) (195.30 / 2048.0))
#define CONST_MEAS_I_SEC_AN     ((fp32_t) ( 64.32 / 2048.0))
#define CONST_MEAS_I_SEC_BN     ((fp32_t) ( 64.32 / 2048.0))
#define CONST_MEAS_I_SEC_CN     ((fp32_t) ( 64.32 / 2048.0))
#define CONST_MEAS_V_DC         ((fp32_t) (180.00 / 4096.0))
#define CONST_MEAS_V_OUT        ((fp32_t) (134.03 / 4096.0))
#define CONST_MEAS_V_BAT        ((fp32_t) (134.03 / 4096.0))
#define CONST_MEAS_V_REM        ((fp32_t) (134.03 / 4096.0))
#define CONST_MEAS_I_L1         ((fp32_t) ( 32.17 / 2048.0))
#define CONST_MEAS_I_L2         ((fp32_t) ( 32.17 / 2048.0))
#define CONST_MEAS_I_OUT        ((fp32_t) ( 64.32 / 2048.0))

//
// SOFTWARE TIMER AND TIMEOUT CONSTANTS
//

#undef  CONST_TMR_INV_SETTLE
#undef  CONST_TMR_INV_RELAY
#undef  CONST_TMR_INV_CONTACTOR

#define CONST_TMR_INV_SETTLE                    CONST_CALC_TMR_MS(   1UL, 50US)
#define CONST_TMR_INV_RELAY                     CONST_CALC_TMR_MS(   1UL, 50US)
#define CONST_TMR_INV_CONTACTOR                 CONST_CALC_TMR_MS(   1UL, 50US)

#undef  CONST_TMO_INV_DISCHARGE_DC_LINK
#undef  CONST_TMO_INV_PRECHARGE_PASSIVE_I
#undef  CONST_TMO_INV_PRECHARGE_PASSIVE_II
#undef  CONST_TMO_INV_PRECHARGE_ACTIVE

#define CONST_TMO_INV_DISCHARGE_DC_LINK         CONST_CALC_TMR_MS( 800UL, 50US)
#define CONST_TMO_INV_PRECHARGE_PASSIVE_I       CONST_CALC_TMR_MS( 500UL, 50US)
#define CONST_TMO_INV_PRECHARGE_PASSIVE_II      CONST_CALC_TMR_MS(1500UL, 50US)
#define CONST_TMO_INV_PRECHARGE_ACTIVE          CONST_CALC_TMR_MS(3000UL, 50US)

#undef  CONST_TMR_BUB_SETTLE
#undef  CONST_TMR_BUB_RELAY
#undef  CONST_TMR_BUB_CONTACTOR
#undef  CONST_TMR_BUB_CALIBRATION

#define CONST_TMR_BUB_SETTLE                    CONST_CALC_TMR_MS(   1UL, 40US)
#define CONST_TMR_BUB_RELAY                     CONST_CALC_TMR_MS(   1UL, 40US)
#define CONST_TMR_BUB_CONTACTOR                 CONST_CALC_TMR_MS(   1UL, 40US)
#define CONST_TMR_BUB_CALIBRATION               CONST_CALC_TMR_MS(   1UL, 40US)

#undef  CONST_TMO_BUB_DISCHARGE_OUTPUT
#undef  CONST_TMO_BUB_SOFT_START

#define CONST_TMO_BUB_DISCHARGE_OUTPUT          CONST_CALC_TMR_MS(1500UL, 40US)
#define CONST_TMO_BUB_SOFT_START                CONST_CALC_TMR_MS(2000UL, 40US)

#endif //#ifdef PLECS

#endif /* CONSTANTS_H_ */

//==============================================================================================//
//              END OF FILE                                                                     //
//=============================================================================================c//
