//==============================================================================================//
//              CONTROL.C                                                                       //
//=============================================================================================c//

#ifndef control_C_
#define control_C_

#include "control.h"

//==============================================================================================//
//              PI CONTROLLER                                                                   //
//=============================================================================================c//

_Bool CTRL_piInit(volatile CTRL_pi_S *pi)
{
    // Check parameters
    if (pi->KP==0 || (pi->TI==0 && pi->KI==0) || pi->TS<=0)
        return ERR_SET;

    // Reset controller
    CTRL_piReset(pi);

    // Convert parallel to standard PI controller form
    if (pi->TI==0)
    {
        pi->TI = pi->KP / pi->KI;
    }

    // Calculate digital PI controller parameters
    switch (pi->DTYPE)
    {
        case CTRL_DTYPE_TUSTIN :
            pi->B = pi->KP * pi->TS / pi->TI;
            pi->A = pi->KP - 0.5f * pi->B;
            return ERR_CLEAR;

        default :
            return ERR_SET;
    }
}

void CTRL_piReset(volatile CTRL_pi_S *pi)
{
    pi->e1 = 0.0f;
    pi->u1 = 0.0f;
}

//==============================================================================================//
//              STATE MACHINE                                                                   //
//=============================================================================================c//

void CTRL_smInit(volatile CTRL_sm_S *sm)
{
    sm->command  = CTRL_SM_COMMAND_STOP;
    sm->mode     = CTRL_SM_MODE_STOPPING;
    sm->state    = CTRL_SM_STATE_STOPPED;
    sm->substate = 0;
    sm->timer    = 1;
}

void CTRL_smGotoSubstate(volatile CTRL_sm_S *sm, uint16_t SUBSTATE)
{
    sm->substate = SUBSTATE;
    sm->timer = 1;
}

void CTRL_smGotoState(volatile CTRL_sm_S *sm, uint16_t STATE)
{
    sm->state = STATE;
    CTRL_smGotoSubstate(sm, 0);
}

void CTRL_smGotoNextSubstate(volatile CTRL_sm_S *sm)
{
    CTRL_smGotoSubstate(sm, sm->substate+1);
}

void CTRL_smGotoNextState(volatile CTRL_sm_S *sm)
{
    CTRL_smGotoState(sm, sm->state+1);
}

void CTRL_smGotoNextSubstateDelay(volatile CTRL_sm_S *sm, uint32_t DELAY)
{
    if (sm->timer < DELAY)
    {
        sm->timer++;
    }

    else
    {
        CTRL_smGotoNextSubstate(sm);
    }
}

_Bool CTRL_smGotoNextSubstateCondition(volatile CTRL_sm_S *sm, _Bool CONDITION, uint32_t TIMEOUT)
{
    if (CONDITION == 1)
    {
        CTRL_smGotoNextSubstate(sm);
        return CTRL_SM_TIMEOUT_NO;
    }

    else if (sm->timer < TIMEOUT)
    {
        sm->timer++;
        return CTRL_SM_TIMEOUT_NO;
    }

    else
    {
        return CTRL_SM_TIMEOUT_YES;
    }
}

void CTRL_smErrorStopStart(volatile CTRL_sm_S *sm, _Bool ERROR)
{
    // In case of any ERROR, go to the STOPPING PROCEDURE
    if (ERROR == ERR_SET)
    {
        // Start the STOPPING PROCEDURE, but only if the
        // STOPPING PROCEDURE has not already been triggered
        if (sm->mode != CTRL_SM_MODE_STOPPING)
        {
            CTRL_smGotoState(sm, CTRL_SM_STATE_RESET);
            sm->mode = CTRL_SM_MODE_STOPPING;
        }
    }

    // There is no ERROR, should we START or STOP?
    else // (ERR == ERR_CLEAR)
    {
        // Input command is to START
        if (sm->command == CTRL_SM_COMMAND_START)
        {
            // Start the STARTING PROCEDURE, but only if the
            // STARTING PROCEDURE has not already been triggered
            // [this can be done only from the STOPPED state]
            if ((sm->mode != CTRL_SM_MODE_STARTING) && (sm->state == CTRL_SM_STATE_STOPPED))
            {
                CTRL_smGotoState(sm, CTRL_SM_STATE_RESET);
                sm->mode = CTRL_SM_MODE_STARTING;
            }
        }

        // Input command is to STOP
        else // CTRL_SM_COMMAND_STOP
        {
            // Start the STOPPING PROCEDURE, but only if the
            // STOPPING PROCEDURE has not already been triggered
            if (sm->mode != CTRL_SM_MODE_STOPPING)
            {
                CTRL_smGotoState(sm, CTRL_SM_STATE_RESET);
                sm->mode = CTRL_SM_MODE_STOPPING;
            }
        }
    }
}

#endif /* CONTROL_C_ */

//==============================================================================================//
//              END OF FILE                                                                     //
//=============================================================================================c//
