//==============================================================================================//
//              CONTROL.H                                                                       //
//=============================================================================================c//

#ifndef CONTROL_H_
#define CONTROL_H_

#include "utils.h"

#include "error.h"

//==============================================================================================//
//              MACRO FLAGS                                                                     //
//=============================================================================================c//

//
// DISCRETIZATION TYPE FLAGS
//

#define CTRL_DTYPE_IMPULSE              1
#define CTRL_DTYPE_ZOH                  2
#define CTRL_DTYPE_EULER_FORWARD        3
#define CTRL_DTYPE_EULER_BACKWARD       4
#define CTRL_DTYPE_TUSTIN               5

//
// STATE MACHINE STATES
//

#define CTRL_SM_STATE_RESET             0
#define CTRL_SM_STATE_START_OR_STOP     1
#define CTRL_SM_STATE_STARTED           2
#define CTRL_SM_STATE_STOPPED           3

//
// STATE MACHINE COMMANDS
//

#define CTRL_SM_COMMAND_STOP            0
#define CTRL_SM_COMMAND_START           1

//
// STATE MACHINE MODES
//

#define CTRL_SM_MODE_STOPPING           0
#define CTRL_SM_MODE_STARTING           1

//
// STATE MACHINE ERROR
//

#define CTRL_SM_TIMEOUT_NO              0
#define CTRL_SM_TIMEOUT_YES             1

//==============================================================================================//
//              PI CONTROLLER                                                                   //
//=============================================================================================c//

typedef struct CTRL_pi_STRUCT
{
    uint16_t DTYPE; // Discretization type flag
    fp32_t KP;      // Proportional gain
    fp32_t TI;      // Integral time constant
    fp32_t KI;      // Integral gain
    fp32_t TS;      // Sample time (s)
    fp32_t ULO;     // Control action lower limit
    fp32_t UHI;     // Control action upper limit
    fp32_t A;       // Digital PI controller parameter
    fp32_t B;       // Digital PI controller parameter
    fp32_t e1;      // PI controller state e[k-1]
    fp32_t u1;      // PI controller state u[k-1]
} CTRL_pi_S;

_Bool CTRL_piInit(volatile CTRL_pi_S *pi);
void CTRL_piReset(volatile CTRL_pi_S *pi);
UTILS_INLINE fp32_t CTRL_piUpdate(fp32_t e0, volatile CTRL_pi_S *pi);

UTILS_INLINE fp32_t CTRL_piUpdate(fp32_t e0, volatile CTRL_pi_S *pi)
{
    // TODO check recursive equation for different discretization types

    // Calculate new control action
    fp32_t u0 = pi->A*(e0-pi->e1) + pi->B*pi->e1 + pi->u1;

    // Limit the control action (anti-windup)
    if (u0 > pi->UHI)   { u0 = pi->UHI; }
    if (u0 < pi->ULO)   { u0 = pi->ULO; }

    // Update controller states
    pi->e1 = e0;
    pi->u1 = u0;

    // Return control action
    return u0;
}

//==============================================================================================//
//              STATE MACHINE                                                                   //
//=============================================================================================c//

typedef struct CTRL_sm_STRUCT
{
    uint16_t command;       // Start or stop command
    uint16_t mode;          // Starting or stopping mode
    uint16_t state;         // State machine state
    uint16_t substate;      // State machine substate
    uint32_t timer;         // Software timer
    uint16_t flag_1;        // General-purpose flag 1
    uint16_t flag_2;        // General-purpose flag 2
    uint16_t flag_3;        // General-purpose flag 3
    uint16_t flag_4;        // General-purpose flag 4
} CTRL_sm_S;

void CTRL_smInit(volatile CTRL_sm_S *sm);
void CTRL_smGotoSubstate(volatile CTRL_sm_S *sm, uint16_t SUBSTATE);
void CTRL_smGotoState(volatile CTRL_sm_S *sm, uint16_t STATE);
void CTRL_smGotoNextSubstate(volatile CTRL_sm_S *sm);
void CTRL_smGotoNextState(volatile CTRL_sm_S *sm);
void CTRL_smGotoNextSubstateDelay(volatile CTRL_sm_S *sm, uint32_t DELAY);
_Bool CTRL_smGotoNextSubstateCondition(volatile CTRL_sm_S *sm, _Bool CONDITION, uint32_t TIMEOUT);
void CTRL_smErrorStopStart(volatile CTRL_sm_S *sm, _Bool ERROR);

#define CTRL_SM_GOTO_NEXT_SUBSTATE_CONDITION(_COND, _TMO, _REG, _FLD)                       \
    if (CTRL_smGotoNextSubstateCondition(sm, (_COND), _TMO) == CTRL_SM_TIMEOUT_YES)         \
    {                                                                                       \
        ERR_FIELD_SET(_REG, _FLD);                                                          \
    }

#endif /* CONTROL_H_ */

//==============================================================================================//
//              END OF FILE                                                                     //
//=============================================================================================c//
